///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
//
// PlaneBoxIntersection
//
/////////////////////////////////////////////////////////////////////

#ifndef _PLANE_BOX_INTERSECTION_H_
#define _PLANE_BOX_INTERSECTION_H_

#include <Inventor/SbLinear.h>
#include <Inventor/fields/SoSFBox3f.h>
#include <Inventor/fields/SoSFPlane.h>
#include <Inventor/nodes/SoLineSet.h>

class SoFieldSensor;
class PlaneBoxIntersectionImpl;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Shape node that draws the intersection of a plane and a box.
 * 
 * @ingroup MedicalNodes
 * 
 * @DESCRIPTION
 *
 * This node is useful for highlighting. The intersection of a #plane
 * and a #box in the general case is a polygon with up to six points.
 * If the plane does not intersect the box, nothing is drawn.
 *
 * This node is particularly useful for highlighting the location of a
 * clip plane (using the bounding box of the geometry) and for highlighting
 * the outline of an oblique slice (SoObliqueSlice) using the bounding box
 * (extent) of the volume. The plane field may be automatically updated
 * from a dragger or another node such as SoClipPlane or SoObliqueSlice
 * using the connectFrom() method.
 *
 * This node is derived from SoLineSet, but the standard fields of that class
 * (numVertices, startIndex and vertexProperty) are managed internally and
 * should not be modified by the application.
 *
 * The color, line width, etc of the intersection lines can be set using
 * SoMaterial, SoDrawStyle, etc nodes in the usual way.
 *
 * @FILE_FORMAT_DEFAULT
 *    PlaneBoxIntersection {
 *    @TABLE_FILE_FORMAT
 *       @TR plane             @TD 1 0 0 0
 *       @TR box               @TD 0 0 0 0 0 0
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *    SoClipPlane,
 *    SoObliqueSlice,
 *    MoMeshPlaneSlice
 *
 * @PREVIEWFEATURES
 */ 

class INVENTORMEDICAL_API PlaneBoxIntersection : public SoLineSet
{

  SO_NODE_HEADER(PlaneBoxIntersection);

 public:

  /** 3D box to be intersected. Default is an empty box. */
  SoSFBox3f   box;

  /** Plane to intersect with.
   *  Default is plane with normal vector 1,0,0 at distance 0. */
  SoSFPlane   plane;

  /** Initialize this class */
  static void   initClass();
  /** Finish this class */
  static void   exitClass();

  /** Constructor */
  PlaneBoxIntersection();

 protected:

   // Destructor
   virtual ~PlaneBoxIntersection();

  // These sensors ensure that the geometry is updated when the fields change
  SoFieldSensor *m_boxFieldSensor;
  SoFieldSensor *m_planeFieldSensor;

  static void fieldSensorCB(void *, SoSensor *);

  PlaneBoxIntersectionImpl* m_impl;

};

#endif //PlaneBoxIntersection