///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _PLANE_GEOMETRY_INTERSECTION_H_
#define _PLANE_GEOMETRY_INTERSECTION_H_

#include <Medical/InventorMedical.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFPlane.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/sensors/SoFieldSensor.h>

class PlaneGeometryIntersectionImpl;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Class for computing the intersection of geometry with a plane.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 * This class computes and (optionally) renders the intersection of the geometry
 * in the specified #scene graph with the specified #plane.  In curve mode it
 * computes a set of line segments corresponding to the intersection.  In polygon
 * mode it computes a set of polygon boundaries that can be used as a "cap" for
 * clipped geometry.  The intent is that the specified scene graph is one shape.
 * In fact it can be any scene graph, but the result is always a single polygon
 * with multiple boundaries (if necessary).  The computed vertices (and indices
 * in polygon mode) can be accessed using the getShape() method. The curve or
 * polygon may be exported to a file like any other Open Inventor geometry using
 * SoWriteAction, SoSTLWriteAction, etc.
 *
 * Curve mode: @BR
 * By default we compute and render the curve(s) defined by the intersection
 * of the scene geometry and the plane.  This is fast and works reliably for any
 * Open Inventor geometry that produces triangles when an SoCallbackAction is
 * applied (almost all Open Inventor geometry nodes). The appearance of the
 * intersection curve can be controlled like any Open Inventor geometry using
 * SoMaterial, SoDrawStyle, etc. nodes.  The application may need to use an
 * SoPolygonOffset node to "push" the scene geometry behind the intersection
 * curve (polygon offset does not affect the intersection curve because it is
 * a line set).  Curve mode is useful to see where a clip plane intersects
 * the geometry.  Because it is much faster than polygon mode it is useful as
 * a preview of where the polygons will be generated (if possible).
 *
 * Polygon mode: @BR
 * When the #polygon field is set to true, the node converts the
 * intersection curves to polygon boundaries.  All the boundaries are stored in
 * an SoIndexedFaceSet node and an SoShapeHints node is created with the
 * 'windingType' field set to ODD_TYPE. All boundaries have counterclockwise
 * vertex ordering and the face normal vector is the opposite of the current
 * #plane normal vector (because for a clip plane the plane normal points toward
 * the unclipped half space which is effectively toward the interior of the
 * shape).  The appearance of the polygon can be controlled like any Open
 * Inventor geometry using SoMaterial, etc. nodes.  There are cases where
 * polygons cannot be constructed, for example if the geometry is self-intersecting.
 *
 * The coordinates of the intersection boundaries can be accessed using the
 * getShape() method.  This is an SoLineSet in curve mode and an
 * SoIndexedFaceSet in polygon mode.  In either case the coordinates are in
 * the shape's attached SoVertexProperty node.  Note that in curve mode the line
 * segments are in random order and do not form a connected polyline. This
 * allows the intersection geometry to be computed in minimum time and does
 * not (visually) effect rendering. If you need the boundary(s) as a connected
 * polyline for some purpose, use the polygon mode. Note that in polygon mode
 * there are redundant vertices in the vertex list because it still contains
 * all the line segments.  To follow the boundaries in connection order use
 * the coordinate indices in the SoIndexedFaceSet node.
 *
 * Notes:
 *   - Multiple boundaries in polygon mode. @BR
 *     Currently does not test if whether multiple boundaries are "holes" or
 *     "islands".  Rendering should be correct in either case, but eventually
 *     some optimization could be done if all boundaries are disjoint.
 *
 *   - Memory. @BR
 *     Memory allocated for vertices and indices is re-used on each recomputation
 *     and is not freed until the node is destroyed.  This is best for performance.
 *     Eventually there should be a method to force memory to be freed.
 *
 * Limitations:
 *   - Valid geometry. @BR
 *     For polygon mode to work correctly, the geometry must obey the same rules
 *     as a valid STL geometry.  In particular the geometry must form a closed
 *     solid and must not be self-intersecting.  All the intersection boundaries
 *     must be closed and must not intersect any other boundary. @BR
 *     However in curve mode, there are no limitations.  For example, the node
 *     can be used to render the intersection of a surface with a plane.
 *
 *   - Triangulation of the polygon. @BR
 *     This node simply computes the intersection boundary(s).  For rendering it
 *     relies on Open Inventor to convert the polygon to triangles.  In many
 *     cases the resulting triangles have a poor (large) aspect ratio.
 *
 *   - Floating point precision. @BR
 *     Curve mode should work reasonably well with any geometry and any coordinates.
 *     However constructing polygon boundaries is very sensitive to floating point
 *     precision problems. If the results are not good, try adjusting the #tolerance
 *     value.
 *
 *   - Redundant coordinates. @BR
 *     In polygon mode the SoVertexProperty node contains all the coordinates
 *     computed for the intersection line segments.  The SoIndexedFaceSet node
 *     only indexes a subset of these vertices (typically about half). Not removing
 *     the unused vertices saves time and has no effect on rendering.  To get only
 *     the coordinates that are actually used, extract vertices based on the index
 *     values in the SoIndexedFaceSet node. Eventually could be an option.
 *
 *   - Performamce. @BR
 *     This is currently a "brute force" implementation and performance could be better.
 *
 * @FILE_FORMAT_DEFAULT
 *    PlaneGeometryIntersection {
 *    @TABLE_FILE_FORMAT
 *       @TR scene          @TD NONE
 *       @TR plane          @TD 1 0 0 0
 *       @TR polygon        @TD FALSE
 *       @TR automatic      @TD TRUE
 *       @TR tolerance      @TD 1e-6
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *  InventorMedical, SoClipPlane, SoClipPlaneManip, SoSTLWriteAction
 *
 * @PREVIEWFEATURES
 */

class INVENTORMEDICAL_API PlaneGeometryIntersection : public SoSeparator
{
  SO_NODE_HEADER(PlaneGeometryIntersection);

public:

  /** Scene to intersect with plane. Default is none. */
  SoSFNode scene;

  /** Plane to intersect with.
   *  Default is (1,0,0) at 0 (same as SoClipPlane).
   *  Typically this field is connected from the clipping node's 'plane' field.
   */
  SoSFPlane plane;

  /** Specifies whether to compute a capping polygon (default false).
   *  If true, compute a capping polygon.  If false, compute only the intersection curve.
   */
  SoSFBool polygon;

  /** Specifies whether to recompute the intersection automatically or not (default true).
   *  If true, the intersection (curve or polygon depending on the #polygon field)
   *  will be re-computed automatically when the scene or the plane changes.  If false,
   *  the intersection will only be re-computed when the recompute method is called.
   */
  SoSFBool automatic;

  /** Tolerance for testing if vertices are equal.
   *  Default is 1e-6.  This tolerance is used for calculation of triangle
   *  intersections with the plane.
   */
  SoSFFloat tolerance;

  /** Get the shape node that contains the computed vertices and indices.
   *  Returns null if no shape has been computed yet.
   *  In curve mode this is an SoLineSet. In polygon mode this is an SoIndexedFaceSet.
   *  In both cases the vertices are in the shape node's attached SoVertexProperty
   *  node's 'vertex' field.
   *
   *  The application must not modify these nodes.
   */
  const SoShape* getShape() const;

  /** Recompute the intersection.
   *  Recomputes the intersection boundary(s) based on the current #scene and #plane.
   *  Normally used when the #automatic field is set to false.  Useful when the goal
   *  is simply to compute the intersection.  Also useful if recomputation is too
   *  slow. For example, to defer recomputation until the user has finished moving
   *  the clip plane.
   *
   *  Returns 0 if successful.  Any other value means that an error occured during the
   *  computation and the geometry may not be valid.
   */
  int recompute();

  /** Constructor */
  PlaneGeometryIntersection();

  /** Initialize class (called automatically by InventorMedical::init()). */
  static void initClass();

  /** Finish using class (called automatically by InventorMedical::finish()). */
  static void exitClass();

protected:

  PlaneGeometryIntersectionImpl* m_impl;

  /** Detect when one of our fields changes. */
  SoFieldSensor* m_sceneSensor;
  SoFieldSensor* m_planeSensor;
  SoFieldSensor* m_polygonSensor;
  static void fieldSensorCB( void* data, SoSensor* sensor );

  /** Reset geometry */
  void resetGeometry();

  /** Destructor */
  virtual ~PlaneGeometryIntersection();
};
#endif
