///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _MEDICAL_RULER_H
#define _MEDICAL_RULER_H

#include <Medical/InventorMedical.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFString.h>

class SoCone;
class SoFont;
class SoText2;
class SoText3;
class SoLineSet;
class SoDrawStyle;
class SoTransform;
class SoTranslation;
class SoEventCallback;
class SoVertexProperty;
class SoOrthoSliceDetail;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Interactive distance measuring tool for SoOrthoSlice.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This node displays an interactive distance measuring tool on top of an
 *   SoOrthoSlice.  The user can click and drag to display a line and the
 *   physical distance between the clicked point and the current cursor position.
 *
 *   This node is only intended to be used in a "2D" viewing environment
 *   using an SoOrthographicCamera, for example viewing a single slice of
 *   a medical volume.
 *
 *   This node requires that the application handle mouse press, release and
 *   move events.  Mouse move events must be forwarded to the node using the
 *   manageMouseMove method.  See example examples/source/Medical/Analysis/medicalRuler.
 *
 * @FILE_FORMAT_DEFAULT
 *    Ruler {
 *    @TABLE_FILE_FORMAT
 *      @TR globalFactor   @TD 1
 *      @TR label          @TD mm
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical, DicomInfo, Gnomon, Magnifier
 * 
 * @PREVIEWFEATURES
 */ 

class INVENTORMEDICAL_API Ruler : public SoSeparator {

  SO_NODE_HEADER(Ruler);

public:
  
  /** Specifies a scale factor applied to the size of the arrowheads on the line.
   *  Default is 1.
   */
  SoSFFloat globalFactor;

  /** Specifies an (optional) label appended to measure value.
   *  Default is "mm".
   */
  SoSFString label;

  /** The application should call this method with 'true' on the mouse down
   *  event that begins a measure operation, then call this method with 'false'
   *  on each mouse move event during the measure operation.
   *  See the medicalRuler example program for more details.
   */
  bool manageMouseMove(bool firstClick, const SoOrthoSliceDetail* detail);

  /** Returns the internal font node.
   *  The application can use this to modify the appearance of the text.
   */
  SoFont* getFontNode() const;

  /** Constructor. */
  Ruler();

  /** Initialize class (called automatically by InventorMedical::init()). */
  static void initClass();

  /** Finish using class (called automatically by InventorMedical::finish()). */
  static void exitClass();

SoINTERNAL public:
  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

protected:
  virtual ~Ruler();

private:

  /**
  * Build the SceneGraph reprensenting the Gnomon
  */
  void buildRuler();
  void updateGlobalFactor();

  bool    m_text3D;
  SbVec3f m_initPos;

  SoDrawStyle* m_dStyle;
  SoTransform* m_leftConeTransform;
  SoTransform* m_rightConeTransform;
  SoTranslation* m_initialLeftTranslation;
  SoTranslation* m_initialRightTranslation;
  SoCone* m_leftCone;
  SoCone* m_rightCone;

  SoVertexProperty* m_rulerVertexProp;
  SoLineSet*   m_ruler;

  SoFont*      m_annotationFont;
  SoTransform* m_annotationTransform;
  SoText3*     m_rulerAnnotation3D;
  SoText2*     m_rulerAnnotation2D;
};
#endif
