// View manager

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _VIEW_MANAGER_H_
#define _VIEW_MANAGER_H_

#include <Inventor/nodes/SoGroup.h>
#include <Medical/InventorMedical.h>

// It's convenient for applications to get this automatically.
#include "SceneView.h"

class SoNodeSensor;
class SoHandleEventAction;
class SoPickAction;

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG View manager for use with viewer components.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This is a group node that manages a list of SceneView nodes to allow
 *   rendering multiple views inside a render area.
 *
 *   Each SceneView node defines a viewport (sub-region of the render area), a
 *   scene to render (usually including a camera node) and optionally a border.
 *   SceneView nodes may be used without a ViewManager node, but they rely on a
 *   ViewManager node to route user events like mouse button and keyboard presses
 *   to the appropriate SceneView scene graph for handling.
 *
 *   Primary distribution of events is based on the event position, the viewport
 *   and the order the SceneView nodes are traversed. If no mouse buttons are pressed,
 *   then all events are sent to the first SceneView whose viewport contains the
 *   event position.  When a mouse button is pressed, the view that handles that
 *   event "captures" the event stream until all mouse buttons are released.  This
 *   is convenient because it allows the user to drag the cursor anywhere while, for
 *   example, moving the camera.  While at least one mouse button is pressed, all
 *   events are sent to the SceneView that owns the event stream.  This is similar
 *   to the way the classic Open Inventor viewers capture events while a mouse button
 *   is pressed.  It is also similar to the SoHandleEventAction setGrabber() feature
 *   that is used by Open Inventor draggers (see SoDragger). The ViewManager event
 *   capture and setGrabber() are independent and complementary.
 *
 *   The application can also choose to do "global" event handing by adding an
 *   SoEventCallback node above the ViewManager node.  If the application needs to
 *   know which view the event should affect, for example a key press that means to
 *   reset the camera, use the getViewIndex(SbVec2f position) method to know which
 *   view contains the event position.  Finally, on some devices it is not possible
 *   to know the position of some events, for example key press events. So ViewManager
 *   also tracks the last known event position, based on mouse move events.  The
 *   application can query this using getLastEventViewIndex() and getLastEventPosition().
 *
 *   ViewManager also provides convenience methods for setting and getting the
 *   viewports of the child views.  This is convenient for managing the layout of
 *   the viewports.  For example to change from four equal size viewports to one
 *   large viewport plus three small viewports.
 *
 *   @B Note:@b Only SceneView nodes are allowed as children of a ViewManager node.
 *   Attempts to add any other kind of node are ignored.
 *
 *   Example 1:
 *   \code
 *   // Create 2 x 2 view layout
 *   ViewManager* m_viewManager = new ViewManager();
 *   for (int i = 0; i < 4; ++i)
 *     m_viewManager->addView( new SceneView() );
 *   m_viewManager->getView(0)->setViewport( 0, 0.5f, 0.5f, 0.5f );    // Upper left
 *   m_viewManager->getView(1)->setViewport( 0.5f, 0.5f, 0.5f, 0.5f ); // Upper right
 *   m_viewManager->getView(2)->setViewport( 0, 0, 0.5f, 0.5f );       // Lower left
 *   m_viewManager->getView(3)->setViewport( 0.5f, 0, 0.5f, 0.5f );    // Lower right
 *   \endcode
 *
 * 
 * @SEE_ALSO
 *  SceneView
 *
 * @PREVIEWFEATURES 
 */ 

class INVENTORMEDICAL_API ViewManager : public SoGroup {

  SO_NODE_HEADER(ViewManager);

public:

  /** Initialize the class. */
  static void   initClass();

  /** Finish using the class. */
  static void   exitClass();

  /** Constructor. */
  ViewManager();

  /** Add a view.
   *  Adds the specified view to the end of the list. */
  void addView( const SceneView* newView );

  /** Returns the current number of views being managed. */
  int getNumViews() const;

  /** Returns the view with the specified index.
   *  Same as getChild() but avoids the need to cast the result.
   *  Returns null if index is not valid. */
  SceneView* getView( int index ) const;

  /** Returns the first view containing the specified pixel coordinate.
   *  This is the view that would handle an event at this location.
   *  This is convenient, for example, to implement global hot keys that should
   *  only affect the view that contains the event.  Then it is not necessary to
   *  duplicate an SoEventCallback in each view for this purpose.
   *  Returns null if the location is not valid. */
  SceneView* getView( const SbVec2f& position ) const;

  /** Returns the index of the specified view.
   *  Returns -1 if the specified view does not exist. */
  int getViewIndex( const SceneView* view ) const;

  /** Returns a view index for the specified pixel coordinate.
   *  This is the index of the view that would handle an event at this location.
   *  This is convenient, for example, to implement global hot keys that should
   *  only affect the view that contains the event.  Then it is not necessary to
   *  duplicate an SoEventCallback in each view for this purpose.
   *  Returns -1 if the location is not valid. */
  int getViewIndex( const SbVec2f& position ) const;

  /** Returns the index of the view that handled the last event.
   *  If event capture is active (a mouse button was pressed), then the index
   *  returned is the event capture view, not necessarily the same as the view
   *  currently containing the cursor.
   *  Returns -1 if not possible to answer the query.
   */
  int getLastEventViewIndex() const;

  /** Returns the position of the last event handled (in pixels).
   *  Generally Open Inventor sets the cursor position for all mouse and keyboard events.
   *  But this query could be useful if it is necessary to handle an event that does
   *  not have an associated position on screen. Returns (0,0) if no other info.
   */
  SbVec2f getLastEventPosition() const;

  /** Convenience method to set the viewport of a view.
   *  Ignored if the index is out of range. */
  void setViewport( int index, SbVec4f& vport );

  /** Convenience method to set the viewport of a view.
   *  Ignored if the index is out of range. */
  void setViewport( int index, float xorig, float yorig, float xsize, float ysize );

  /** Set event capture view.
   *  If viewIndex >= 0, calls enableEventHandling with true for that view and
   *  false for every other view.  If viewIndex < 0, sets true for all views.
   *  Invalid indices are ignored.  Used internally to manage event capture.
   *
   *  Generally the application should not call this method.
   */
  void setEventCaptureView( int viewIndex );

 SoEXTENDER public:

  /** Override generic child methods to enforce children are SceneView nodes. */
  virtual void addChild( SoNode* child );
  virtual void insertChild( SoNode *child, int newChildIndex );
  virtual void replaceChild( SoNode* oldChild, SoNode* newChild );
  virtual void replaceChild( int index, SoNode* newChild );

protected:
  /** Destructor */
  virtual ~ViewManager();

  // Actions we handle directly
  virtual void handleEvent( SoHandleEventAction* action );

  // Internal state
  int  m_numViews;               // Remember this to check for changes
  unsigned int m_buttonsPressed; // Which buttons are currently pressed?

  SbVec2f m_eventPos;            // Position of last event handled
  int     m_eventViewIndex;      // View that handled the last event
  int     m_captureViewIndex;    // View that "owns" the event stream

  SoNodeSensor* m_sensor;
  static void sensorCB( void* data, SoSensor* sensor );
};

#endif