/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  _MIEXTRACTORCALLBACK_H
#define  _MIEXTRACTORCALLBACK_H

#include <Inventor/STL/string>

/**
* @DTEXT  Abstract base class for extraction callback interfaces.
*
* @ingroup MeshVizXLM_Extract 
* 
* @DESCRIPTION 
* This abstract class specifies the interface used by extractors to notify the application
* when an extraction starts, stops or when it is in progress.
* For instance this class allows an update of any kind of progress bar.
*
* An extraction is conceptually divided into sequential phases and each phase has multiple
* iteration steps.
*
* When an extraction starts, MeshViz calls the beginExtract() method. The number of phases
* in the extraction is passed to this method. When the extraction is complete, MeshViz
* will call the endExtract() method.
* However due to possible nested extractions, beginExtract() and endExtract() calls may occur
* several times during an extraction process. Both methods are called the same number of times.
* Therefore monitoring the end of an overall extraction can be performed by incrementing a counter
* on beginExtract() calls and decreasing it on endExtract() calls.
*
* A phase is a logical part of the job of the extraction. The number of phases is passed to the
* beginExtract() method. When a phase starts, MeshViz calls the beginPhase() method with an id
* and a name for the phase as well as the number of iteration steps in the phase. When the phase
* is complete, MeshViz will call the endPhase() method. During a phase, MeshViz will call the
* endStep() method with the number of iteration steps completed.  The percentage of the phase
* completed is this value divided by the number of iteration steps in the phase.  The number
* of calls to endStep() during a phase is controlled by (or at least influenced by) the value
* returned from the application's getEndStepCallPeriod().
*
* The value returned by getEndStepCallPeriod() is just a suggestion to MeshViz. MeshViz controls
* the number of calls to endStep() in order to prevent a large increase of the total time of extraction.
* If the phase needs a small number of iteration steps, (for instance a small number of cells
* in the mesh) then endStep() is probably never called even if getEndStepCallPeriod() returned
* a small value.
*
* Each phase could be associated to a single progress bar. If the application uses only one 
* progress bar, the progress bar can be initialized in the implementation of beginPhase(), 
* and it can be closed in the implementation of endPhase().
* The progress bar can be updated at regular steps with the implementation of endStep().
*
* MeshViz may interrupt a phase with an early call to endPhase() and even restart a phase if
* necessary by calling beginPhase() with the same phase identifier but possibly a different
* number of iterations.
*
* The return values of beginPhase(), endPhase() and endStep() cancel the mesh extraction if false when
* the callback is directly set to some extractors (see MiExtractorBase::setExtractorCallback()) and
* it will throw a MiCancellationException. Refer to each extractor documentation to know if it
* can be interrupted during processing.
* However the return values are ignored when the callback is set to a representation node
* (see MoMeshRepresentation::setExtractorCallback()). As this may change in future releases, we advise
* to return true by default.
*
* @note On a very large mesh, setting an extractor callback could slow down the extraction,
* especially on multi-core cpu and when the extraction is done in parallel mode.
* (see MoMeshRepresentation::setExtractorCallback() and MoMeshRepresentation#parallel). 
* However, if getEndStepCallPeriod() returns a value >= 1, endStep() is never called preventing
* any slow down.
*/
class MiExtractorCallback
{
public:
  virtual ~MiExtractorCallback() {}

  /**
  * This method is called each time a new extraction starts and only if the mesh, 
  * the cell filter, the tessellator or the data sets have changed since the last extraction.
  * @param extractorName name of this extraction.
  * @param geomChanged true if the geometry of the extracted mesh must be updated.
  * @param topoChanged true if the topology of the extracted mesh must be updated.
  * @param dataSetChanged true if the data sets associated to the extracted mesh must be updated.
  * @param numPhases number of phases that will be done by this extraction.
  */
  virtual void beginExtract(const std::string extractorName, 
                            bool geomChanged, bool topoChanged, bool dataSetChanged, 
                            size_t numPhases ) = 0;

  /**
  * This method is called each time a new extraction ends and only if the mesh, 
  * the cell filters, the tessellator or the data sets have changed since the last extraction.
  * After this call, the extracted mesh has been updated and can then be parsed again to retrieve the new values.
  */
  virtual void endExtract() = 0;
  
  /**
   * This method is called each time a new phase of the current extraction starts.
   * @param phaseId id of this new phase.
   * @param phaseName name of this phase.
   * @param numIterationInPhase number of iterations that will be executed during this phase.
   * @return returned value not used yet.
   */
  virtual bool beginPhase(size_t phaseId, std::string phaseName, size_t numIterationInPhase ) = 0;

  /**
   * This method is called each time a phase of the current extraction is finished.
   * @return returned value not used yet.
   */
  virtual bool endPhase() = 0;
  
  /**
   * This method is called each time a step of the current phase is finished.
   * This method could be used to update a progress bar.
   * @param numIterationDone total number of iterations done since the phase beginning.
   * To update a progress bar, the percentage of the job done in the current phase is
   * numIterationDone/numIterationInPhase (see beginPhase())
   * @return returned value not used yet.
   */
  virtual bool endStep(size_t numIterationDone) = 0;

  /**
  * Defines the period of calling the endStep() method.
  * A reasonable value is 0.05 which means endStep()
  * is called after each step of ~5%.
  *   - period >= 1 : endStep() is never called.
  *   - period <= 0 : the endStep() is called as much as possible.
  *                   But it is limited in order to prevent a large increase of the 
  *                   total extraction time.
  */
  virtual double getEndStepCallPeriod() = 0;

};

#endif /* _MIEXTRACTORCALLBACK_H */


