/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  _MISKINEXTRACTUNSTRUCURED_H
#define  _MISKINEXTRACTUNSTRUCURED_H

#include <MeshVizXLM/extractors/MiBaseExtractor.h>
#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>
#include <MeshVizXLM/mesh/data/MiDataSetI.h>
#include <MeshVizXLM/extrmesh/MeXSurfaceMeshUnstructured.h>
#include <MeshVizXLM/extrmesh/MeXLineMeshUnstructured.h>
#include <MeshVizXLM/extrmesh/data/MeXDataSetI.h>
#include <MeshVizXLM/tessellator/MiTessellator.h>

class MiCellFilterI;
class MeXPointMeshCurvilinear;

/**
 * @DTEXT Skin extractor for an unstructured volume mesh.
 * 
 * @ingroup MeshVizXLM_Extract_Skin
 * 
 * @DESCRIPTION Performs 3 types of extractions:
 *
 * extractSkin() builds a MeXSurfaceMeshUnstructured interfaces containing the skin of the given mesh.
 * The skin of a mesh is made up of all the faces that belong to only one cell. Scalar and vector sets can
 * be mapped onto this surface by calling extractScalarSet() and extractVec3Set().
 *
 * extractLineSkin() builds a MeXLineMeshUnstructured interface containing the line skin of the given mesh, which is 
 * made up of the edges from the faces that belong to the skin. When working with quadratic meshes, only outer edges
 * of non-linear cells are extracted. 
 *
 * extractPoints() builds an MeXPointMeshCurvilinear interface containing the point skin of the given mesh, which is 
 * made up of the nodes from the faces that belong to the skin.
 *
 * Internal faces and edges of a mesh will not be displayed except if they are adjacent to dead cells.
 *
 * extractScalarSet() builds the extracted scalarset corresponding to the latest skin extracted (surface, line or points).
 *
 * extractVec3Set() builds the extracted vectorset corresponding to the latest skin extracted (surface, line or points).
 *
 * @note The skin extraction processing can be cancelled by setting a MiExtractorCallback, which will throw a MiCancellationException.

 */
class MESHVIZXLM_EXTR_API MiSkinExtractUnstructured : virtual public MiBaseExtractor
{
public:
  virtual ~MiSkinExtractUnstructured() {}

  /**
  * @copydoc MiCellExtractIjk::getNewInstance()
  */
  static MiSkinExtractUnstructured* getNewInstance(const MiVolumeMeshUnstructured& mesh, bool parallel = true, MiTessellator* tessellator=NULL);

  /**
  * Add a new cell range constraint to iterate over when performing the skin extraction.
  * Default is to iterate over all cells in the mesh.
  * Adding a new range to the extractor changes its state, thus triggering a new extraction
  * when calling extractSkin() even if the mesh remains the same.
  * Note: For now, only the first added cellrange is considered during the extraction. Further added ones
  * are ignored.
  * @param rbegin the smallest cell index in the range.
  * @param rend the largest cell index in the range (not included in extraction).
  * Note: The value UNDEFINED_ID is interpreted as the minimum/maximum cell index available in 
  * the current mesh.
  */
  virtual void addCellRange(size_t rbegin, size_t rend) = 0;

  /**
  * Remove all cell ranges previously added.
  * Clearing all ranges changes the extractor state, thus triggering a new extraction when
  * calling extractSkin() even if the mesh remains the same.
  */
  virtual void clearCellRanges() = 0;

  /**
  * @copydoc MiSkinExtractIjk::extractSkin()
  */
  virtual const MeXSurfaceMeshUnstructured& extractSkin(const MiCellFilterI* cellFilter=NULL) = 0;

  /**
  * Builds the topology and geometry of the line skin.
  * The line skin of the given mesh is made up of the edges from the faces that belong to the skin. 
  * When working with quadratic meshes, only outer edges of non-linear cells are extracted. 
  * Throws a MiCancellationException if cancelled (see MiExtractorCallback).
  * @param cellFilter The cell filter.
  * @return The line skin extracted as an unstructured line mesh.
  */
  virtual const MeXLineMeshUnstructured& extractLineSkin(const MiCellFilterI* cellFilter=NULL) = 0;

  /**
  * Builds the topology and geometry of the point skin.
  * The point skin of the given mesh is made up of the nodes from the faces that belong to the skin.
  * Throws a MiCancellationException if cancelled (see MiExtractorCallback).
  * @param cellFilter The cell filter.
  * @return the skin extracted as a point mesh.
  */
  virtual const MeXPointMeshCurvilinear& extractPoints(const MiCellFilterI* cellFilter=NULL) = 0;
  
  /**
  * Extracts a scalar set from the volume mesh. The scalars are mapped onto the 
  * current extracted skin refering to either the line skin or the surface skin depending on
  * whether extractSkin() or extractLineSkin() was last called.
  *
  * @param inputSet the input scalar set defined on the volume mesh
  * @return the output scalar set defined on the latest extracted mesh
  */
  virtual const MeXScalardSetI& extractScalarSet(const MiScalardSetI& inputSet) = 0;

  /**
  * Extracts a vector set from the volume mesh. The vectors are mapped onto the 
  * current extracted skin refering to either the line skin or the surface skin depending on
  * whether extractSkin() or extractLineSkin() was last called.
  *
  * @param inputSet the input vector set defined on the volume mesh
  * @return the output vector set defined on the latest extracted mesh
  */
  virtual const MeXVec3dSetI& extractVec3Set(const MiVec3dSetI& inputSet) = 0;
 
  /**
  * @copydoc MiSkinExtractIjk::getExtract()
  */
  virtual const MeXSurfaceMeshUnstructured& getExtract() const = 0;

  /**
  * Returns the result of the extraction as a line mesh.
  */
  virtual const MeXLineMeshUnstructured& getLineExtract() const = 0;

  /**
  * Returns the result of the extraction as a point mesh.
  */
  virtual const MeXPointMeshCurvilinear& getPointExtract() const = 0;

};

#endif /* _MISKINEXTRACTUNSTRUCURED_H */


