/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  _MO_MESHCELLSHAPES_
#define  _MO_MESHCELLSHAPES_

#include <MeshVizXLM/MbVec3.h>

#include <MeshVizXLM/mapping/nodes/MoMeshSurfaceRepresentation.h>

#include <Inventor/fields/SoMFInt32.h>
#include <MeshVizXLM/extractors/MiCellExtractIj.h>
#include <MeshVizXLM/extractors/MiCellExtractIjk.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterI.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterIj.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterIjk.h>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

class MiCellExtractUnstructured;
class MiCellExtractHexahedronIjk;

class MdMeshRepresentation;

class MiVolumeMeshHexahedronIjk;
class MiTopologyI;
class MiGeometryI;

class MxDefaultStreamStringSet;

/**
 * @DTEXT  Rendering node that displays a set of cells of a mesh
 * 
 * @ingroup MeshVizXLM_Mapping_Representation
 * 
 * @DESCRIPTION
 *  This node renders an unstructured surface mesh defined by a set of cells in the original mesh.
 *  The set of cells to be rendered is specified by a list of cell indices. 
 *  The size of the rendered cells can be scaled up or down using the scaleFactor field and
 *  annotation of the cell and node names can optionally be displayed.
 *
 *  The cell name annotation is defined in the following way:
 *
 *    - If an MoStringSet node is inherited in the scene graph with the suitable type and PER_CELL binding,
 *      the cell name is defined by the string returned by MiStringSetI::get or MiStringSetIj::get
 *      or MiStringSetIjk::get.
 *      The suitable type depends on the mesh type handled by the inherited MoMesh instance. For instance
 *      if the MoMesh handles an unstructured mesh (see MiVolumeMeshUnstructured and
 *      MiSurfaceMeshUnstructured) a suitable string set as cell name is of type MiStringSetI. If
 *      the MoMesh handles an MiVolumeMeshHexahedronIjk, a suitable string set as cell name is of
 *      type MiStringSetIjk.
 *      If several MoStringSet are inherited with the suitable string set type and PER_CELL binding,
 *      the last one on the state list is used.
 *
 *    - If no suitable MoStringSet is inherited in the scene graph, the cell name is defined
 *      by the string corresponding to the serialization of its cell ids. For Ij or Ijk mesh
 *      type the ij or ijk ids are separated by commas.
 *
 *  The node name annotation is defined in the same way but the suitable binding is
 *  obviously PER_NODE.
 *
 *  The cells can be colored using a scalar set defined in the colorScalarSetId inherited field.
 *  This is an index into the list of scalar sets existing in the traversal state
 *  (see the MoScalarSetxxx nodes). To disable coloring set this field to -1.
 *
 * @FILE_FORMAT_DEFAULT
 *    MeshCellShape {
 *    @TABLE_FILE_FORMAT
 *       @TR colorScalarSetId           @TD 0
 *       @TR parallel                   @TD TRUE
 *       @TR inclusiveList              @TD TRUE
 *       @TR cellIndices                @TD 0
 *       @TR scaleFactor                @TD 1.0
 *       @TR showCellName               @TD FALSE
 *       @TR showNodeName               @TD FALSE
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *   MiCellExtractUnstructured, MiCellExtractHexahedronIjk,
 *   MiCellExtractIjk, MiCellExtractIj
 * 
 */

class MESHVIZXLM_DMAP_API MoMeshCellShape : public MoMeshSurfaceRepresentation {

  SO_NODE_HEADER(MoMeshCellShape) ;

 public:

  /**
   * Constructor.
   */
  MoMeshCellShape() ;

  /**
  * When set to TRUE, the cellIndices field will represent the list of cells to display and
  * the list of cells to exclude otherwise. Default is TRUE. @n
  */
  SoSFBool inclusiveList;

  /**
  * Field representing the list of cells given by their index in the mesh.@n
  * If the mesh is unstructured, each value of cellIndices represents one cell.@n
  * If the mesh is a structured surface mesh, each cell is defined by a pair of indices (cellIndices(i),cellIndices(i+1)).@n
  * If the mesh is a structured volume mesh, each cell is defined by three indices (cellIndices(i),cellIndices(i+1),cellIndices(i+2)).@n
  * So, depending on the type of mesh, one, two or three indices are necessary to define each cell.
  * @note If the index list is empty, no cells are displayed when inclusiveList is set to TRUE and
  * all cells otherwise. Default is empty.
  */
  SoMFInt32  cellIndices;

  /**
  * Field representing the factor to resize extracted cells.@n
  * A factor lower than one will shrink cells, whereas a factor greater than one will expand cells.@n
  * Default is 1.0, which will not resize cells.
  */
  SoSFFloat factor;

  /**
  * When set to TRUE, the cell's name is displayed near the cell. Default is FALSE. @n
  * The documentation of the class MoMeshCellShape describes how the cell's name is defined.
  */
  SoSFBool showCellName;

  /**
  * When set to TRUE, the node's names are displayed near the cell. Default is FALSE. @n
  * The documentation of the class MoMeshCellShape describes how the node's name is defined.
  */
  SoSFBool showNodeName;

  /**
  * Field representing the distance to shift the display of node names away from the
  * node coordinates.@n
  * A negative offset will display node names inside the cell, whereas a positive offset 
  * will display them outside the cell.@n
  * @note The offset is relative to the cell's size if the relativeOffset field is set to TRUE.
  * Otherwise it represents an absolute distance.
  * Default is 0.1, which will slightly shift node names outside the cell.
  */
  SoSFFloat offset;

  /**
  * When set to TRUE, the distance between the node and the name to display is relative to 
  * the size of the cell@n
  * @note For unstructured meshes, the offset field is then multiplied by the getRelativeSize() method 
  * from the cell interface. A default implementation of getRelativeSize() returns 1 which will
  * leave the offset as an absolute distance. getRelativeSize() needs to be overloaded to activate
  * this feature for unstructured mesh (see class MiCell).
  * Default is FALSE.
  */
   SoSFBool relativeOffset;

  //@{
  /** 
  * Get extractor for reuse outside this node. 
  * @note The returned pointer may be null if a mesh has not been set in the scene graph 
  * or if no render action has traversed the scene graph. 
  * Use the method corresponding to the type of mesh stored in the scene graph. 
  * 
  */
  const MiCellExtractUnstructured*    getUnstructuredExtractor();
  const MiCellExtractHexahedronIjk* getHexahedronIjkExtractor();
  const MiCellExtractIjk*             getIjkExtractor();
  const MiCellExtractIj*              getIjExtractor();
  //@}

#if SoDEPRECATED_BEGIN(10000)
  SoDEPRECATED_METHOD(10000,"Use getHexahedronIjkExtractor() instead.")
  const MiCellExtractHexahedronIjk* getUnstructuredIjkExtractor();
#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(8500)

  SoDEPRECATED_METHOD(8500,"Use getIjkExtractor() instead.")
  const MiCellExtractIjk*             getRegularExtractor();

  SoDEPRECATED_METHOD(8500,"Use getIjExtractor() instead.")
  const MiCellExtractIj*              getSurfaceRegularExtractor();

#endif /** @DEPRECATED_END */

  /*----------------------------------------------------------------------------*/
 SoINTERNAL public:
  static void initClass() ;
	static void exitClass() ;

 SoEXTENDER public:
  virtual void doExtract(SoAction *action);

 protected:
  template <MeshType _meshType> friend class MdLineCellExtract;

  void clear();

  template<typename _Tuple> std::vector<_Tuple>& buildIndexList();
  
  std::vector<size_t>& buildIndexListI();
  std::vector<size_t> m_indexListI;
  std::vector<MiCellExtractIj::couple>& buildIndexListIj();
  std::vector<MiCellExtractIj::couple> m_indexListIj;
  std::vector<MiCellExtractIjk::triplet>& buildIndexListIjk();
  std::vector<MiCellExtractIjk::triplet> m_indexListIjk;

  template< MeshType _meshType>
  void doExtract(SoState * state,const MiMesh* mesh);

  const MiStringSetI* getCellNamesI(SoState* state, bool &isCellNamesModified);
  const MiStringSetIj* getCellNamesIj(SoState* state, bool &isCellNamesModified);
  const MiStringSetIjk* getCellNamesIjk(SoState* state, bool &isCellNamesModified);
  const MiStringSetI* getNodeNamesI(SoState* state, bool &isNodeNamesModified);

  size_t m_timeStampOfLastCellNames;
  size_t m_timeStampOfLastNodeNames;

  class DisplayName
  {
  public:
    DisplayName() {}
    DisplayName(std::string name,const MbVec3d& pos) : m_name(name), m_pos(pos) {}
    std::string m_name; // Name of the cell
    MbVec3d     m_pos;  // Where to display the name
    bool operator==(const DisplayName& c1) const {return m_name == c1.m_name && m_pos ==c1.m_pos;}
  };
  std::vector<DisplayName> m_cellNames;
  std::vector<DisplayName> m_nodeNames;

  template <typename _Iterator, typename _Topology, typename _Tuple>  
  void getDisplayCellList(const _Topology& topology, std::vector<_Tuple>& indexList, std::vector<_Tuple>& cellIds);

  void buildDisplayNamesListI(SoState* state, const MiMeshUnstructuredI& mesh, const MiCellFilterI* cellFilter);
  void buildDisplayNamesListIj(SoState* state, const MiTopologyIj& topology, const MiGeometryI& geometry, const MiCellFilterIj* cellFilter);
  void buildDisplayNamesListIjk(SoState* state, const MiVolumeMeshHexahedronIjk& mesh, const MiCellFilterIjk* cellFilter, bool displayNodes = true);

  void displayNames(std::vector<DisplayName>& newDisplayNames, std::vector<DisplayName>& currentDisplayNames, SoSeparator*& nameSep);

  SoSeparator* m_nodeNamesSep;
  SoSeparator* m_cellNamesSep;

  class CellFilter : public MiCellFilterI, public MiCellFilterIj, public MiCellFilterIjk
  { 
  public:
    CellFilter();
    bool acceptCell(size_t cellIndex) const;
    bool acceptCell(size_t i, size_t j) const;
    bool acceptCell(size_t i, size_t j, size_t k) const;
    size_t getTimeStamp() const;
    void set(std::vector<size_t>& exclusionListI, const MiCellFilterI* userCellFilter);
    void set(std::vector<MiCellExtractIj::couple>& exclusionListIj, const MiCellFilterIj* userCellFilter);
    void set(std::vector<MiCellExtractIjk::triplet>& exclusionListIjk, const MiCellFilterIjk* userCellFilter);
  private:
    template < typename _vectorT > void copy( _vectorT& exclusionListI, _vectorT& newListI);
    std::vector<size_t> m_exclusionListI;
    std::vector<MiCellExtractIj::couple> m_exclusionListIj;
    std::vector<MiCellExtractIjk::triplet> m_exclusionListIjk;
    const MiCellFilterI*   m_userCellFilterI;
    const MiCellFilterIj*  m_userCellFilterIj;
    const MiCellFilterIjk* m_userCellFilterIjk;
    const MiCellFilter*    m_currentCellFilter;
    mutable size_t m_userTS;
    mutable size_t m_timeStamp;
  };
  CellFilter m_excludedCellFilter;

  static MxDefaultStreamStringSet* s_streamStringSet;
} ;

template <>
inline std::vector<size_t>& MoMeshCellShape::buildIndexList()
{ 
  return buildIndexListI(); 
}

template <>
inline std::vector<MiCellExtractIj::couple>& MoMeshCellShape::buildIndexList()
{ 
  return buildIndexListIj(); 
}

template <>
inline std::vector<MiCellExtractIjk::triplet>& MoMeshCellShape::buildIndexList()
{ 
  return buildIndexListIjk(); 
}


#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /* _MO_MESHCELLSHAPES_ */



