/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MiGeometryHexahedronIjk_H
#define _MiGeometryHexahedronIjk_H

#include <MeshVizXLM/mesh/MiMeshIjk.h>
#include <MeshVizXLM/mesh/geometry/MiGeometry.h>

/**
* @DTEXT Generic geometry interface for hexahedron mesh Ijk.
* 
* @ingroup MeshVizXLM_Mesh_Geometry
* 
* @DESCRIPTION
*    Generic geometry interface representing a 3D grid of hexahedron cells,
*    where each hexahedron is defined explicitly with 8 independent coordinates.
*    Each hexahedron is accessible via 3 indices (i,j,k).
*
*    This type of geometry is typically used by the MiVolumeMeshVertexHexahedronIjk interface. 
* 
*    The vertices of each hexahedron are numbered as follows (see getCellCoords for more details):
* \verbatim
                         J
                         |
                         |

                        n3----------n2
                        /|          /|
                      /  |        /  |
                    /    |      /    |
                  n7---------n6      |
                   |     |    |      |
                   |    n0----|-----n1    --- I
                   |    /     |     /
                   |  /       |   /
                   |/         | /
                  n4---------n5

                /
              /
            K
  \endverbatim
* 
*/
class MESHVIZXLM_EXTR_API MiGeometryHexahedronIjk : virtual public MiGeometry
{
public:
  /**
  * Returns the coordinates of the hexahedron (i,j,k) cell.
  * The vertices of each hexahedron are numbered as follows :
  *   - The first 4 vertices define the "bottom" face of the hexahedron. 
  *     This face is on the logical IJ plane K=k.
  *   - The last 4 vertices define the "top" face of the hexahedron. 
  *     This face is on the logical IJ plane K=k+1.
  * @IMAGE MiGeometryHexahedronIjk.png "Vertices numbering"
  *
  * @param i [in] the id of this hexahedron on the axis I. 0 <= i < meshTopologyIjk.getNumCellsI().
  * @param j [in] the id of this hexahedron on the axis J. 0 <= j < meshTopologyIjk.getNumCellsJ().
  * @param k [in] the id of this hexahedron on the axis K. 0 <= k < meshTopologyIjk.getNumCellsK().
  * @param v0,v1,v2,v3,v4,v5,v6,v7 [out] the 8 vertices of the hexahedron (i,j,k).
  */
  virtual void getCellCoords(size_t i, size_t j, size_t k, MbVec3d& v0, MbVec3d& v1, MbVec3d& v2, MbVec3d& v3,
                                                           MbVec3d& v4, MbVec3d& v5, MbVec3d& v6, MbVec3d& v7) const = 0;

  /**
  * Hint about hexahedron organization in memory. 
  * This information aims at optimizing the geometry traversal for extractors.
  * Returns the geometry internal hexahedron layout in memory as 6 enum values covering all possible cases. 
  *
  * For example, LAYOUT_KJI means that: 
  *   - the hexahedron (i+1,j,k) is consecutive in memory to the hexahedron (i,j,k) and,
  *   - the row of hexahedron (j+1,k) is consecutive in memory to the row of hexahedron (j,k) and,
  *   - the slice of hexahedron (k+1) is consecutive in memory to the slice of hexahedron (k) 
  *
  * The fastest way to go through a hexahedron mesh with LAYOUT_KJI should be to perform the following triple loop:
  * @verbatim
    for each k with 0 <= k < numK
     for each j with 0 <= j < numJ
       for each i with 0 <= i < numI
         geometry.getCellCoords(i,j,k,n0, ... ,n7);
    @endverbatim
  * The default implementation returns LAYOUT_UNKNOWN meaning that the hexahedron storage organization is unknown, 
  * for instance when hexahedron are computed on the fly.
  * In that case, extractors perform the hexahedron traversal in the way that minimizes the required 
  * memory for their algorithms. 
  */
  virtual MiMeshIjk::StorageLayout getStorageLayout() const { return MiMeshIjk::LAYOUT_UNKNOWN; }
};

#endif


