/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MiHexahedronTopologyExplicitIjk_H
#define _MiHexahedronTopologyExplicitIjk_H

#include <MeshVizXLM/mesh/MiMeshIjk.h>
#include <MeshVizXLM/mesh/topology/MiTopologyIjk.h>

//-----------------------------------------------------------------------------
/**
* @DTEXT List of volume cells interface.
* 
* @ingroup MeshVizXLM_Mesh_Topology
* 
* @DESCRIPTION
*    A generic abstract interface for an unstructured explicit hexahedron volume mesh topology. @BR
*
*    Each hexahedron cell has 6 faces which can be in any plane of
*    the 3D space. Each face is a quadrangle not necessarily a square.
*    The 4 first indices define the polygon of a face of the
*    hexahedron. The 4 last indices define the polygon of the 
*    opposite face.
*
*    Facets and nodes are numbered as follows :
* \verbatim
                         J
                         |
                         |

                        n3----------n2   facet 0 = 0123
                        /|          /|   facet 1 = 4765
                      /  |        /  |   facet 2 = 0374
                    /    |      /    |   facet 3 = 1562
                  n7---------n6      |   facet 4 = 0451
                   |     |    |      |   facet 5 = 3267
                   |    n0----|-----n1    --- I
                   |    /     |     /
                   |  /       |   /
                   |/         | /
                  n4---------n5

                /
              /
            K
  \endverbatim
*
* @SEE_ALSO
*   MbHexahedronTopologyExplicitIjk
* 
*/
class MESHVIZXLM_EXTR_API MiHexahedronTopologyExplicitIjk : public MiTopologyIjk
{
public:
  /**
  * Returns the 8 indices of a cell of this topology. The cell 
  * is identified by three ids i,j,k.
  */
  virtual void getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                                                size_t& n4, size_t& n5, size_t& n6, size_t& n7)  const = 0;

  /**
  * Hint about cells organization in memory. 
  * This information aims at optimizing the topology traversal for extractors.
  * Returns the topology internal cells layout in memory as 6 enum values covering all possible cases. 
  *
  * For instance LAYOUT_KJI means that: 
  *   - the cell (i+1,j,k) is consecutive in memory to the cell (i,j,k) and,
  *   - the row of cells (j+1,k) is consecutive in memory to the row of cells (j,k) and,
  *   - the slice of cells (k+1) is consecutive in memory to the slice of cells (k) 
  *
  * The fastest way to go through the cells with LAYOUT_KJI should be to perform the following triple loop:
  * @verbatim
    for each k with 0 <= k < numK
     for each j with 0 <= j < numJ
       for each i with 0 <= i < numI
         topology.getCellNodeIndices(i,j,k,n0, ... ,n7);
    @endverbatim
  * The default implementation returns LAYOUT_UNKNOWN meaning that the cell storage organization is unknown, 
  * for instance when cells are computed on the fly.
  * In that case, extractors perform the topology traversal in the way that minimizes the required 
  * memory for their algorithms. 
  */
  virtual MiMeshIjk::StorageLayout getStorageLayout() const { return MiMeshIjk::LAYOUT_UNKNOWN; }

  /**
  * @copydoc MiVolumeTopologyExplicitI::getEndNodeId()
  */
  virtual size_t getEndNodeId() const = 0;

  /**
  * @copydoc MiVolumeTopologyExplicitI::getBeginNodeId()
  */
  virtual size_t getBeginNodeId() const { return 0; }
};


#endif


