/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <string>
#include <memory>
#include <vector>

namespace RemoteViz
{
  namespace Rendering
  {

    class ClientImpl;
    class Connection;
    class ClientSettings;
    struct ClientParameters;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Represents a client application instance using RemoteViz.
     *
     *  In the case of HTML5 applications, a client represents a web page. See RemoteVizRenderArea.
     *
     *  In the case of the SoRemoteVizClient node, a client represents a single
     *  instance of the application using the node.
     *
     * When a client first connects to the RemoteViz service (e.g by calling the
     * RemoteVizRenderArea method connectTo()), a Client object is automatically
     * created and the ServiceListener method onConnectedClient() is called with
     * the client id. The client object can be queried using Service::getClient().
     * A Connection object is also created. A RenderArea object is created if the
     * requested render area does not already exist. In any case the Connection
     * can be queried using the #getConnection() method and the associated
     * RenderArea can be queried using the Connection::getRenderArea() method.
     *
     * A service has zero clients initially and may allow one client or multiple
     * clients (see ServiceListener methods onPendingCreateRenderArea and
     * onPendingShareRenderArea). While the service is running, you can get the
     * current client(s) using the Service methods getNumClients() and getClient().
     *
     * A client has at least one connection but may have multiple connections if
     * the service allows that. See the methods #getNumConnections() and #getConnection().
     *
     * When a client is disconnected, the ServiceListener method onDisconnectedClient()
     * is called. Following this call, the client object will no longer be referenced by
     * RemoteViz and will be disposed unless the application holds a reference.
     *
     * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */

    class RENDERSERVICE_API Client {

      /*! \cond PRIVATE */
      friend class ServiceImpl;
      friend class ConnectionImpl;
      /*! \endcond */

    public:

      /**
       *  Gets the state of the Client: connected or disconnected.
       *  Connected (true) means the Client has at least one active Connection.
       *  When a Client is being disconnected, the ServiceListener method onDisconnectedClient()
       *  is called. Just before this call, the Client state is set to "disconnected".
       *  Following this call, the Client object will no longer be referenced by
       *  RemoteViz and will be disposed unless the application holds a reference.
       *  RemoteViz will never reuse the object internally.
       *
       *  \return true if the client is connected or false if the client is disconnected.
       *  [OIV-WRAPPER PROPERTY{IsConnected},GETTER]
       */
      bool isConnected() const;
      /**
       *  Gets the id of the client.
       *  The client id is a Globally Unique Identifier (GUID) created by RemoteViz.
       *
       *  \return the ID identifying the client
       */
      const std::string& getId() const;
      /**
       *  Gets a Connection of the client.
       *
       *  \param id : ID identifying the connection.
       *
       *  \return the connection object if the id exists, otherwise returns null.
       */
      std::shared_ptr<Connection> getConnection(const std::string& id) const;
      /**
       *  Gets a Connection of the client.
       *  See #getNumConnections().
       *
       *  \param index : index identifying the client
       *
       *  \return the connection object if the index exists, otherwise returns null.
       */
      std::shared_ptr<Connection> getConnection(unsigned int index) const;
      /**
       *  Gets the number of client connections.
       *
       *  \return the number of connections
       */
      unsigned int getNumConnections() const;
      /**
       * HTML5 client: Returns the domain name of the web host to which the client is connected. @BR
       * Specifically, RemoteVizClient.js uses the JavaScript command "window.location.host".
       * Note that this command returns an empty string when the URL is a file path (starting
       * with "file://"). In this case, there is no domain name.
       *
       *  SoRemoteVizClient node: Returns the string "SoRemoteVizClient".
       *
       *  \return the application name.
       */
      const std::string& getApplicationName() const;
      /**
       *  HTML5 client: Returns the value of the user-agent header sent by the client web browser.
       *  SoRemoteVizClient node: Returns the operating system (OS) running the client application.
       *
       *  \return the environment.
       */
      const std::string& getEnvironment() const;
      /**
       *  Returns if the client supports image streaming.
       *
       *  \return true if image streaming is supported.
       */
      bool isImageStreamingSupported() const;
      /**
       *  Returns if the client supports video streaming.
       *
       *  \return true if video streaming is supported.
       */
      bool isVideoStreamingSupported() const;
      /**
       *  Sends a text message to all the connections of the client.
       *
       *  \param message : the message to be sent to all the connections of the client
       *
       *  \param excludedConnections : optional - the connections that should not receive the message.
       *
       *  \return true if the message has been successfully sent to all connections, otherwise returns false.
       */
      bool sendMessage(const std::string& message, std::vector<std::shared_ptr<Connection>> excludedConnections = {}) const;
      /**
       *  Sends a binary message to all the connections of the client.
       *
       *  \param buffer : the binary buffer to be sent to all connections of the client
       *
       *  \param excludedConnections : optional - the connections that should not receive the message.
       *
       *  \return true if the message has been successfully sent to all connections, otherwise returns false.
       */
      bool sendMessage(const std::vector<unsigned char>& buffer, std::vector<std::shared_ptr<Connection>> excludedConnections = {}) const;
      /**
       *  Disconnects the client. A KICKED disconnect message will be sent to all connections of the client, 
       *  then they will be disconnected.
       */
      void disconnect();
      /**
       *  Gets the client settings.
       *
       *  \return the client settings object
       */
      std::shared_ptr<ClientSettings> getSettings() const;

    protected:
      /*! \cond PRIVATE */
      std::shared_ptr<ClientImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<ClientImpl> pImpl;

      /** Constructor */
      Client(const std::string &id, const ClientParameters& clientParameters);

      /** Deleted constructors */
      Client(const Client&) = delete;
      Client& operator= (const Client&) = delete;

      /** Create an instance of the class */
      static std::shared_ptr<Client> createInstance(const std::string &id, const ClientParameters& clientParameters);

    };

  }
}
