/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <memory>

namespace RemoteViz
{
  namespace Rendering
  {

    class ClientSettingsImpl;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Settings that define a Client.
     *
     *  The application can query this object from a Client object.
     *  A Client object is passed to various listener methods such as
     *  ServiceListener#onConnectedClient and ServiceListener#onInitializedClient.
     *  Clients can also be queried using Service#getClient().
     *
     *  ClientSettings is used to set the network bandwidth to be used by the Client.
     *  This value is not a hard limit, but higher values imply higher image quality and
     *  higher frames per second. The policy can be set for each connection (see
     *  ConnectionSettings#setFrameEncodingPolicy). If a client has multiple connections,
     *  the bandwidth value for each connection is the client value divided by the number of connections.
     *
     *  If network calibration is enabled (true by default, see NetworkPerformance), when
     *  a client connects to the service, RemoteViz will measure the available network bandwidth.
     *  The default implementation of ServiceListener#onInitializedClient() will query this
     *  value (NetworkPerformance#getBandwidth) and call #setBandwidth(). If the application
     *  already knows the available network bandwidth, it is not necessary to enable
     *  calibration, simply call #setBandwidth().
     *
     *  To measure available network bandwidth, RemoteViz has to cause network congestion,
     *  which implies that the user is blocked for a period of time. Therefore RemoteViz only
     *  measures network bandwidth @I once@i, when the client first connects (and only if
     *  calibration is enabled). Of course network bandwidth can change at any time. If the
     *  application has new information about the available network bandwidth, #setBandwidth()
     *  can be called at any time.
     *
     *  The application can query this object from a Client object.
     *  The Client object can obtained in, e.g., ServiceListener::onPendingCreateRenderArea().
     *
     * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API ClientSettings
    {

      /*! \cond PRIVATE */
      friend class ClientImpl;
      friend class ConnectionImpl;
      friend class ServiceImpl;
      /*! \endcond */

    public:

      //@{
      /**
      *  Sets the network bandwidth to be used by the client.
      *  Default value is 800Mbps.
      *
      *  This value is not a hard limit, but it influences the image quality and the
      *  number of interactive frames (frames per second) sent to the client. The
      *  policy can be set for each connection (see ConnectionSettings#setFrameEncodingPolicy).
      *  The higher the value, the better the image quality and the number of interactive frames.
      *  See discussion in the class description.
      *
      *  If the client has more than one connection, the bandwidth allowed for each
      *  connection is this value divided by the number of connections.
      *
      *  \param val : network bandwidth in bits per second
      *
      *  @see ServiceListener#onConnectedClient
      */
      void setBandwidth(unsigned int val);
      /**
      *  Gets the maximum network bandwidth to be used by the client.
      *  Default value is 800Mbps.
      *
      *  This value influences the quality and the number of interactive frames sent to the client (see ConnectionSettings#setFrameEncodingPolicy).
      *  The higher the value, the better the quality and the number of interactive frames.
      *
      *  \return the network bandwidth in bits per second
      *
      *  @see ServiceListener#onConnectedClient
      */
      unsigned int getBandwidth() const;
      //@}

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<ClientSettingsImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Constructor */
      ClientSettings();

      /** Deleted constructors */
      ClientSettings(const ClientSettings&) = delete;
      ClientSettings& operator= (const ClientSettings&) = delete;

      /** Pointer to implementation */
      std::shared_ptr<ClientSettingsImpl> pImpl;
    };

  }
}
