/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>
#include <RemoteViz/Rendering/EncodedFrame.h>

#include <string>
#include <memory>
#include <vector>

namespace RemoteViz
{
  namespace Rendering
  {

    class RenderArea;

    class ConnectionImpl;

    class Client;

    class ConnectionSettings;

    class ConnectionParameters;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Represents a connection from a Client to a RenderArea managed by the RemoteViz service.
     *
     *  A Connection object is created automatically by RemoteViz when a client connects
     *  to the service. Each connection belongs to a single Client and a single RenderArea.
     *  Each Client may have multiple connections, for example when multiple render areas
     *  are created on the same browser page. Each RenderArea may have multiple connections
     *  if it is shared, i.e. displayed in multiple client render areas.
     *  Use the Service object to query a client, then use the Client object to query its connections.
     *
     *  A Connection object is passed to many of the RenderAreaListener methods including
     *  onOpenedConnection(), onInitializedConnection() and onSendingFrame().
     *
     *  Using a Connection object, the application can query the current width and height
     *  of the client "container" in pixels and the last RenderArea width and height
     *  requested by the client.  Note that the render area and the container are not
     *  required to be the same size. If the aspect ratios are different, then the
     *  rendered image will be displayed inside the container according to the
     *  "displayMode" specified when the client called the RemoteVizRenderArea constructor
     *  or set the 'displayMode' field of the SoRemoteVizClient node. Possible values
     *  are FIT (default), CROP and STRETCH.
     *
     *  A Connection allows the application to send text or binary messages to the associated
     *  client. A JavaScript client will receive the message using a listener (see
     *  the RemoteVizRenderArea method addServiceListener).  Messages from a client
     *  are received using one of the RenderAreaListener::onReceivedMessage() methods.
     *
     * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API Connection {

      /*! \cond PRIVATE */
      friend class ConnectionManager;
      friend class RenderAreaImpl;
      friend class ClientImpl;
      friend class KeepFramesPerSecondPolicyImpl;
      friend class KeepFrameQualityPolicyImpl;
      /*! \endcond */

    public:

      /**
       *  Gets the state of the Connection: open or closed.
       *  RemoteViz does not keep a reference to the Connection object after the
       *  connection is closed. If the application kept a reference to the object,
       *  this method can be used to query its state. If the application did not
       *  keep a reference, then the Connection object will be disposed after the
       *  connection is closed.
       *
       *  \if_cpp
       *  Note: RemoteViz objects are not reference counted like Open Inventor nodes,
       *  but std::shared_ptr provides a similar mechanism.
       *  \endif
       *
       *  The state "closed" will be set just before triggering the listener RenderAreaListener#onClosedConnection.
       *
       *  \return true if the connection is open or false if the connection is closed.
       *  [OIV-WRAPPER PROPERTY{IsOpen},GETTER]
       */
      bool isOpen() const;
      /**
       *  Gets the id of the connection.
       *  The id is a Globally Unique Identifier (GUID) created by RemoteViz.
       *
       *  \return the ID identifying the connection
       */
      const std::string& getId() const;
      /**
       *  Gets the connection settings.
       *
       *  \return the connection settings object
       */
      std::shared_ptr<ConnectionSettings> getSettings() const;
      /**
       *  Gets the connection parameters. These are field-value pairs included in the url
       *  during the client connection. See RemoteVizRenderArea#connectTo().
       *
       *  \return the connection parameters object
       */
      std::shared_ptr<const ConnectionParameters> getParameters() const;
      /**
       *  Gets the RenderArea associated with the connection.
       *
       *  \return the renderArea object
       */
      std::shared_ptr<RenderArea> getRenderArea() const;
      /**
       *  Gets the last renderArea width requested by the client.
       * 
       *  \return the requested renderArea width
       */
      unsigned int getRequestedWidth() const;
      /**
       *  Gets the last renderArea height requested by the client.
       * 
       *  \return the requested renderArea height
       */
      unsigned int getRequestedHeight() const;
      /**
       *  Gets the client container width.
       * 
       *  \return the client container width
       */
      unsigned int getContainerWidth() const;
      /**
       *  Gets the client container height.
       * 
       *  \return the client container height
       */
      unsigned int getContainerHeight() const;
      /**
       *  Gets the client associated with this connection.
       *
       *  \return the ID identifying the client
       */
      std::shared_ptr<Client> getClient() const;
      /**
       *  Sends a text message to the client. The client-side message event will be triggered.
       *
       *  \param message : the message to be sent to the client
       *
       *  \return true if the message has been successfully sent, otherwise returns false.
       */
      bool sendMessage(const std::string& message) const;
      /**
       *  Sends a binary message to the client. The client-side message event will be triggered.
       *
       *  \param buffer : the binary message to be sent to the client
       *
       *  \return true if the message has been successfully sent, otherwise returns false.
       * [OIV-WRAPPER-ARG ARRAY,NO_WRAP&ARRAY_LENGTH{buffer}]
       */
      bool sendMessage(const std::vector<unsigned char>& buffer) const;
      /**
       *  Closes the connection. A KICKED disconnect message will be sent to the client.
       *
       */
      void close();
      /**
       *  Gets the last encoded frame sent by the connection
       *
       *  \return encoded frame.
       */
      EncodedFrame getLastEncodedFrame() const;

    protected:
      /*! \cond PRIVATE */
      std::shared_ptr<ConnectionImpl> getImpl() const;
      /*! \endcond */

    private:	
      /** Pointer to implementation */
      std::shared_ptr<ConnectionImpl> pImpl;

      /** Constructor */
      Connection();

      /** Deleted constructors */
      Connection(const Connection&) = delete;
      Connection& operator= (const Connection&) = delete;

      /** Create an instance of the class */
      static std::shared_ptr<Connection> createInstance();

    };

  }
}
