/*=======================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
***                                                                                ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
***                                                                                ***
***                        RESTRICTED RIGHTS LEGEND                                ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
***                                                                                ***
***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
***                        BORDEAUX, FRANCE                                        ***
***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <memory>
#include <string>

namespace RemoteViz
{
  namespace Rendering
  {
    class HTTPRequestImpl;
    class HTTPHeaders;
    class Uri;

    /**
    * @RVEXT
    *
    * @ingroup RemoteViz
    *
    * @DESCRIPTION
    * This class encapsulates an HTTP request message received by the service.
    *
    * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
    * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
    */
    class RENDERSERVICE_API HTTPRequest
    {
    public:

      /**
       * Creates a GET / HTTP/1.1 HTTP request.
       */
      HTTPRequest();

      /**
       * Creates a HTTP request with the given method, URI (Uniform Resource Identifier) and version.
       */
      HTTPRequest(const std::string& method, const std::string& uri, int versionMajor, int versionMinor);

      /**
       * Creates a HTTP request with the given method, URI (Uniform Resource Identifier), version and headers.
       */
      HTTPRequest(const std::string& method, const std::string& uri, int versionMajor, int versionMinor, const HTTPHeaders& headers);

      /**
       *  Copy constructor
       */
      HTTPRequest(const HTTPRequest& obj);

      /**
       *  Copy assignment operator
       */
      HTTPRequest& operator=(const HTTPRequest& obj);

      /**
       * Destructor
       */
      ~HTTPRequest();

      /**
       *  Returns the URI (Uniform Resource Identifier) being requested.
       */
      const Uri& getURI() const;

      /**
       *  Returns the request method ("GET", "HEAD", "POST", "PUT", "DELETE", "CONNECT", "OPTIONS", "TRACE", "PATH").
       */
      const std::string& getMethod() const;

      ///@{
      /**
       *  Returns the HTTP version for incoming request (example "HTTP/1.1").
       */
      const std::string& getVersion() const;

      /**
       *  HTTP uses a "major.minor" numbering scheme to indicate versions of the protocol.
       *  Returns the version major.
       */
      int getVersionMajor() const;

      /**
       *  HTTP uses a "major.minor" numbering scheme to indicate versions of the protocol.
       *  Returns the version minor.
       */
      int getVersionMinor() const;
      ///@}

      ///@{
      /**
       * Returns the value of the Content-Length header field.
       * The value -1 indicates that the length is unknown (Content-Length header is not present).
       */
      int64_t getContentLength() const;

      /**
       * Returns the value of the Content-Type header field.
       */
      const std::string& getContentType() const;

      /**
       * Returns the value of the Host header field.
       */
      const std::string& getHost() const;

      /**
       * Returns HTTP headers.
       */
      const HTTPHeaders& getHeaders() const;
      ///@}

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<HTTPRequestImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<HTTPRequestImpl> pImpl;
    };

  }
}
