/*=======================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
***                                                                                ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
***                                                                                ***
***                        RESTRICTED RIGHTS LEGEND                                ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
***                                                                                ***
***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
***                        BORDEAUX, FRANCE                                        ***
***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <memory>
#include <string>

namespace RemoteViz
{
  namespace Rendering
  {
    class HTTPResponseImpl;
    class HTTPHeaders;

    /**
    * @RVEXT
    *
    * @ingroup RemoteViz
    *
    * @DESCRIPTION
    * This class encapsulates an HTTP response message.
    *
    * Used in the ServiceListener onHTTPRequest() method.
    *
    * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{false}]
    * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY]
    */
    class RENDERSERVICE_API HTTPResponse
    {
    public:

      /**
      *  Enum for the HTTP status codes.
      */
      enum HTTPStatus
      {
        UNKNOWN = 0,

        /* 1xx - Informational */
        CONTINUE = 100,
        SWITCHING_PROTOCOLS = 101,
        PROCESSING = 102,
        EARLY_HINTS = 103,

        /* 2xx - Successful */
        OK = 200,
        CREATED = 201,
        ACCEPTED = 202,
        NON_AUTHORITATIVE_INFORMATION = 203,
        NO_CONTENT = 204,
        RESET_CONTENT = 205,
        PARTIAL_CONTENT = 206,
        MULTI_STATUS = 207,
        ALREADY_REPORTED = 208,
        IM_USED = 226,

        /* 3xx - Redirection */
        MULTIPLE_CHOICES = 300,
        MOVED_PERMANENTLY = 301,
        FOUND = 302,
        SEE_OTHER = 303,
        NOT_MODIFIED = 304,
        USE_PROXY = 305,
        TEMPORARY_REDIRECT = 307,
        PERMANENT_REDIRECT = 308,

        /* 4xx - Client Error */
        BAD_REQUEST = 400,
        UNAUTHORIZED = 401,
        PAYMENT_REQUIRED = 402,
        FORBIDDEN = 403,
        NOT_FOUND = 404,
        METHOD_NOT_ALLOWED = 405,
        NOT_ACCEPTABLE = 406,
        PROXY_AUTHENTICATION_REQUIRED = 407,
        REQUEST_TIMEOUT = 408,
        CONFLICT = 409,
        GONE = 410,
        LENGTH_REQUIRED = 411,
        PRECONDITION_FAILED = 412,
        PAYLOAD_TOO_LARGE = 413,
        URI_TOO_LONG = 414,
        UNSUPPORTED_MEDIA_TYPE = 415,
        RANGE_NOT_SATISFIABLE = 416,
        EXPECTATION_FAILED = 417,
        MISDIRECTED_REQUEST = 421,
        UNPROCESSABLE_ENTITY = 422,
        LOCKED = 423,
        FAILED_DEPENDENCY = 424,
        UPGRADE_REQUIRED = 426,
        PRECONDITION_REQUIRED = 428,
        TOO_MANY_REQUESTS = 429,
        REQUEST_HEADER_FIELDS_TOO_LARGE = 431,
        CONNECTION_CLOSED_WITHOUT_RESPONSE = 444,
        UNAVAILABLE_FOR_LEGAL_REASONS = 451,
        CLIENT_CLOSED_REQUEST = 499,

        /* 5xx - Server Error */
        INTERNAL_SERVER_ERROR = 500,
        NOT_IMPLEMENTED = 501,
        BAD_GATEWAY = 502,
        SERVICE_UNAVAILABLE = 503,
        GATEWAY_TIMEOUT = 504,
        HTTP_VERSION_NOT_SUPPORTED = 505,
        VARIANT_ALSO_NEGOTIATES = 506,
        INSUFFICIENT_STORAGE = 507,
        LOOP_DETECTED = 508,
        NOT_EXTENDED = 510,
        NETWORK_AUTHENTICATION_REQUIRED = 511,
        NETWORK_CONNECT_TIMEOUT_ERROR = 599
      };

      /**
       * Creates a HTTP/1.1 response with "200 OK" status.
       */
      HTTPResponse();

      /**
       * Creates a HTTP/1.1 response with the given status and a standard reason phrase.
       */
      HTTPResponse(HTTPStatus statusCode);

      /**
       * Creates a HTTP/1.1 response with the given status and reason phrase.
       */
      HTTPResponse(HTTPStatus statusCode, const std::string& reason);

      /**
       * Creates a HTTP/1.1 response with the given status, reason phrase and message body data.
       */
      HTTPResponse(HTTPStatus statusCode, const std::string& reason, const std::string& body);

      /**
       * Creates a HTTP response with the given status, reason phrase, message body data, version and headers.
       */
      HTTPResponse(HTTPStatus statusCode, const std::string& reason, const std::string& body, int versionMajor, int versionMinor, const HTTPHeaders& headers);

      /**
       *  Copy constructor
       */
      HTTPResponse(const HTTPResponse& obj);

      /**
       *  Copy assignment operator
       */
      HTTPResponse& operator=(const HTTPResponse& obj);

      /**
       * Destructor
       */
      ~HTTPResponse();

      /**
       *  Returns the HTTP status.
       */
      HTTPStatus getStatusCode() const;

      /**
       *  Returns the HTTP reason phrase.
       */
      const std::string& getReason() const;

      ///@{
      /**
       *  Returns the HTTP version for incoming request (example "HTTP/1.1").
       */
      const std::string& getVersion() const;

      /**
       *  HTTP uses a "major.minor" numbering scheme to indicate versions of the protocol.
       *  Returns the version major.
       */
      int getVersionMajor() const;

      /**
       *  HTTP uses a "major.minor" numbering scheme to indicate versions of the protocol.
       *  Returns the version minor.
       */
      int getVersionMinor() const;
      ///@}

      /**
       * Returns the HTTP message body data (optional).
       * Returns an empty string if the response does not have a body message.
       */
      const std::string& getBody() const;

      /**
       * Returns the size of the body, in bytes (Content-Length entity header)
       */
      int64_t getContentLength() const;

      /**
       * Returns HTTP headers.
       */
      const HTTPHeaders& getHeaders() const;

      /**
       * Returns the standard HTTP reason phrase for a HTTP status code.
       */
      static std::string getReasonPhrase(HTTPStatus status);

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<HTTPResponseImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<HTTPResponseImpl> pImpl;

    };

  }
}
