/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <memory>
#include <RemoteViz/Rendering/IFrameEncodingPolicy.h>

namespace RemoteViz
{

  namespace Rendering
  {

    class Connection;
    class KeepFramesPerSecondPolicyImpl;

    /**
     *  @RVEXT
     *
     *  @ingroup RemoteViz
     *
     *  This class provides a calculation policy which allows to manage the FPS (frames per second) and the quality of
     *  interactive frames sent from Remoteviz service depending on the network bandwidth allocated to the Connection.
     *
     *  If the Client owns multiple connections, the network bandwidth allocated to each Connection will be equitably shared.
     *  Otherwise, the network bandwidth allocated to the Connection will be equal to the network bandwidth allocated to the Client. 
     *  The network bandwidth allocated to the Client can be defined from ClientSettings#setBandwidth.
     *  This policy works as described below:
     *  - When the network bandwidth allocated to the Connection decreases, the FPS value will keep the FPS target value 
     *    and the frame quality will decrease.
     *
     *  - If the frame quality decreases until the frame quality threshold, then the FPS will start to decrease.
     *
     *  - When the network bandwidth allocated to the Connection increases, the FPS will increase until it reaches the FPS 
     *    target and then the frame quality will increase until it reaches the frame quality target.
     *
     *  See ConnectionSettings#setFrameEncodingPolicy to define the frame encoding policy.
     *
     *  @SEE_ALSO
     *  KeepFrameQualityPolicy
     */
    class RENDERSERVICE_API KeepFramesPerSecondPolicy : public IFrameEncodingPolicy
    {
    public:
      /**
       *  Constructor.
       *  The FPS target will be set to 30. \n
       *  The frame quality target will be set to 0.7. \n
       *  The frame quality threshold will be set to 0.3. \n
       *
       */
      KeepFramesPerSecondPolicy();

      /**
       *  Convenience Constructor.
       *
       *  \param fpsTarget : FPS target, the value range is [1, 60].
       *  \param qualityTarget : frame quality target, the value range is [0, 1].
       *  \param qualityThreshold : frame quality threshold, the value range is [0, 1]. (qualityThreshold <= qualityTarget)
       *
       */
      KeepFramesPerSecondPolicy( unsigned int fpsTarget, float qualityTarget, float qualityThreshold );

      /**
       *  Destructor.
       *
       */
      virtual ~KeepFramesPerSecondPolicy();

      /**
       *  Gets the FPS calculated after calling the method update().
       *  The value range is [1, 60].
       *
       *  \return the FPS.
       */
      virtual unsigned int getFramesPerSecond() const override;

      /**
       *  Gets the interactive frame quality calculated after calling the method update().
       *  The value range is [0, 1].
       *
       *  \return the interactive frame quality.
       */
      virtual float getFrameQuality() const override;

      /**
       *  Update value of the FPS and frame quality.
       *  The connection will be used to get values such encode frame buffer size or bandwidth.
       *
       *  \param connection : connection used to get values.
       */
      virtual void update( std::shared_ptr<Connection> connection ) override;

    protected:
      /*! \cond PRIVATE */
      /** Returns a pointer to implementation */
      std::shared_ptr<KeepFramesPerSecondPolicyImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<KeepFramesPerSecondPolicyImpl> pImpl;
    };
  }
}
