/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <memory>
#include <SoDeprecationRules.h>

namespace RemoteViz
{
  namespace Rendering
  {

    class NetworkPerformanceImpl;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  This class manages measures of service quality of the network.
     *
     *  RemoteViz will pass an instance of this class to the methods
     *  ServiceListener#onConnectedClient() and ServiceListener#onInitializedClient().
     *
     *  If calibration is enabled (true by default), RemoteViz will measure available
     *  network bandwidth and latency when a client first connects to the service
     *  (between the onConnectedClient and onInitializedClient calls) and store the
     *  values in this object. These measures are used to adjust image quality or
     *  frames per second (see ConnectionSettings#setFrameEncodingPolicy).
     *  The default implementation of ServiceListener#onInitializedClient() will query
     *  the measured bandwidth value (#getBandwidth) and call ClientSettings#setBandwidth().
     *
     *  To measure available network bandwidth, RemoteViz has to cause network congestion,
     *  which implies that the user is blocked for a period of time. Therefore RemoteViz only
     *  measures network bandwidth @I once@i for each client, when the client first connects
     *  (and only if calibration is enabled). To disable calibration or change the duration
     *  of the calibration, implement the ServiceListener#onConnectedClient() method.
     *
     * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API NetworkPerformance
    {
      /*! \cond PRIVATE */
      friend class ClientSettingsImpl;
      friend class ClientImpl;
      /*! \endcond */

    public:

      //@{
      /**
      *  Enable or disable network performance calibration. Default value is true.
      *
      *  If calibration is enabled, a measurement of the latency and bandwidth is
      *  performed after the first connection establishment and before sending frames.
      *  If calibration is disabled, the estimated bandwidth and latency will be 0.
      *  It is better to reduce the calibration time rather than disable completely.
      *
      *  \param val : true to enable, false to disable
      *
      *  @see ServiceListener#onConnectedClient
      *
      * [OIV-WRAPPER PROPERTY{CalibrationEnabled},SETTER]
      */
      void enableCalibration(bool val);
      /**
      *  Return if the network performance calibration is enabled. Default is true.
      *
      *  \return true if enabled, false if disabled
      *
      * [OIV-WRAPPER PROPERTY{CalibrationEnabled},GETTER]
      */
      bool isCalibrationEnabled() const;
      //@}

      //@{
      /**
      *  Sets the network performance calibration maximum duration in milliseconds.
      *  Default value is 5 seconds.
      *  Frames will be send to the client after this duration.
      *  Reducing calibration time reduces the accuracy of the measurements.
      *
      *  \param val : max calibration duration in milliseconds
      */
      void setMaxCalibrationDuration(unsigned int val);
      /**
      *  Gets the network performance calibration maximum duration in milliseconds.
      *  Default value is 5 seconds.
      *  Frames will be send to the client after this duration.
      *
      *  \return the calibration maximum duration in milliseconds
      */
      unsigned int getMaxCalibrationDuration() const;
      //@}

      //@{
      /**
      *  Gets the estimated network bandwidth between the service and the client.
      *  Default is 0 when the object is first created, but RemoteViz will set
      *  this value if network calibration is enabled.
      *
      *  This value influences the quality and the number of interactive frames sent to the client
      *  (see ConnectionSettings#setFrameEncodingPolicy).
      *  The higher the value, the better the quality and the number of interactive frames.
      *  This value is estimated during the network calibration step (if enabled).
      *
      *  \return the network bandwidth in bit per seconds
      */
      unsigned int getBandwidth() const;
      //@}

      //@{
      /**
      *  Gets the estimated network latency (round-trip time) between the service and the client.
      *  Default is 0 when the object is first created, but RemoteViz will set
      *  this value if network calibration is enabled.
      *
      *  This value influences the quality of interactions.
      *  The lower the value, the better the user interactions.
      *  This value is estimated during the network calibration step (if enabled).
      *
      *  \return the network latency in milliseconds
      */
      unsigned int getLatency() const;
      //@}

    protected:
      /*! \cond PRIVATE */
      std::shared_ptr<NetworkPerformanceImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<NetworkPerformanceImpl> pImpl;

      /** Constructor */
      NetworkPerformance();

      /** Deleted constructors */
      NetworkPerformance(const NetworkPerformance&) = delete;
      NetworkPerformance& operator= (const NetworkPerformance&) = delete;

    };
  }
}
