/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <string>
#include <memory>
#include <vector>

class SoSceneManager;

class SoTouchManager;

class SoGLContext;

namespace RemoteViz
{
  namespace Rendering
  {

    class Connection;

    class RenderAreaImpl;

    class RenderAreaListener;

    class RenderAreaHardware;
    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Defines the rendering area for Open Inventor rendering.
     *  Implement the class RenderAreaListener to receive RenderArea events.
     *
     *  When using ServiceSettings#INVENTOR_SERVICE or ServiceSettings#INVENTOR_APPLICATION service mode,
     *  this class provides access to the scene manager and the touch manager to modify the rendering.
     *
     *  A RenderArea object may be created automatically by RemoteViz or created explicitly
     *  by the application.  When a client requests a connection (see
     *  RemoteVizRenderArea#connectTo()), it specifies a render area id.
     *  If a render area with that id does not exist, RemoteViz will create one and
     *  ServiceListener#onPendingCreateRenderArea() will be called.  If a render area with the
     *  requested id exists, RemoteViz will use that one and ServiceListener#onPendingShareRenderArea()
     *  will be called. An application might want to explicitly create a render area, for
     *  example, to preload a default scene graph.
     *
     *  A RenderArea may have zero, one or multiple connections.
     *
     *  A RenderArea allows the application to send text or binary messages to the associated
     *  client. A JavaScript client will receive the message using a listener (see
     *  RemoteVizRenderArea#addServiceListener).  Messages from a client
     *  are received using one of the RenderAreaListener::onReceivedMessage() methods.
     *
     * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API RenderArea {

      /*! \cond PRIVATE */
      friend class ServiceImpl;
      friend class ConnectionImpl;
      /*! \endcond */

    public:

      /**
       *  Gets the state of the RenderArea: alive or disposed.
       *  RemoteViz does not keep a reference to the RenderArea object after the
       *  object is disposed. If the application kept a reference to the object,
       *  this method can be used to query its state.
       *
       *  \if_cpp
       *  Note: RemoteViz objects are not reference counted like Open Inventor nodes,
       *  but std::shared_ptr provides a similar mechanism.
       *  \endif
       *
       *  The state "disposed" will be set just after triggering the listener ServiceListener#onDisposingRenderArea.
       *
       *  \return true if the renderArea is disposed or false if the renderArea is alive.
       *  [OIV-WRAPPER PROPERTY{IsDisposed},GETTER]
       */
      bool isDisposed() const;
      /**
       *  Gets the renderArea id.
       *
       *  \return the ID identifying the renderArea
       */
      const std::string& getId() const;
      /**
       *  Gets an existing Connection to the renderArea.
       *
       *  \param id : ID identifying the Connection
       *
       *  \return the connection object if the id exists, otherwise returns null.
       */
      std::shared_ptr<Connection> getConnection(const std::string& id) const;
      /**
       *  Gets an existing Connection to the renderArea
       *
       *  \param index : index identifying the Connection
       *
       *  \return the connection object if the index exists, otherwise returns null.
       */
      std::shared_ptr<Connection> getConnection(unsigned int index) const;
      /**
       *  Gets the number of connections (see Connection) that exist for this renderArea.
       *
       *  \return the number of connections
       */
      unsigned int getNumConnections() const;
      /**
       *  Gets the Open Inventor scene manager associated with the renderArea.
       *  This object is created automatically by RemoteViz.
       *
       *  In the ServiceSettings#INDEPENDENT_SERVICE mode, this method returns null.
       *
       *  \return the sceneManager
       *
       * [OIVJAVA-WRAPPER-RETURN-TYPE STRONG_REF]
       */
      SoSceneManager* getSceneManager() const;
      /**
       *  Gets the Open Inventor touch manager associated with the renderArea.
       *  This object is created automatically by RemoteViz.
       *
       *  In the ServiceSettings#INDEPENDENT_SERVICE mode, this method returns null.
       *
       *  \return the touchManager
       */
      SoTouchManager* getTouchManager() const;
      /**
       *  Gets the OpenGL context associated with the renderArea.
       *  This object is created automatically by RemoteViz.
       *
       *  In the ServiceSettings#INDEPENDENT_SERVICE mode, this method returns null.
       *
       *  \return the OpenGL context
       */
      SoGLContext* getGLContext() const;
      /**
       *  Sends a text message to the renderArea connections
       *
       *  \param message : the message to be sent to all connections of the renderArea
       *
       *  \param excludedConnections : an excluded connections array 
       *
       *  \return true if the message has been successfully sent to all connections, otherwise returns false.
       */
      bool sendMessage(const std::string& message, std::vector<std::shared_ptr<Connection>> excludedConnections = {}) const;
      /**
       *  Sends a binary message to the renderArea connections
       *
       *  \param buffer : the binary buffer to be sent to all connections of the renderArea
       *
       *  \param excludedConnections : an excluded connections array
       *
       *  \return true if the message has been successfully sent to all connections, otherwise returns false.
       */
      bool sendMessage(const std::vector<unsigned char>& buffer, std::vector<std::shared_ptr<Connection>> excludedConnections = {}) const;
      /**
       *  Gets the renderArea width.
       *
       *  \return the renderArea width
       */
      unsigned int getWidth() const;
      /**
       *  Gets the renderArea height.
       *
       *  \return the renderArea height
       */
      unsigned int getHeight() const;
      /**
       *  Resizes the renderArea at a specific size. After resizing, a notification will be sent to all the 
       *  renderArea clients.
       *  The minimum size is (128, 128) and the maximum size is (3840, 2160).
       *  If the size is inferior or superior to the limit, it will be adjusted automatically.
       *  @see RenderAreaListener#onResize
       *
       *  \param width : new width of the renderArea
       *
       *  \param height : new height of the renderArea
       *
       */
      void resize(unsigned int width, unsigned int height);
      /**
       *  Closes all connections of the renderArea and disposes it.
       *  If connections are still open, this method will close them and dispose the renderArea. While executing
       *  this method, the listeners ServiceListener#onDisposingRenderArea and ServiceListener#onDisposedRenderArea
       *  will be triggered and a DISPOSED disconnect message will be sent to all clients of the renderArea. 
       *  The listener RenderAreaListener#onClosedConnection will not be triggered after closing connections.
       *
       *  See #isDisposed().
       * 
       */
      void closeConnectionsAndDispose();
      /**
       *  Adds a RenderAreaListener to manage the renderArea.
       *
       *  \param listener : object that listens to the renderArea events
       *
       * [OIVJAVA-WRAPPER HELPER_BEGIN{onAddListener(listener)}]
       */
      void addListener(std::shared_ptr<RenderAreaListener> listener);
      /**
       *  Removes a RenderAreaListener
       *
       *  \param listener : object that listens to the renderArea events
       *
       * [OIVJAVA-WRAPPER HELPER_BEGIN{onRemoveListener(listener)}]
       */
      void removeListener(std::shared_ptr<RenderAreaListener> listener);
      /**
       *  Removes all RenderAreaListeners
       *
       * [OIVJAVA-WRAPPER HELPER_BEGIN{onRemoveAllListeners()}]
       */
      void removeAllListeners();
      /**
       *  Gets the number of RenderAreaListener
       *
       *  \return number of RenderAreaListener.
       */
      unsigned int getNumListeners() const;
      /**
       *  Gets the Graphics Processing Unit (GPU) used to render images.
       *  GPU can be defined in ServiceListener#onPendingCreateRenderArea.
       *  @see RenderAreaHardware#setGpu
       *
       *  \return gpu used to compute the render.
       */
      const std::string& getGpu() const;

      /**
       *  Sets the message which will be associate with the next rendered frames.
       *  This message can be recovered on a EncodedFrame.
       *  @see EncodedFrame#getMessage
       *  @see Connection#getLastEncodedFrame
      */
      void setFrameMessage( const std::string& message );

    protected:
      /*! \cond PRIVATE */
      std::shared_ptr<RenderAreaImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<RenderAreaImpl> pImpl;

      /** Constructor */
      RenderArea(const std::string& id, const RenderAreaHardware& renderAreaHardware);

      /** Deleted constructors */
      RenderArea(const RenderArea&) = delete;
      RenderArea& operator= (const RenderArea&) = delete;

      /** Create an instance of the class */
      static std::shared_ptr<RenderArea> createInstance(const std::string& id, unsigned int width, unsigned int height, const RenderAreaHardware& renderAreaHardware);
    };

  }
}

