/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>

#include <string>
#include <memory>

/**
 * @namespace RemoteViz::Rendering
 * \brief This namespace manages the server-side rendering mechanism.
 *  It is described in the following diagram:
 * \image html RemoteVizRenderingAPI.jpg
 */

namespace RemoteViz
{
  namespace Rendering
  {

    class RenderArea;

    class ServiceImpl;

    class Client;

    class ServiceSettings;

    class ServiceListener;

    class RenderAreaHardware;
    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Defines the RemoteViz rendering service.
     *
     *  This class uses the singleton pattern restricting its 
     *  instantiation to a single unique object in the program.  Use the Service::instance() method
     *  to get the Service object.
     *
     *  To use RemoteViz, the service must first be opened with the method Service::open().
     *  Create an instance of ServiceSettings to set properties such as the Service IP address and port.
     *
     *  Create an implementation of class ServiceListener to receive notifications from the Service.
     *  Register a listener using the addListener() method.
     *
     *  The Service object can also be used to query the current Clients and RenderAreas.
     *
     *  Thread-safety : @BR
     *  All methods in the RemoteViz API are thread-safe. I.e., they are guaranteed to be free of race conditions 
     *  when accessed by multiple threads simultaneously.  But note this limitation with the class Service:
     *  - The methods Service::open(), Service::close() and Service::dispatch() @I must @i be called from the same thread.
     *
     * @SEE_ALSO
     * ServiceSettings,
     * ServiceListener,
     * Client,
     * RenderArea
     *
     * [OIVJAVA-WRAPPER-CLASS BASIC_TYPE{false}]
     */
    class RENDERSERVICE_API Service
    {
   
      /*! \cond PRIVATE */
      friend class RenderAreaImpl;
      friend class ConnectionImpl;
      friend class ClientImpl;
      friend class RenderAreaDeleter;
      friend class MonitoringImpl;
      friend class ServiceDeleter;
      /*! \endcond */

    public:

      /**
       *  Renderering service instance (Singleton)
       *
       *  \return the service instance
       * [OIVNET-WRAPPER PROPERTY{Instance},GETTER]
       */
      static Service* instance();
      /**
       *  This method has to be used with the ServiceSettings#INVENTOR_SERVICE and ServiceSettings#INDEPENDENT_SERVICE mode in a loop.
       *  It processes all waiting RemoteViz events.
       */
      void dispatch() const;
      /**
       *  Opens the rendering service.
       *  
       *  \param settings : the settings of the instance. This object cannot be null.
       *
       * @SEE_ALSO
       *    Service#getSettings
       *
       *  \return true if successful, otherwise returns false.
       */
      bool open(std::shared_ptr<const ServiceSettings> settings);
      /**
       *  Closes the rendering service. 
       *  If renderAreas still exist, this method will dispose them and close the rendering service.
       *
       *  \return true if successful, otherwise returns false.
       */
      bool close();
      /**
       *  This method is used to know if the rendering service is running.
       *
       *  \return true if running; otherwise returns false.
       */
      bool isRunning() const;
      /**
       *  Forces the creation of a renderArea. \n This method is optional. 
       *  The other way to create a renderArea is to return true to the callback 
       *  ServiceListener#onPendingCreateRenderArea when a connection requests it.\n 
       *  If the renderArea already exists, the call will be ignored.
       *
       *  \param id : ID identifying the RenderArea
       *
       *  \param width : renderArea width
       *
       *  \param height : renderArea height
       *
       *  \param renderAreaHardware : renderArea hardware settings.
       */
      void createRenderArea(const std::string& id, unsigned int width, unsigned int height, std::shared_ptr<const RenderAreaHardware> renderAreaHardware);
      /**
       *  Gets a RenderArea from id. Range is [0..getNumRenderAreas - 1].
       *
       *  \param id : ID identifying the RenderArea
       *
       *  \return the renderArea object if the id exists, otherwise returns null.
       */
      std::shared_ptr<RenderArea> getRenderArea(const std::string& id) const;
      /**
       *  Gets a RenderArea from index. Range is [0..getNumRenderAreas - 1].
       *
       *  \param index : index identifying the renderArea
       *
       *  \return the renderArea object if the index exists, otherwise returns null.
       */
      std::shared_ptr<RenderArea> getRenderArea(unsigned int index) const;
      /**
       *  Gets the number of renderAreas
       *
       *  \return the renderArea number of the Instance.
       */
      unsigned int getNumRenderAreas() const;
      /**
       *  Gets the settings of the service
       *
       *  \return the service settings object
       */
      std::shared_ptr<const ServiceSettings> getSettings() const;
      /**
       *  Adds a ServiceListener
       *
       *  \param listener : object that listens to the service events
       */
      void addListener(std::shared_ptr<ServiceListener> listener);
      /**
       *  Removes a ServiceListener
       *
       *  \param listener : object that listens to the service events
       */
      void removeListener(std::shared_ptr<ServiceListener> listener);
      /**
       *  Removes all ServiceListeners
       */
      void removeAllListeners();
      /**
       *  Gets the number of ServiceListener
       *
       *  \return number of ServiceListener.
       */
      unsigned int getNumListeners() const;
      /**
       *  Gets a Client from id. Range is [0..getNumClients - 1].
       *
       *  \param id : ID identifying the client
       *
       *  \return the client object if the id exists, otherwise returns null.
       */
      std::shared_ptr<Client> getClient(const std::string& id) const;
      /**
       *  Gets a Client from index. Range is [0..getNumClients - 1].
       *
       *  \param index : index identifying the client
       *
       *  \return the client object if the index exists, otherwise returns null.
       */
      std::shared_ptr<Client> getClient(unsigned int index) const;
      /**
       *  Gets the number of clients
       *
       *  \return the number clients attached to the service.
       */
      unsigned int getNumClients() const;
 
    protected:
      /*! \cond PRIVATE */
      std::shared_ptr<ServiceImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<ServiceImpl> pImpl;

      /** Constructor */
      Service();

      /** Deleted constructors */
      Service(const Service&) = delete;
      Service& operator= (const Service&) = delete;

    };
  }
}
