/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/sys/port.h>
#include <RemoteViz/Rendering/LibHelper.h>
#include <SoDeprecationRules.h>

#include <memory>
#include <string>

namespace RemoteViz
{
  namespace Rendering
  {

    class ServiceSettingsImpl;
    class Monitoring;

    /**
     * @RVEXT
     *
     * @ingroup RemoteViz
     *
     * @DESCRIPTION
     *  Settings that define the rendering Service.
     *  A ServiceSettings object must be passed to the Service#open() method.
     *
     * [OIVJAVA-WRAPPER-CLASS SHARED_POINTER_USE]
     * [OIVNET-WRAPPER-CLASS AUTO_PROPERTY,SHARED_POINTER_USE]
     */
    class RENDERSERVICE_API ServiceSettings
    {

      /*! \cond PRIVATE */
      friend class ServiceImpl;
      /*! \endcond */

    public:
      /**
       *  Constructor
       */
      ServiceSettings();

      /**
       *  Copy constructor
       */
      ServiceSettings(const ServiceSettings& obj);

      /**
       *  Copy assignment operator
       */
      ServiceSettings& operator=(const ServiceSettings& obj);

      /**
       *  Destructor
       */
      ~ServiceSettings();

      //@{
      /**
       *  Sets the IP address.\n
       *  IP version 4 (IPv4: X.X.X.X) and IP version 6 (IPv6: X::X::X::X::X::X::X::X) are supported. 
       *  Default value is "127.0.0.1"\n
       *  To make the service accept connections on all interfaces, use "0.0.0.0".
       *
       *  \param ip : the IP address used by the instance
       */
      void setIP(const std::string& ip);
      /**
       *  Gets the IP address. Default value is "127.0.0.1".
       *
       *  \return the IP address used by the instance
       */
      const std::string& getIP() const;
      //@}

      //@{
      /**
       *  Sets the port of the service. Default value is 8080.\n
       *  In case you are behind a firewall, please ensure your IT department configured the firewall to accommodate
       *  the selected port.
       *  [OIVJAVA-WRAPPER-ARG WRAP_AS{int}]
       *  \param port : the port used by the instance
       */
      void setPort(unsigned short port);
      /**
       *  Gets the port of the service. Default value is 8080.\n
       *  [OIVJAVA-WRAPPER-RETURN-TYPE WRAP_AS{int}]
       *  \return the port used by the instance
       */
      unsigned short getPort() const;
      //@}

      //@{
      /**
       *  The run mode is used to change the behaviour of the RemoteViz execution in three environments.\n 
       *  INVENTOR_SERVICE mode is used to run a RemoteViz application based on the Open Inventor render engine. This is the default mode. 
       *  This mode requires a main loop.\n
       *  INVENTOR_APPLICATION is used to share the rendering of a scene graph between an OIV application and a RemoteViz application.
       *  This mode uses the main loop of an Open Inventor component (SoWinExaminerViewer, SoXtRenderArea, ...).\n
       *  INDEPENDENT_SERVICE mode is used to run a RemoteViz application based on its own render engine. In this mode, the Open Inventor render 
       *  engine is disabled. This mode requires a main loop.\n
       */
      enum RunMode
      {
        INVENTOR_SERVICE,
        INVENTOR_APPLICATION,
        INDEPENDENT_SERVICE
      };
      /**
       *  Sets the run mode of the service. Default value is INVENTOR_SERVICE.\n
       *
       *  \param runmode : the type of run mode used by the instance
       */
      void setRunMode(ServiceSettings::RunMode runmode);
      /**
       *  Gets the run mode of the service. Default value is INVENTOR_SERVICE.\n
       *
       *  \return the run mode used by the instance
       */
      ServiceSettings::RunMode getRunMode() const;
      //@}

      //@{
      /**
       *  Gets the current host name.\n
       *
       *  \return the current host name
       */
      std::string getHostname() const;
      //@}

      //@{
      /**
       *  Each enumeration represents an Open Inventor extension. Use setUsedExtensions to activate them.\n
       */
      enum Extensions 
      {
        VOLUMEVIZ = 0x1, // 2^0, bit 0
        VOLUMEVIZLDM = 0x2, // 2^1, bit 1
        MESHVIZ = 0x4, // 2^2, bit 2
        MESHVIZXLM = 0x8, // 2^3, bit 3
        HARDCOPY = 0x20, // 2^5, bit 5
        CATIA5READER = 0x40, // 2^6, bit 6
        CATIA6READER = 0x80, // 2^7, bit 7
        DWGREADER = 0x100, // 2^8, bit 8
        IGESREADER = 0x200, // 2^9, bit 9
        JTREADER = 0x400, // 2^10, bit 10
        PROEREADER = 0x800, // 2^11, bit 11
        SOLIDEDGEREADER = 0x1000, // 2^12, bit 12
        STEPREADER = 0x2000, // 2^13, bit 13
        SOLIDWORKSREADER = 0x4000, // 2^14, bit 14
        UGREADER = 0x8000, // 2^15, bit 15
        VDAREADER = 0x10000, // 2^16, bit 16
        XMTREADER = 0x20000 // 2^17, bit 17
      };
      /**
       *  Sets the Open Inventor extensions used by a RemoteViz application.
       *  Convenient method that will be used by Service::open() to check if all the licenses
       *  necessary to start the Service are available (necessary licenses are explained below)
       *  \if_cpp and to call necessary Open Inventor init() methods \endif\.
       *
       *  RemoteViz and Open Inventor (if used) are protected by a license key mechanism that
       *  limits its use to specified computers or network environments based on a commercial
       *  agreement. Depending on the license agreement, license keys are floating licenses
       *  managed by a license server or an encrypted "master password" string (see SoLockManager).
       *
       *  One RemoteViz license is required to start the service. (An SDK license for 
       *  development/debugging or a Runtime license for deployment.)
       *
       *  If the Service uses Open Inventor for rendering (runMode is not INDEPENDENT_SERVICE),
       *  then one Open Inventor license plus one license of each used extension is required to
       *  start the service. (An SDK license for development/debugging or a Runtime license for
       *  deployment.) If this method is called and licenses are missing, Service::open()
       *  will return false. If this method is not called, the licenses will not be checked
       *  until the first call to create an Open Inventor node.
       *  Alternatively the application can do explicit license checks by calling
       *  SoLicensesInfo::check() before opening the Service or creating any of the Open Inventor nodes.
       *  Note: Enabling the VolumeVizLDM extension automatically enables the VolumeViz extension.
       *
       *
       *  \if_cpp
       *  In C++, this method also calls the init() methods for Open Inventor and all specified
       *  extensions (e.g. SoDB::init()). The corresponding finish() methods will be automatically
       *  called when the Service is stopped. Alternatively, the application can explicitly call
       *  init methods before creating any Open Inventor nodes. In that case, the application should
       *  also call the corresponding finish() methods when the Service is stopped.
       *  \endif
       *
       *  When using the run mode INDEPENDENT_SERVICE, this method has no effect, i.e., only one RemoteViz
       *  license will be required to start the service.
       *
       *  \param extensions : the bitmask of the Open Inventor extensions used in RemoteViz. 
       *  Many extensions can be activated by using the logical OR operator. \n 
       *  For example : setUsedExtensions(VOLUMEVIZ | MESHVIZ)
       * [OIVJAVA-WRAPPER NAME{setOivUsedExtensions},HELPER_BEGIN{checkExtensions(extensions)}]
       * [OIV-WRAPPER-ARG WRAP_AS{#Extensions}&ENUM_SET]
       */
      void setUsedExtensions(unsigned int extensions);
      /**
       *  Gets the used Open Inventor extensions.\n
       *
       *  \return the bitmask of used Open Inventor extensions.
       * [OIV-WRAPPER-RETURN-TYPE WRAP_AS{#Extensions},ENUM_SET]
       * [OIVJAVA-WRAPPER NAME{getOivUsedExtensions}]
       */
      unsigned int getUsedExtensions() const;
      //@}

      //@{
      /**
       *  Gets the monitoring object which allows to manage metrics listeners that monitor states and performances of the service.\n
       *
       *  \returns the monitoring object.
       *
       */
      std::shared_ptr<Monitoring> getMonitoring() const;
      //@}

      //@{
      /**
       *  Secures the connection between the server and the clients.
       *  RemoteViz uses the WebSocket protocol to enable communication between the client and the service. 
       *  In a production environment, it is highly recommended to use secure WebSocket. 
       *  Using a secure WebSocket connection improves confidentiality and reliability because the connection is encrypted with Transport Layer Security (TLS), 
       *  which reduces the risk of a variety of attacks such as man-in-the-middle tampering with your data.
       *
       *  The URL scheme used by the clients to connect to the service must be wss://
       *
       *  \param publicCertificateFilePath : Path to a file containing the public certificate used to authenticate the server to the clients.
       *  The certificate has to be a PEM ("Privacy Enhanced Mail") encoded certificate.
       *
       *  \param privateKeyFilePath : Path to a file containing the private key used to sign the server key exchange between the client and the server.
       *  If the private key is protected by a passphrase, set the passphrase into the ServiceListener#onRequestedPrivateKeyPassphrase callback.
       *
       *  \param enabledSecurityProtocols : Defines the security protocols used to secure the exchange between the client and the server.
       *  By default, TLSv1.1, TLSv1.2 and TLSv1.3 protocols are enabled. 
       *  A bitmask of the security protocols is used. Many security protocols can be enabled by using the logical OR operator. \n 
       * [OIV-WRAPPER-ARG IN,IN,WRAP_AS{#SecurityProtocols}&ENUM_SET]
       */
      void enableSecureConnection(const std::string& publicCertificateFilePath, const std::string& privateKeyFilePath, unsigned int enabledSecurityProtocols = TLSv1_1 | TLSv1_2 | TLSv1_3);
      /**
       *  Gets the SSL engine activation.
       *
       *  \return true if the secure connection is enabled.
       */
      bool isSecureConnection() const;
      //@}

      /**
       *  Gets the public certificate authenticating the server to the clients.
       *
       *  \return the path to the file containing the certificate.
       */
      const std::string& getPublicCertificateFile() const;

      /**
       *  Gets the private key used to sign the server key exchange between the client and the server.
       *
       *  \return the path to the file containing the private key.
       */
      const std::string& getPrivateKeyFile() const;

      //@{
      /**
       *  Each enumeration represents a security protocol. Use enableSecureConnection to enable them.\n
       */
      enum SecurityProtocols 
      {
        SSLv2 = 0x1, // 2^0, bit 0
        SSLv3 = 0x2, // 2^1, bit 1
        TLSv1 = 0x4, // 2^2, bit 2
        TLSv1_1 = 0x8, // 2^3, bit 3
        TLSv1_2 = 0x10, // 2^4, bit 4
        TLSv1_3 = 0x20 // 2^5, bit 5
      };
      /**
       *  Gets the security procotols used in RemoteViz.\n
       *
       *  \return the bitmask of used security protocols.
       * [OIV-WRAPPER-RETURN-TYPE WRAP_AS{#SecurityProtocols}]
       */
      unsigned int getEnabledSecurityProtocols() const;
      //@}

    protected:
      /*! \cond PRIVATE */
      std::shared_ptr<ServiceSettingsImpl> getImpl() const;
      /*! \endcond */

    private:
      /** Pointer to implementation */
      std::shared_ptr<ServiceSettingsImpl> pImpl;

    };

  }
}
