/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Dec 2000)
** Modified by : T.DUFOUR (MMM yyyy)
**=======================================================================*/

#ifndef _SO_VOLUME_DETAIL_
#define _SO_VOLUME_DETAIL_

#include <Inventor/details/SoSubDetail.h>
#include <Inventor/SbPImpl.h>
#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>
#include <LDM/elements/SoDataRangeElement.h>

class SoVolumeData;
class SoROI;
class SbLine;
class SoTransferFunction;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoVolumeDetail)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoVolumeDetail
//
//  Abstract class for SoVolumeRenderDetail and SoVolumeSkinDetail nodes.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * @VREXT Base class for detail information about picking in a data volume.
 *
 * @ingroup VolumeVizDetails
 *
 * @DESCRIPTION
 *   A successful pick operation returns an SoPickedPoint object. If the picked
 *   geometry is an SoVolumeSkin or SoVolumeRender, use the @B getDetail @b method
 *   and cast the result to the appropriate class (SoVolumeSkinDetail or
 *   SoVolumeRenderDetail) to get extra information about the location of the pick.
 *
 *   SoVolumeDetail returns information about the first (closest)
 *   non-transparent voxel along the pick ray and/or information about all the voxels
 *   intersected by the pick ray (called a profile). For the first non-transparent
 *   voxel (if any) on the pick ray, it returns XYZ position, IJK location, and data
 *   value. For the profile defined by all the intersected voxels, it returns:
 *
 *   - Profile position in object space (two XYZ points)
 *
 *   - Profile location in the data volume (two IJK indices)
 *
 *   - Number of values along the profile
 *
 *   - Each voxel value and corresponding IJK location
 *
 *   This class returns information about the data values in the picked voxels.
 *   SoVolumeRenderDetail also returns information about the mapped RGBA colors
 *   of the picked voxels.
 *
 * @B GPU picking vs. CPU picking @b
 * \par
 *   VolumeViz can compute the picked voxel(s) and voxel colors on the GPU. @BR
 *   GPU picking is generally faster than CPU picking, but more importantly it is
 *   the only way to be sure of getting the correct result if the application is using
 *   a custom shader (see SoVolumeShader and VVizComputeFragmentColor) to compute voxel
 *   color and opacity. CPU picking always uses the color and opacity values assigned
 *   by the SoTransferFunction node.  Therefore the result should be the same when using
 *   SoTransferFunction to assign color and opacity, but when using a custom shader the
 *   CPU picking algorithm is not able to know the color and opacity assigned by the shader.
 * \par
 *   @B Important@b: To enable GPU picking
 *   - The @I realValue@i parameter must be false when calling the getXXX methods. @BR
 *   - The SoRayPickAction must be aware of the SoSceneManager. @BR
 *     SoHandleEventAction does this automatically, so it is not necessary for the application
 *     to take any action when using an SoEventCallback node and calling the
 *     getPickedPoint() method.  However if the application creates its own instance of SoRayPickAction,
 *     then it must call the method SoAction::setSceneManager() on that instance.  If the scene manager
 *     is not specified, a warning message is issued and CPU picking is done.
 * \par
 *   Also be aware of the following trade-off.  If the realValue parameter is set to false,
 *   the voxel value returned may be based on lower resolution (sub-sampled) data.  The
 *   values should be approximately the same, but to be sure of getting values from the
 *   full resolution data, set the realValue parameter to true.
 *
 * @B Limitations:@b
 *   - If multiple volumes are being combined under an SoMultiDataSeparator,
 *     the detail class only returns values for the first volume in the scene graph.
 *   - Because picking returns actual values of voxel, there may be a small difference
 *     between what you see on screen and what you get with picking. When visualizing
 *     a volume using LINEAR or CUBIC interpolation, you visualize interpolated data.
 *     Picking returns informations such as voxel value, voxel postion, etc... so picking
 *     return information of voxel values, not interpolated one. Visible areas may be not pickable
 *     and unvisibke areas may be pickable.
 *     When using NEAREST interpolation for rendering, picking matches what is displayed.
 *
 * @SEE_ALSO
 *    SoDetail,
 *    SoFenceSliceDetail,
 *    SoVolumeRenderDetail,
 *    SoVolumeSkinDetail,
 *    SoVolumeRender,
 *    SoVolumeSkin
 *
 *
 */
class VOLUMEVIZ_API SoVolumeDetail : public SoDetail {

  SO_DETAIL_HEADER(SoVolumeDetail);
  SO_PIMPL_BASE_PUBLIC_HEADER(SoVolumeDetail);

 public:
  /**
   * Constructor.
   */
  SoVolumeDetail();
  /**
   * Destructor.
   */
  virtual ~SoVolumeDetail();

  /**
   * Returns the profile position in object space (world coordinates).
   *
   * Specifically the @I profile@i parameter contains the XYZ coordinates of the
   * first and last points on the profile.
   */
  void getProfileObjectPos(SbVec3f profile[2]) const;

  /**
   * Returns the number of values along the profile and the profile position in data
   * space (voxel coordinates).
   *
   * Specifically the @I profile@i parameter contains the IJK coordinates of the
   * first and last points on the profile.  The number of values is the number of
   * of voxels intersected by the pick ray.
   * [OIVJAVA-WRAPPER-RETURN-TYPE NO_WRAP]
   * [OIVJAVA-WRAPPER DUPLICATE{getProfileNumValues}]
   * [OIVJAVA-WRAPPER-ARG APPLY_TO{getProfileNumValues} NO_WRAP]
   */
  int  getProfileDataPos(SbVec3i32 profile[2]) const;

  /**
   * Returns the @B index@b'th value (as an integer) and its position in object and data space.
   *
   * The number of values along the profile can be queried using the 
   * #getProfileDataPos() method.  However @B index @b is automatically clamped
   * to the valid range of values.
   *
   * If @B realValue @b is TRUE, VolumeViz will return the actual voxel value from 
   * the full resolution data. Note that this may require a read from disk if full
   * resolution data is not currently in memory. If FALSE, VolumeViz will return the
   * voxel value from the (possibly subsampled) data currently in memory.
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   * [OIV-WRAPPER-ARG IN,OUT,OUT,IN]
   * [OIVJAVA-WRAPPER PACK{ProfileValue}]
   * [OIVJAVA-WRAPPER-RETURN-TYPE NAME{voxelValue}]
   */
  int64_t getProfileValue(int index, SbVec3i32& pos, SbVec3f* objPos = 0, SbBool realValue = FALSE) const;

  /**
   * Returns the @B index@b'th value (as a double) and its position in object and data space.
   *
   * The number of values along the profile can be queried using the 
   * #getProfileDataPos() method.  However @B index @b is automatically clamped
   * to the valid range of values.
   *
   * If @B realValue @b is TRUE, VolumeViz will return the actual voxel value from 
   * the full resolution data. Note that this may require a read from disk if full
   * resolution data is not currently in memory. If FALSE, VolumeViz will return the
   * voxel value from the (possibly subsampled) data currently in memory.
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   * [OIV-WRAPPER-ARG IN,OUT,OUT,IN]
   * [OIVJAVA-WRAPPER PACK{ProfileValueD}]
   * [OIVJAVA-WRAPPER-RETURN-TYPE NAME{voxelValue}]
   */
  double getProfileValueD(int index, SbVec3i32& pos, SbVec3f* objPos, SbBool realValue = FALSE) const;

  /**
   * Retrieves the value and position of the first non-transparent voxel along the pick ray (if any).
   * Depending on the argument realValue, the voxel value is either fetched from the real value in full
   * resolution data or fetched from the displayed data tile on the GPU which is not necessarily in full
   * resolution. Fetching the data in full resolution may require a read from the disk if the full-resolution
   * data is not currently in memory.
   *
   * @param [out] value the integer value of the first non-transparent voxel along the pick ray.
   * @param [out] pos the position of the picked voxel in data space (IJK voxel coordinates)
   * @param [out] objPos the position of the picked voxel in object space (XYZ world coordinates).
   * @param [in] realValue if TRUE, the integer value is fetched from the full resolution data. If FALSE, it returns the voxel value from the (possibly subsampled) data currently in GPU memory, thus the voxel value that is displayed in the render area. The displayed voxel is not necessarily in full resolution.
   * @return FALSE if all voxels along the ray pick are fully transparent, thus all out parameters are meaningless
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   * [OIV-WRAPPER-ARG OUT,OUT,OUT,IN]
   * [OIVJAVA-WRAPPER PACK{TransparentLongValue}]
   */
  virtual SbBool getFirstNonTransparentValue(int64_t& value, SbVec3i32& pos, SbVec3f* objPos = 0, SbBool realValue = FALSE) const;

  /**
   * Retrieves the value and position of the first non-transparent voxel along the pick ray (if any).
   * Depending on the argument realValue, the voxel value is either fetched from the real value in full
   * resolution data or fetched from the displayed data tile on the GPU which is not necessarily in full
   * resolution. Fetching the data in full resolution may require a read from the disk if the full-resolution
   * data is not currently in memory.
   *
   * @param [out] value the double value of the first non-transparent voxel along the pick ray.
   * @param [out] pos the position of the picked voxel in data space (IJK voxel coordinates)
   * @param [out] objPos the position of the picked voxel in object space (XYZ world coordinates).
   * @param [in] realValue if TRUE, the integer value is fetched from the full resolution data. If FALSE, it returns the voxel value from the (possibly subsampled) data currently in GPU memory, thus the voxel value that is displayed in the render area. The displayed voxel is not necessarily in full resolution.
   * @return FALSE if all voxels along the ray pick are fully transparent, thus all out parameters are meaningless
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   * [OIV-WRAPPER-ARG OUT,OUT,OUT,IN]
   * [OIVJAVA-WRAPPER NAME{getFirstNonTransparentDoubleValue},PACK{TransparentDoubleValue}]
   */
  virtual SbBool getFirstNonTransparentValue(double& value, SbVec3i32& pos, SbVec3f* objPos = 0, SbBool realValue = FALSE) const;

  /** @copydoc SoDetail::copy() */
  virtual SoDetail* copy() const;

 SoEXTENDER public:
  void setDetails(SbLine line, SoVolumeData* vd, SoTransferFunction* tf, SoROI* r);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /** Set pair of dataset/id for this detail */
  void setDataSetIdPairList(const SoLDM::DataSetIdPairList& dsPairList);

  /** Set pair of dataset/id for this detail */
  void setDrawStyleMap(const SoVolumeDataDrawStyle::MaskDrawStyleValueMap& styleMap);

  /** Set the transfer function list and the id of the tf used for intersections*/
  void setTransferFunctionList(const std::vector<SoTransferFunction*>& tfList, int intersectionTfId);

  /** Set transfer function list */
  void setIsovalueMap(const SoVolumeDataDrawStyle::IsovalueMap& isomap);

  /** Set transfer function list */
  void setDataRangeMap(const SoDataRangeElement::DataRangeIdMap& dataRangeMap);

  /** Transparency of the volume data */
  void setVolumeRenderTransparency(float transparency);

 protected:
  /** Copy constructor */
  SoVolumeDetail(const SoVolumeDetail& detail);
};

#endif /* _SO_VOLUME_DETAIL_ */


