/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T.DUFOUR (Jun 2002)
**=======================================================================*/

#ifndef _SO_VOLUME_RENDER_DETAIL_
#define _SO_VOLUME_RENDER_DETAIL_

#include <Inventor/details/SoSubDetail.h>
#include <VolumeViz/details/SoVolumeDetail.h>

SO_PIMPL_PUBLIC_DECLARATION(SoVolumeRenderDetail)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoVolumeRenderDetail
//
//  Detail about an SoVolumeRender node.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * @VREXT Stores detail information about a picked voxel or pick ray in a data volume.
 *
 * @ingroup VolumeVizDetails
 *
 * @DESCRIPTION
 *   A successful pick operation returns an SoPickedPoint object. If the picked
 *   geometry is an SoVolumeRender node, use the @B getDetail @b method and cast
 *   the result to this class to get extra information about the the pick.
 *
 *   SoVolumeRenderDetail returns information about the first (closest)
 *   non-transparent voxel along the pick ray and/or information about all the voxels
 *   intersected by the pick ray (called a profile). For the first non-transparent
 *   voxel (if any) on the pick ray, it returns XYZ position, IJK location, and data
 *   value. For the profile defined by all the intersected voxels, it returns:
 *
 *   - Profile position in object space (two XYZ points)
 *
 *   - Profile location in the data volume (two IJK indices)
 *
 *   - Number of values along the profile
 *
 *   - Each voxel value and corresponding IJK location
 *
 *   Use the methods in SoVolumeDetail to get information about the data values
 *   of the picked voxels.  Use the getRgba... methods in this class to get information
 *   about the mapped RGBA colors of the picked voxels.  Because these methods
 *   return colors, they always return information based on the (possibly
 *   subsampled) tiles currently in memory.
 *
 *   GPU picking: Please see the discussion of GPU picking in SoVolumeDetail.
 *   
 * @B Limitations:@b
 *   - If multiple volumes are being combined under an SoMultiDataSeparator,
 *     the detail class only returns values for the first volume in the scene graph.
 *
 * @SEE_ALSO
 *    SoDetail,
 *    SoVolumeDetail,
 *    SoVolumeRender
 *
 *
 */
class VOLUMEVIZ_API SoVolumeRenderDetail : public SoVolumeDetail {

  SO_DETAIL_HEADER(SoVolumeRenderDetail);
  SO_PIMPL_PUBLIC_HEADER(SoVolumeRenderDetail);

 public:
  /**
   * Constructor.
   */
  SoVolumeRenderDetail();
  /**
   * Destructor.
   */
  virtual ~SoVolumeRenderDetail();

  /**
   * Returns the number of volume masks applied to the first non-transparent voxel.
   *
   * If @B realValue @b is TRUE, VolumeViz will return the actual value from
   * the full resolution data. Note that this may require a read from disk if full
   * resolution data is not currently in memory. If FALSE, VolumeViz will return the
   * value from the (possibly subsampled) data currently in memory.
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   */
  size_t getMaskCount(SbBool realValue = FALSE) const;

  /**
   * Returns the i-th volume mask applied to the first non-transparent voxel.
   *
   * @B index @b must be in the valid range returned by #getMaskCount().
   *
   * If @B realValue @b is TRUE, VolumeViz will return the actual value from
   * the full resolution data. Note that this may require a read from disk if full
   * resolution data is not currently in memory. If FALSE, VolumeViz will return the
   * value from the (possibly subsampled) data currently in memory.
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   */
  SoLDM::DataSetIdPair getMask(size_t index, SbBool realValue = FALSE)  const;

  /**
   * Returns the style applied to the first non-transparent voxel (and its isovalue
   * if the style is SoVolumeDataDrawStyle::ISOSURFACE). @BR
   *
   * If @B realValue @b is TRUE, VolumeViz will return the actual value from
   * the full resolution data. Note that this may require a read from disk if full
   * resolution data is not currently in memory. If FALSE, VolumeViz will return the
   * value from the (possibly subsampled) data currently in memory.
   *
   * @warning To use GPU picking, you must specify realValue = FALSE.
   */
  void getVoxelStyleAndIsoValue(SoVolumeDataDrawStyle::DrawStyle& style, double &iso, SbBool realValue = FALSE)  const;

  /**
  * Returns the RGBA value, object space position, data space position
  * and tileId for each voxel intersected by the pick ray.
  * Voxels with a alpha value less or equal to opacityThreshold are ignored.
  * Values returned are based on the (possibly subsampled) tiles currently in memory.
  */
  void getRgbaValues(std::vector<SbVec4ub>& rgbaValues, std::vector<SbVec3f>& objectPos,
                     std::vector<SbVec3i32>& dataPos, std::vector<SoLDMTileID>& tileIds,
                     float opacityThreshold = -1.f) const;

  /**
  * Returns the RGBA value and object space position of each voxel intersected by the pick ray.
  * Plus their object space position.
  * Voxels with a alpha value less or equal to opacityThreshold are ignored.
  * Values returned are based on the (possibly subsampled) tiles currently in memory.
  */
  void getRgbaValues(std::vector<SbVec4ub>& rgbaValues, std::vector<SbVec3f>& objectPos, float opacityThreshold = -1.f) const;

  /**
  * Returns the RGBA value of each voxel intersected by the pick ray.
  * Voxels with a alpha value less or equal to opacityThreshold are ignored.
  * Values returned are based on the (possibly subsampled) tiles currently in memory.
  */
  void getRgbaValues(std::vector<SbVec4ub>& rgbaValues, float opacityThreshold = -1.f) const;

  /**
  * Returns the RGBA value and data space position of each voxel intersected by the pick ray.
  * Voxels with a alpha value less or equal to opacityThreshold are ignored.
  * Values returned are based on the (possibly subsampled) tiles currently in memory.
  */
  void getRgbaValues(std::vector<SbVec4ub>& rgbaValues, std::vector<SbVec3i32>& dataPos, float opacityThreshold = -1.f) const;

  /**
  * Returns the tileID of each voxel intersected by the pick ray.
  * Values returned are based on the (possibly subsampled) tiles currently in memory.
  */
  const std::vector<SoLDMTileID>& getRgbaValuesTileIds() const;

  /** @copydoc SoVolumeDetail::copy() */
  virtual SoDetail* copy() const;

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /** Set rgba values crossing the ray.
   * @param rgbaValues rgba values
   * @param points Corresponding positions in volumeData XYZ space.
   * rgbaValues must have the same size that points. */
  void setRgbaValues(const std::vector<SbVec4ub>& rgbaValues, const std::vector<SbVec3f>& points);

private:
  /** Copy constructor */
  SoVolumeRenderDetail(const SoVolumeRenderDetail& detail);
};

#endif /* _SO_VOLUME_RENDER_DETAIL_ */


