#ifndef  _SO_FENCE_SLICE_
#define  _SO_FENCE_SLICE_

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/fields/SoMFVec2f.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/SbBox.h>
#include <Inventor/STL/vector>
#include <Inventor/STL/pair>

#include <LDM/SoLDMTileID.h>


#include <VolumeViz/nodes/SoSlice.h>

class SoCache ;
class SoLDMNodeFrontManager;
class SoVolumeData;
class SoVolumeStateFenceSlice;
class SoVolumeFaceSet;
class SoVertexProperty;

/** 
  * @VREXT Fence slice shape node.
  *
  * @ingroup VolumeVizNodes
  *
  * @DESCRIPTION
  * This node renders a strip (connected series) of slices.  The strip is defined
  * by a set of 2D vertices that form a polyline.  Each segment of the polyline
  * is extruded along the axis specified in the #axis field to form (in effect)
  * an oblique slice.
  * The default axis is Z, so the 2D points are treated as (X,Y) values. 
  * The points may be outside the 3D extent of the volume, but only the portion 
  * of the slice inside the volume will be drawn (subject to region of interest 
  * and other clipping nodes).
  *
  *   The SoVolumeData node on which this shape is applied can be specified with #dataSetId.
  *   When this field is set to 0, the last SoVolumeData node on state is used.
  *
  * A similar
  * effect could be obtained using volume geometry (e.g. SoVolumeFaceSet), but
  * SoFenceSlice is more convenient and is optimized for this specific case.
  *
  * The 2D coordinates are interpreted according to the following table.
  * See the code example below.
  * \par
  *   @TABLE_1B
  *     @TR Fence axis	@TD Coordinate axes
  *     @TR X	@TD Y , Z
  *     @TR Y	@TD Z , X
  *     @TR Z	@TD X , Y
  *   @TABLE_END
  *
  * For a non-RGBA (scalar valued) volume,
  * each voxel's RGBA value is determined by the current SoDataRange and SoTransferFunction.
  * The current @B diffuse color @b and @B transparency @b (set, for example, with
  * an SoMaterial node) modify the appearance of the slice.  This means that, for
  * example, the current transparency can be used as a @I global alpha @i value
  * to modulate the overall opacity of the slice.
  * For an RGBA volume each voxel's RGBA value comes directly from the volume data.
  *
  * The #interpolation field controls how the texture is interpolated.
  *
  * The #alphaUse field (SoSlice) controls how the voxel's alpha component is used when drawing the fence slice.
  *
  * Optionally a bump mapping effect may be applied. Normal vectors are automatically
  * computed from the data value gradient.  The #enableBumpMapping and #bumpScale
  * fields (SoSlice) control whether bump mapping is active and the intensity of the effect.
  *
 *   @B Notes: @b 
 *   - Transformation matrices: @BR
 *     The volume size and orientation (like geometry) can be modified by
 *     transformation nodes in the scene graph and this in turn modifies the
 *     appearance of volume visualization nodes.
 *     However the same transformation must be applied to the volume data node
 *     and all volume visualization nodes associated with that volume.  So effectively
 *     any transformation nodes that affect the volume must be placed @B before @b
 *     the volume data node.
 *
 *   - Picking: @BR
 *     The entire slice is pickable, even where it is transparent as a result of the
 *     current transfer function.  Currently SoFenceSlice does not provide an SoDetail
 *     object.
 *
 *   - Interpolation: @BR
 *     Interpolation is specified using the #interpolation field. The default (LINEAR)
 *     does bi-linear interpolation between voxel values. The NEAREST value can be used
 *     to display individual voxels. For best image quality we recommend using the
 *     MULTISAMPLE_12 value.
 *
 *   - Data range: @BR
 *     By default VolumeViz maps the entire range of the voxel's data type
 *     (e.g. 0..65535 for unsigned short) into the colormap.  This is often
 *     correct for byte (8 bit) voxels, but seldom correct for 16 bit voxels
 *     and never correct for floating point voxels. Use an SoDataRange node
 *     to specify the actual (or desired) range of data values to be mapped.
 *     Also use an SoDataRange node to implement brightness/contrast control
 *     like the Window/Level setting commonly used with medical images.
 *
 *   - Clipping: @BR
 *     Volume primitives can be clipped using a region of interest (SoROI), geometry
 *     (SoVolumeClippingGroup) and/or height fields (SoUniformGridClipping).
 *     They are also clipped by OpenGL clipping planes (SoClipPlane), but we recommend
 *     using the VolumeViz clipping nodes instead.
 *
 *   - Material: @BR
 *     The color of each voxel is modulated by the current diffuse color in the traversal
 *     state. The default diffuse color is 0.8,0.8,0.8. This results in full intensity
 *     values in the color map being displayed as 80% intensity. Therefore we recommend
 *     adding an SoMaterial node before the slice and setting its diffuseColor field to
 *     full white (1,1,1).
 *
 *   - Transparency: @BR @BR
 *     - Typically the color map (SoTransferFunction) used for volume rendering (SoVolumeRender)
 *       assigns transparency (alpha < 1) to some voxel values.  If you want to use the same
 *       color map for slice rendering, but render the slice completely opaque, set the #alphaUse
 *       field to ALPHA_OPAQUE. This overrides the alpha values in the color map (or an RGBA
 *       volume). However it does not affect transparency assigned using an SoMaterial node. @BR @BR
 *     - If you want to adjust the overall transparency of the slice, add an SoMaterial node and
 *       set its transparency field (keeping alphaUse set to ALPHA_AS_IS). Effectively a scale
 *       factor 1-transparency is applied to each voxel's alpha value. @BR @BR
 *     - Intersecting transparent slices cannot be rendered correctly by the basic blending
 *       transparency algorithms. To render this case correctly, set the transparency algorithm
 *       to SORTED_PIXEL using the viewer class or SoGLRenderAction.
 *
 *   - Voxel edges: @BR
 *     The edges of the voxels can also be rendered.
 *     See options in the SoVolumeRenderingQuality node.
 *
 *   - Custom shaders: @BR
 *     The current SoVolumeShader node, if any, allows custom shaders to be defined for
 *     special computation or rendering effects, including blending multiple volumes.
 *
 *   - Composition with Multiple Data: @BR
 *     It is possible to compose datasets that have different dimensions,
 *     tile sizes and transformations. @BR
 *     In order to help fetch the correct data values in custom shaders,
 *     texture coordinates conversion functions are provided in the
 *     @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 *     For instance,
 *     \code
 *     vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 *     \endcode
 *     can be used to convert texture coordinates related to one dataset to
 *     texture coordinates related to another dataset. @BR
 *     The conversion is based solely on the transformations applied to each
 *     dataset, which are defined by their model matrix and their extent. @BR
 *     Please note that the model matrix of a dataset is defined by to the
 *     SoTransformation nodes that are placed @B before @b the SoDataSet node in
 *     the order of the traversal.
 *
 *   - Performance: @BR
 *     - Tile size: @BR
 *       For backward compatibility, the default tile size is still only 64. This is quite
 *       small for modern CPU/GPU hardware. The smaller the tile size, the larger the total
 *       number of tiles that must be managed by VolumeViz. This overhead can be significant,
 *       especially for operations that require reloading the data textures on the GPU, for
 *       example, changing the data range (SoDataRange). For smaller volumes, like 512^3,
 *       it can be efficient to set the tile size large enough to contain the entire volume.
 *       For very large volumes, larger tile sizes are efficient for SoVolumeRender but
 *       somewhat inefficient for slice rendering because complete tiles must be loaded
 *       even though the slice only uses part of the data. Applications should experiment.
 *       @BR
 *       For volumes stored in LDM file format, the tile size must be specified when the
 *       volume is converted to LDM (see SoConverter and the "-t" option).  For other data
 *       data formats the tile size can be specified using the SoVolumeData node's
 *       @I ldmResourceParameters@i field, but only after setting the @I filename@i field
 *       or calling the \if_dotnet SetReader() \else setReader() \endif method.
 *
 *     - Tile cache policy:  
 *       It specifies how the tiles are stored in CPU memory. The selected policy
 *       can significantly impact the data loading performance versus the CPU memory footprint.
 *       See SoLDMResourceParameters::tileCachePolicy for detail.
 *
 *   - Hardware requirements: @BR
 *     This node needs a graphic card with support for GLSL shader, vertex buffer 
 *     objects (VBO) and framebuffer object (FBO). Use the isSupported() method to
 *     check if the current graphics board can render a FenceSlice.
 *
  * @EXAMPLE
  * Please see SoObliqueSlice for a complete code example.
  * The following shows how to set up the #axis and #points fields of SoFenceSlice.
  * \par
  * \if_cpp
  * \code
  *   SoFenceSlice* pFenceSlice = new SoFenceSlice;
  *     pFenceSlice->axis = SoFenceSlice::Y;
  *     pFenceSlice->points.set1Value( 0, SbVec2f(-0.2f, -0.66f) );
  *     pFenceSlice->points.set1Value( 1, SbVec2f( 0.2f, -0.4f ) );
  *     pFenceSlice->points.set1Value( 2, SbVec2f(-0.2f,  0.4f ) );
  *     pFenceSlice->points.set1Value( 3, SbVec2f( 0.2f,  0.66f) );
  * \endcode
  * \endif
  * \if_dotnet
  * \code
  *   SoFenceSlice fenceSlice   = new SoFenceSlice();
  *     fenceSlice.axis.Value = SoFenceSlice.AxisType.Y;
  *     fenceSlice.points[0] = new SbVec2f(-0.2f, -0.66f);
  *     fenceSlice.points[1] = new SbVec2f(0.2f, -0.4f);
  *     fenceSlice.points[2] = new SbVec2f(-0.2f, 0.4f);
  *     fenceSlice.points[3] = new SbVec2f(0.2f, 0.66f);
  * \endcode
  * \endif
  * \if_java
  * \code
  *   SoFenceSlice fenceSlice = new SoFenceSlice();
  *     fenceSlice.axis.setValue( SoFenceSlice.AxisType.Y );
  *     fenceSlice.points.set1Value( 0, new SbVec2f(-0.2f, -0.66f) );
  *     fenceSlice.points.set1Value( 1, new SbVec2f( 0.2f, -0.4f ) );
  *     fenceSlice.points.set1Value( 2, new SbVec2f(-0.2f,  0.4f ) );
  *     fenceSlice.points.set1Value( 3, new SbVec2f( 0.2f,  0.66f) );
  * \endcode
  * \endif
  *
  * \par
  * @TABLE_1B
  *   @TR Fence slice on Y axis (Colt example data set):
  *   @TR @IMAGE Img_FenceYaxis2.png
  *   @TR Fence slice on X axis (Colt example data set):
  *   @TR @IMAGE Img_FenceXaxis2.png
  *   @TR Fence slice on Z axis (Colt example data set):
  *   @TR @IMAGE Img_FenceZaxis2.png
  * @TABLE_END
  *
  * @FILE_FORMAT_DEFAULT
  *    FenceSlice {
  *    @TABLE_FILE_FORMAT
  *       @TR dataSetId       @TD 0
  *       @TR points          @TD [ ]
  *       @TR axis            @TD Z
  *       @TR interpolation   @TD LINEAR
  *       @TR alphaUse        @TD ALPHA_BINARY
  *       @TR useRGBA         @TD FALSE
  *       @TR alternateRep    @TD NULL
  *       @TR enableBumpMapping @TD FALSE
  *       @TR bumpScale       @TD 1.0
  *    @TABLE_END
  *    }
  *
  * @ACTION_BEHAVIOR
  *    SoGLRenderAction @BR
  *        Draws a textured shape based on current SoVolumeData, SoTransferFunction,
  *        and SoROI nodes.
  *
  *    SoGetBoundingBoxAction @BR
  *        Computes the bounding box that encloses the fence slice.
  *
  *
  * @SEE_ALSO
  *    SoVolumeData,
  *    SoDataRange,
  *    SoTransferFunction,
  *    SoROI,
  *    SoSlice
  *    SoObliqueSlice,
  *    SoOrthoSlice,
  *    SoFenceSliceDetail
  *
  *
  */
class VOLUMEVIZ_API SoFenceSlice : public SoSlice
{
  SO_NODE_HEADER( SoFenceSlice );

public:
  /**
   * Constructor.
   */
  SoFenceSlice();

  /**
   * Specifies the SoVolumeData node to use.
   *
   * This is useful when datasets of different dimensions are present in the scene graph.
   * Please see SoMultiDataSeparator for more details.
   *
   * When set to 0, the last SoVolumeData node on state is used.
   * Default is 0.
   *
   * @FIELD_SINCE_OIV 10.11.0
   */
  SoSFInt32 dataSetId;

  /**
   * Set of points defining a lineset. @BR
   * If #axis is:
   * - X: points are Y,Z coordinates
   * - Y: points are Z,X coordinates
   * - Z: points are X,Y coordinates
   *
   * All points should be inside the 3D extent of the volume.
   */
  SoMFVec2f points;

  enum Axis {
    X,
    Y,
    /** (Default) */
    Z
  };

  /**
   * Extrusion axis: X, Y, or Z.
   * @useenum{Axis}. Default is Z.
   */
  SoSFEnum axis;

  /** Returns true if graphic card can render an SoFenceSlice.
   * GPU must support GLSL.
   * When using a debug build of Open Inventor, some "no context available"
   * warning messages may be generated. You can ignore them or see
   * SoGLExtension for an example of using SoGLContext to avoid them.
   */
  static SbBool isSupported(SoState* state=NULL);

SoEXTENDER public:
  /**
   * Compute the bounding box
   */
  virtual void computeBBox( SoAction *action, SbBox3f &box, SbVec3f &center );

  virtual void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);

  /** Return true if the given tile has geometry */
  bool intersectGeometry(const SoLDMTileID& tile) const;

  virtual void doAction( SoAction *action );

  virtual void callback( SoCallbackAction *action );

  virtual void GLRender( SoGLRenderAction *action );

  virtual void rayPick( SoRayPickAction *action );

  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

SoINTERNAL public:

  static void initClass();
  static void exitClass();

  /**
   * Returns true if the shape intersects the given tile
   */
  virtual bool intersect( SoLDMNodeFrontManager* nfm, const SoLDMTileID& tileId );

  /** Handle notification */
  virtual void notify(SoNotList *list);

  /**
   * @copydoc SoVolumeShape::setRenderProgress
   */
  virtual void setRenderProgress(SoProgressIndicator* ps) override;

protected:

  /**
   * Generate primitive for the raypick action
   */
  virtual void generatePrimitives(SoAction *action);

  /**
   * Render the fence
   */
  virtual void doRendering(SoGLRenderAction *action);

  /**
   * Destructor
   */
  virtual ~SoFenceSlice();

SoINTERNAL protected :

  virtual void ldmAction( SoLdmValuationAction* action );

private:

  typedef std::vector<SbVec3f> VertexVector;
  typedef std::pair<float, float> FloatPair;

  /** 
   * Update m_geometry if needed and return it. 
   * If no action given in parameter, just the pointer to 
   * m_geometry is returned to allow field access
   */
  SoVolumeFaceSet* getGeometry( SoAction* action = NULL) const;

  /** Compute internal vertex properties */
  void computeVertexProperties( SoAction* action );

  /**
   * Compute intersecion of tileBbox with segment p0-p1
   * Clipped vertices are put into clippedLine
   * @param alpha holds interpolation factor for the 2 points
   * of clippedLine
   * @return false if no intersection
   */
  bool intersectLine(const SbBox3f& tileBbox, const SbVec2f& p0,
                     const SbVec2f& p1, SbVec2f* clippedLine,
                     FloatPair& alpha) const;

  SoVertexProperty *m_vertexProperty;
  mutable SoVolumeFaceSet* m_geometry;

  /** Per faces normals */
  std::vector<SbVec3f> m_normals;

  bool m_needRecomputeGeom;

  /** Mutex to protect access to m_geometry */
  mutable SbThreadMutex m_mutex;

};

#ifdef _WIN32
#pragma warning( pop )
#endif
#endif // _SO_FENCE_SLICE_


