/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#ifndef  _SO_OBLIQUE_SLICE_
#define  _SO_OBLIQUE_SLICE_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/nodes/SoShape.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFPlane.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/STL/vector>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoSlice.h>
#include <LDM/SoLDM.h>

class SoVolumeData;
class SoCache;
class SoLDMObliqueSliceGeom;

class SoObliqueSliceLdm;
class SoObliqueSliceInterface;
class SoObliqueSlicePaging;
class SoVolumeStateObliqueSlice;
class SoGetBoundingBoxAction;
class SoCallback;

/**
 * @VREXT Oblique slice shape node.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 *   This node defines an oblique (arbitrarily oriented) slice which is the intersection of the 
 *   volume data defined by an SoVolumeData node and the plane defined by the #plane field.
 *
 *   The SoVolumeData node can be specified with #dataSetId. When this field is set to 0,
 *   the last SoVolumeData node on state is used.
 *
 *   For a non-RGBA (scalar valued) volume, 
 *   each voxel's RGBA value is determined by the current SoDataRange and SoTransferFunction.
 *   The current @B diffuse color @b and @B transparency @b (set, for example, with
 *   an SoMaterial node) modify the appearance of the slice.  This means that, for
 *   example, the current transparency can be used as a @I global alpha @i value
 *   to modulate the overall opacity of the slice.  
 *   For an RGBA volume each voxel's RGBA value comes directly from the volume data.
 *
 *   The #interpolation field controls how the texture is interpolated.
 *
 *   The #alphaUse field (SoSlice) controls how the voxel's alpha component is used when drawing the slice.
 *
 *   Optionally a bump mapping effect may be applied. Normal vectors are automatically
 *   computed from the data value gradient.  The #enableBumpMapping and #bumpScale
 *   fields (SoSlice) control whether bump mapping is active and the intensity of the effect.
 *
 *   @B Notes: @b 
 *   - Transformation matrices: @BR
 *     The volume size and orientation (like geometry) can be modified by
 *     transformation nodes in the scene graph and this in turn modifies the
 *     appearance of volume visualization nodes.
 *     However the same transformation must be applied to the volume data node
 *     and all volume visualization nodes associated with that volume.  So effectively
 *     any transformation nodes that affect the volume must be placed @B before @b
 *     the volume data node.
 *
 *   - Plane normal: @BR
 *     The "polarity" of the plane normal (for example whether it is +Z or -Z) doesn't affect
 *     rendering the oblique slice.  However it's very important if you also use that plane to
 *     define a clipping plane. The clipping plane will clip away everything on the side of the
 *     plane *opposite* the normal vector direction. So to clip away the "front" (+Z) of the 
 *     volume set the normal to 0,0,-1.
 *
 *   - Picking: @BR
 *     The entire slice is pickable, even where it is transparent as a result of the
 *     current transfer function.  The SoObliqueSliceDetail class allows you to get the
 *     voxel position and value after picking.
 *
 *   - Dragging: @BR
 *     It is possible to interactively translate and rotate slices using an Open
 *     Inventor dragger, e.g. SoTranslate1Dragger. However the dragger's field values
 *     must be converted to an SbPlane. For a dragger that has a convenient "plane"
 *     field that can be directly connected, see the ClipPlaneDragger article in the
 *     Resources section of the Open Inventor Forum: http://forum.openinventor.com/showthread.php?t=733 
 *     (registration is required to view this section of the forum).
 *
 *   - Interpolation: @BR
 *     Interpolation is specified using the #interpolation field. The default (LINEAR)
 *     does bi-linear interpolation between voxel values. The NEAREST value can be used
 *     to display individual voxels. For best image quality we recommend using the
 *     TRILINEAR or MULTISAMPLE_12 value.
 *
 *   - Data range: @BR
 *     By default VolumeViz maps the entire range of the voxel's data type
 *     (e.g. 0..65535 for unsigned short) into the colormap.  This is often
 *     correct for byte (8 bit) voxels, but seldom correct for 16 bit voxels
 *     and never correct for floating point voxels. Use an SoDataRange node
 *     to specify the actual (or desired) range of data values to be mapped.
 *     Also use an SoDataRange node to implement brightness/contrast control
 *     like the Window/Level setting commonly used with medical images.
 *
 *   - Clipping: @BR
 *     Volume primitives can be clipped using a region of interest (SoROI), geometry
 *     (SoVolumeClippingGroup) and/or height fields (SoUniformGridClipping).
 *     They are also clipped by OpenGL clipping planes (SoClipPlane), but we recommend
 *     using the VolumeViz clipping nodes instead.
 *
 *   - Material: @BR
 *     The color of each voxel is modulated by the current diffuse color in the traversal
 *     state. The default diffuse color is 0.8,0.8,0.8. This results in full intensity
 *     values in the color map being displayed as 80% intensity. Therefore we recommend
 *     adding an SoMaterial node before the slice and setting its diffuseColor field to
 *     full white (1,1,1).
 *
 *   - Transparency: @BR @BR
 *     - Typically the color map (SoTransferFunction) used for volume rendering (SoVolumeRender)
 *       assigns transparency (alpha < 1) to some voxel values.  If you want to use the same
 *       color map for slice rendering, but render the slice completely opaque, set the #alphaUse
 *       field to ALPHA_OPAQUE. This overrides the alpha values in the color map (or an RGBA
 *       volume). However it does not affect transparency assigned using an SoMaterial node. @BR @BR
 *     - If you want to adjust the overall transparency of the slice, add an SoMaterial node and
 *       set its transparency field (keeping alphaUse set to ALPHA_AS_IS). Effectively a scale
 *       factor 1-transparency is applied to each voxel's alpha value. @BR @BR
 *     - Intersecting transparent slices cannot be rendered correctly by the basic blending
 *       transparency algorithms. To render this case correctly, set the transparency algorithm
 *       to SORTED_PIXEL using the viewer class or SoGLRenderAction.
 *
 *   - Voxel edges: @BR
 *     The edges of the voxels can also be rendered.
 *     See options in the SoVolumeRenderingQuality node.
 *
 *   - Custom shaders: @BR
 *     The current SoVolumeShader node, if any, allows custom shaders to be defined for
 *     special computation or rendering effects, including blending multiple volumes.
 *
 *   - Composition with Multiple Data: @BR
 *     It is possible to compose datasets that have different dimensions,
 *     tile sizes and transformations. @BR
 *     In order to help fetch the correct data values in custom shaders,
 *     texture coordinates conversion functions are provided in the
 *     @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 *     For instance,
 *     \code
 *     vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 *     \endcode
 *     can be used to convert texture coordinates related to one dataset to
 *     texture coordinates related to another dataset. @BR
 *     The conversion is based solely on the transformations applied to each
 *     dataset, which are defined by their model matrix and their extent. @BR
 *     Please note that the model matrix of a dataset is defined by to the
 *     SoTransformation nodes that are placed @B before @b the SoDataSet node in
 *     the order of the traversal.
 *
 *   - Performance: @BR
 *     - Tile size: @BR
 *       For backward compatibility, the default tile size is still only 64. This is quite
 *       small for modern CPU/GPU hardware. The smaller the tile size, the larger the total
 *       number of tiles that must be managed by VolumeViz. This overhead can be significant,
 *       especially for operations that require reloading the data textures on the GPU, for
 *       example, changing the data range (SoDataRange). For smaller volumes, like 512^3,
 *       it can be efficient to set the tile size large enough to contain the entire volume.
 *       For very large volumes, larger tile sizes are efficient for SoVolumeRender but
 *       somewhat inefficient for slice rendering because complete tiles must be loaded
 *       even though the slice only uses part of the data. Applications should experiment.
 *       @BR
 *       For volumes stored in LDM file format, the tile size must be specified when the
 *       volume is converted to LDM (see SoConverter and the "-t" option).  For other data
 *       data formats the tile size can be specified using the SoVolumeData node's
 *       @I ldmResourceParameters@i field, but only after setting the @I filename@i field
 *       or calling the \if_dotnet SetReader() \else setReader() \endif method.
 *
 *     - Tile cache policy:  
 *       It specifies how the tiles are stored in CPU memory. The selected policy
 *       can significantly impact the data loading performance versus the CPU memory footprint.
 *       See SoLDMResourceParameters::tileCachePolicy for detail.
 *
 * @EXAMPLE
 *   For simple data sets, a basic VolumeViz rendering could be achieved
 *   with only a few nodes: minimally an SoVolumeData node to identify
 *   the data set and one rendering node.  However most data sets need
 *   at least some of the additional nodes shown here in order to get
 *   a correct and useful rendering. Most applications will need
 *   additional nodes to take advantage of region of interest, interaction,
 *   clipping and other VolumeViz features.  Please consider
 *   the code shown here as simply a guideline and a starting point for
 *   exploring the many powerful features available in Open Inventor.
 *   
 *   Note that some of the property nodes (data, material, color map,
 *   etc) will typically be shared by multiple rendering nodes.  In other
 *   words the volume usually only needs to be loaded once, using a single
 *   SoVolumeData node, then multiple slices and/or regions can be rendered
 *   using that data node.
 *
 *   Also note that this example is for a data volume, where each voxel can be
 *   considered a discrete sample from a continuous data field and interpolation
 *   should be used to compute values between voxel centers. If you are
 *   rendering a label volume, then each voxel is an "id" assigning that
 *   voxel to a specific material, object, etc.  In this case, set the
 *   @I interpolation@i field to NEAREST to disable interpolation.
* \if_cpp
* \code
*   // Keep volume viz separate from geometry
*   SoSeparator* volSep = new SoSeparator();
*     root->addChild( volSep );
*
*   // Load volume data
*   SoVolumeData* volData = new SoVolumeData();
*     volData->fileName = "$OIVHOME/examples/data/VolumeViz/3DHead.vol";
*     volSep->addChild( volData );
*
*   // Set range of data values to visualize.
*   // Not required for 8-bit voxels, critical for larger data types.
*   // The getMinMax() call may be expensive for non-LDM file formats.
*   SoDataRange* volRange = new SoDataRange();
*     if (volData->getDatumSize() > 1) {
*       double minVal, maxVal;
*       volData->getMinMax( minVal, maxVal );
*       volRange->min = minVal;
*       volRange->max = maxVal;
*     }
*     volSep->addChild( volRange );
*
*   // Load opaque intensity ramp
*   SoTransferFunction* volTF = new SoTransferFunction();
*     volTF->predefColorMap = SoTransferFunction::INTENSITY;
*     volSep->addChild( volTF );
*
*   // Display slice at full intensity
*   SoMaterial* volMat = new SoMaterial();
*     volMat->diffuseColor.setValue( 1, 1, 1 );
*     volSep->addChild( volMat );
*
*   // Remove tile boundary artifacts while moving. 
*   SoVolumeShader* volShader = new SoVolumeShader();
*     volShader->interpolateOnMove = TRUE;
*     volSep->addChild( volShader );
*
*   // Display a Z axis slice at center of volume
*   SoObliqueSlice* slice = new SoObliqueSlice();
*     SbBox3f volext = volData->extent.getValue();
*     SbVec3f center = volext.getCenter();
*     slice->plane         = SbPlane( SbVec3f(0,0,1), center[2] );
*     slice->interpolation = SoObliqueSlice::MULTISAMPLE_12;
*     volSep->addChild( slice );
* \endcode
* \endif
* \if_dotnet
* \code
*   SoSeparator volSep = new SoSeparator();
*   root.AddChild( volSep );
*
*   // Load volume data
*   SoVolumeData volData = new SoVolumeData();
*   volData.fileName.Value = "$OIVNETHOME/src/demos/data/VolumeViz/3DHead.vol";
*   volSep.AddChild( volData );
*
*   // Set range of data values to visualize.
*   // Not required for 8-bit voxels, critical for larger data types.
*   // The getMinMax() call may be expensive for non-LDM file formats.
*   SoDataRange volRange = new SoDataRange();
*   if (volData.GetDatumSize() > 1)
*   {
*       double minVal, maxVal;
*       volData.GetMinMax( out minVal, out maxVal);
*       volRange.min.Value = minVal;
*       volRange.max.Value = maxVal;
*   }
*   volSep.AddChild( volRange );
*
*   // Load opaque intensity ramp
*   SoTransferFunction volTF = new SoTransferFunction();
*   volTF.predefColorMap.Value = SoTransferFunction.PredefColorMaps.INTENSITY;
*   volSep.AddChild( volTF );
*
*   // Display slice at full intensity
*   SoMaterial volMat = new SoMaterial();
*   volMat.diffuseColor.SetValue(1, 1, 1);
*   volSep.AddChild( volMat );
*
*   // Remove tile boundary artifacts while moving. 
*   SoVolumeShader volShader = new SoVolumeShader();
*   volShader.interpolateOnMove.Value = true;
*   volSep.AddChild( volShader );
*
*   SoObliqueSlice slice = new SoObliqueSlice();
*   SbBox3f volext = volData.extent.Value;
*   SbVec3f center = volext.GetCenter();
*   slice.plane.Value = new SbPlane(new SbVec3f(0, 0, 1), center[2]);
*   slice.interpolation.Value = SoVolumeShape.Interpolations.MULTISAMPLE_12;
*   volSep.AddChild( slice );
* \endcode
* \endif
* \if_java
* \code
*   // Keep volume viz separate from geometry
*   SoSeparator volSep = new SoSeparator();
*     root.addChild( volSep );
*
*   // Load volume data
*   SoVolumeData volData = new SoVolumeData();
*     volData.fileName.setValue( "$OIVJHOME/data/VolumeViz/3DHead.vol" );
*     volSep.addChild( volData );
*
*   // Set range of data values to visualize.
*   // Not required for 8-bit voxels, critical for larger data types.
*   // The getMinMax() call may be expensive for non-LDM file formats.
*   SoDataRange volRange = new SoDataRange();
*     if (volData.getDatumSize() > 1) {
*       double[] minmax;
*       minmax = volData.getDoubleMinMax();
*       volRange.min.setValue( minmax[0] );
*       volRange.max.setValue( minmax[1] );
*     }
*     volSep.addChild( volRange );
*
*   // Load opaque intensity ramp
*   SoTransferFunction volTF = new SoTransferFunction();
*     volTF.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.INTENSITY );
*     volSep.addChild( volTF );
*
*   // Display slice at full intensity
*   SoMaterial volMat = new SoMaterial();
*     volMat.diffuseColor.setValue( 1, 1, 1 );
*     volSep.addChild( volMat );
*
*   // Remove tile boundary artifacts while moving.
*   SoVolumeShader volShader = new SoVolumeShader();
*     volShader.interpolateOnMove.setValue( true );
*     volSep.addChild( volShader );
*
*   // Display a Z axis slice at center of volume
*   SoObliqueSlice slice = new SoObliqueSlice();
*     SbBox3f volext = volData.extent.getValue();
*     SbVec3f center = volext.getCenter();
*     slice.plane.setValue( new SbPlane( new SbVec3f(0,0,1), center.getZ() ) );
*     slice.interpolation.setValue( SoVolumeShape.Interpolations.MULTISAMPLE_12 );
*     volSep.addChild( slice );
* \endcode
* \endif
 *
 * @FILE_FORMAT_DEFAULT
 *    ObliqueSlice {
 *    @TABLE_FILE_FORMAT
 *       @TR dataSetId       @TD 0
 *       @TR plane           @TD 0, 0, 1, 0
 *       @TR interpolation   @TD LINEAR
 *       @TR alphaUse        @TD ALPHA_BINARY
 *       @TR useRGBA         @TD FALSE
 *       @TR alternateRep    @TD NULL
 *       @TR enableBumpMapping @TD FALSE
 *       @TR bumpScale       @TD 1.0
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws a textured polygon based on current SoVolumeData, SoTransferFunction, and
 *        SoROI nodes.
 *
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses the slice.
 *
 * @SEE_ALSO
 *    SoVolumeData,
 *    SoDataRange,
 *    SoTransferFunction,
 *    SoROI,
 *    SoSlice,
 *    SoOrthoSlice,
 *    SoVolumeShader
 *
 *
 */
class VOLUMEVIZ_API SoObliqueSlice : public SoSlice {
  SO_NODE_HEADER( SoObliqueSlice );

 public:

  /**
   * Specifies the SoVolumeData node to use.
   *
   * This is useful when datasets of different dimensions are present in the scene graph.
   * Please see SoMultiDataSeparator for more details.
   *
   * When set to 0, the last SoVolumeData node on state is used.
   * Default is 0.
   *
   * @FIELD_SINCE_OIV 10.11.0
   */
  SoSFInt32 dataSetId;

  /**
   * Plane defining the slice. The default plane corresponds to the XY plane.
   */
  SoSFPlane plane;

  /**
   * Constructor.
   */
  SoObliqueSlice();



SoEXTENDER public:
  /**
   * Compute the bounding box
   */
  void computeBBox(SoAction *, SbBox3f &box, SbVec3f &center);

  /**
   * Implements SoWriteAction for this node.
   * Creates alternateRep subgraph if enabled.
   */
  virtual void write(SoWriteAction *action);
  
  /**
   * Returns an alternate represention for this node. @BR
   * This is typically a scene graph that gives approximately the same
   * appearance using only core Open Inventor nodes.  For example a
   * slice can be represented using a FaceSet and a Texture2 node.
   */
  virtual SoNode* getAlternateRep( SoAction* action );

  virtual void GLRender( SoGLRenderAction* action );

SoINTERNAL public:
  /**
   * Node init
   */
  static void initClass();
  static void exitClass();

  virtual bool intersect(SoLDMNodeFrontManager* nfm, const SoLDMTileID& tileID) override;

  /**
   * Create a subgraph for the alternateRep field that gives approximately
   * the same appearance, but using only core Open Inventor nodes.  Used by the
   * write() method when SoVolumeRendering::setWriteAlternateRep(TRUE) was
   * called.  Might be useful in other cases.
   */
  virtual SoNode* createAlternateRep( SoAction *action );

  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  /**
   * @return LDM slice identifier
   */
  int getLDMSliceID() const;

  bool m_backFace;

protected:

  /**
 * Send all fields to the SoOrthoSliceInterface
 */
  void setupSliceInterface();

  /**
   * Generate primitive for the raypick action
   */
  void generatePrimitives(SoAction* action);

  /**
   * Create triangle detail for the raypick action
   */
  SoDetail* createTriangleDetail(SoRayPickAction*,
    const SoPrimitiveVertex* v1,
    const SoPrimitiveVertex*,
    const SoPrimitiveVertex*,
    SoPickedPoint*);

  /**
   * Render the slice
   */
  void doRendering(SoGLRenderAction* action);

  /**
   * Reads data into instance. Returns FALSE on error.
   * Deletes alternateRep subgraph unless requested not to.
   */
  virtual SbBool readInstance(SoInput* in, unsigned short flags);

  /**
   * Destructor
   */
  virtual ~SoObliqueSlice();

  /**
   * remember if we created the altRep
   */
  SbBool m_altRepCreatedHere;

  SoObliqueSliceInterface* m_sliceInterface;
  SoObliqueSliceLdm* m_sliceLdm;

  SoVolumeStateObliqueSlice* m_vs;

private:
  SoCallback* m_preEltsCBNode;
  SoCallback* m_postEltsCBNode;
  int m_savedVertexOrdering;
  int m_savedShapeType;
  SoLDM::DataSetIdPair m_savedMasterIdPair;


  friend class SoObliqueSliceLdm;
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // _SO_OBLIQUE_SLICE_

