/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#ifndef _SO_ORTHO_SLICE_
#define _SO_ORTHO_SLICE_

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <VolumeViz/nodes/SoSlice.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/SbPImpl.h>

class SoChildList;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoOrthoSlice)

/**
 * @VREXT Ortho slice shape node.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 *   This node defines an ortho (axis aligned) slice along the X, Y, or Z axis of the volume data
 *   defined by an SoVolumeData node. The slice orientation and position are defined by the #axis
 *   and #sliceNumber fields.
 *
 *   The SoVolumeData node can be specified with #dataSetId. When this field is set to 0,
 *   the last SoVolumeData node on state is used.
 *
 *   For a non-RGBA (scalar valued) volume, 
 *   each voxel's RGBA value is determined by the current SoDataRange and SoTransferFunction.
 *   The current @B diffuse color @b and @B transparency @b (set, for example, with
 *   an SoMaterial node) modifies the appearance of the slice.  This means that, for
 *   example, the current transparency can be used as an @I global alpha @i value
 *   to modulate the overall opacity of the slice.  
 *   For an RGBA volume each voxel's RGBA value comes directly from the volume data.
 *
 *   The #interpolation field controls how the texture is interpolated.
 *
 *   The #alphaUse field (SoSlice) controls how the voxel's alpha component is used when drawing the slice.
 *
 *   Optionally a bump mapping effect may be applied. Normal vectors are automatically
 *   computed from the data value gradient.  The #enableBumpMapping and #bumpScale
 *   fields (SoSlice) control whether bump mapping is active and the intensity of the effect.
 *
 *   @B Notes: @b 
 *   - Drawing position: @BR
 *     The SoOrthoSlice geometry passes through the center of the voxels in the specified
 *     slice. So, for example, an ortho slice with sliceNumber = 0 will be drawn one-half
 *     voxel size (on the slice axis) in from the bounding box of the volume. This is
 *     slightly different from SoVolumeSkin. A volume skin is approximately the same as six
 *     ortho slices, but each face of the skin is drawn at the outer edge of the voxel.
 *
 *   - Transformation matrices: @BR
 *     The volume size and orientation (like geometry) can be modified by
 *     transformation nodes in the scene graph and this in turn modifies the
 *     appearance of volume visualization nodes.
 *     However, the same transformation must be applied to the volume data node
 *     and all volume visualization nodes associated with that volume.  So effectively
 *     any transformation nodes that affect the volume must be placed @B before @b
 *     the volume data node.
 *
 *   - Picking: @BR
 *     The entire slice is pickable, even where it is transparent as a result of the
 *     current transfer function.  The SoOrthoSliceDetail class allows you to get the
 *     voxel position and value after picking.
 *
 *   - Dragging: @BR
 *     It is possible to interactively translate and rotate slices using an Open
 *     Inventor dragger, e.g. SoTranslate1Dragger. However the dragger's position in
 *     XYZ space must be converted to a slice number. For a dragger that is specific
 *     to ortho slices, see the class SoOrthoSliceDragger.
 *
 *   - Interpolation: @BR
 *     Interpolation is specified using the #interpolation field. The default (LINEAR)
 *     does bi-linear interpolation between voxel values. The NEAREST value can be used
 *     to display individual voxels. For best image quality we recommend using the
 *     MULTISAMPLE_12 value.
 *
 *   - Data range: @BR
 *     By default VolumeViz maps the entire range of the voxel's data type
 *     (e.g. 0..65535 for unsigned short) into the colormap.  This is often
 *     correct for byte (8 bit) voxels, but seldom correct for 16 bit voxels
 *     and never correct for floating point voxels. Use an SoDataRange node
 *     to specify the actual (or desired) range of data values to be mapped.
 *     Also use an SoDataRange node to implement brightness/contrast control
 *     like the Window/Level setting commonly used with medical images.
 *
 *   - Clipping: @BR
 *     Volume primitives can be clipped using a region of interest (SoROI), geometry
 *     (SoVolumeClippingGroup) and/or height fields (SoUniformGridClipping).
 *     They are also clipped by OpenGL clipping planes (SoClipPlane), but we recommend
 *     using the VolumeViz clipping nodes instead. @BR
 *     Optionally, this node also defines a clipping plane. Similar to SoClipPlane, this
 *     clipping plane affects all subsequent geometry, including SoVolumeRender, but does
 *     not of course affect the ortho slice itself.
 *     The #clipping and #clippingSide fields control whether clipping is
 *     active and which half-space is clipped.  Clipping side FRONT means that the clip
 *     plane (removes) clips away geometry in the positive direction on the slice axis.
 *
 *   - Border: @BR
 *     Border can be enabled with the #enableBorder field, which allows to draw a border around the slice following volume boundary.
 *     However, the border of the original slice are displayed even if the slice is cropped by a SoROI (or SoROIManip).
 *
 *   - Material: @BR
 *     The color of each voxel is modulated by the current diffuse color in the traversal
 *     state. The default diffuse color is 0.8,0.8,0.8. This results in full intensity
 *     values in the color map being displayed as 80% intensity. Therefore we recommend
 *     adding an SoMaterial node before the slice and setting its diffuseColor field to
 *     full white (1,1,1).
 *
 *   - Transparency: @BR @BR
 *     - Typically the color map (SoTransferFunction) used for volume rendering (SoVolumeRender)
 *       assigns transparency (alpha < 1) to some voxel values.  You can use the same color map
 *       for slice rendering, but note that the default value of the #alphaUse field (since
 *       Open Inventor 10.0) is ALPHA_OPAQUE, meaning that alpha values in the color map are
 *       ignored. However, it does not affect the transparency assigned by an SoMaterial node. @BR @BR
 *     - If you want to adjust the overall transparency of the slice, set the #alphaUse field to
 *       ALPHA_AS_IS, add an SoMaterial node and set its transparency field. Effectively a scale
 *       factor 1-transparency is applied to each voxel's alpha value. @BR @BR
 *     - Intersecting transparent slices cannot be rendered correctly by the basic blending
 *       transparency algorithms. To render this case correctly, set the transparency algorithm
 *       to SORTED_PIXEL using the viewer class or SoGLRenderAction.
 *
 *   - Voxel edges: @BR
 *     The edges of the voxels can also be rendered.
 *     See options in the SoVolumeRenderingQuality node.
 *
 *   - Custom shaders: @BR
 *     The current SoVolumeShader node, if any, allows custom shaders to be defined for
 *     special computation or rendering effects, including blending multiple volumes.
 *
 *   - Composition with Multiple Data: @BR
 *     It is possible to compose datasets that have different dimensions,
 *     tile sizes and transformations. @BR
 *     In order to help fetch the correct data values in custom shaders,
 *     texture coordinates conversion functions are provided in the
 *     @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 *     For instance,
 *     \code
 *     vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 *     \endcode
 *     can be used to convert texture coordinates related to one dataset to
 *     texture coordinates related to another dataset. @BR
 *     The conversion is based solely on the transformations applied to each
 *     dataset, which are defined by their model matrix and their extent. @BR
 *     Please note that the model matrix of a dataset is defined by to the
 *     SoTransformation nodes that are placed @B before @b the SoDataSet node in
 *     the order of the traversal.
 *
 *   - Large Slice mode: @BR
 *     When the "large slice" mode is enabled (see SoSlice::largeSliceSupport), if all the
 *     required full resolution tiles have already been loaded, then
 *     the slice data is taken from LDM system memory cache as usual. But if some required 
 *     tiles are not currently in memory, the required slice data will be loaded directly 
 *     from the volume reader @I without @i loading the complete tiles. This reduces disk I/O
 *     and reduces the amount of system memory required to display the slice at full resolution,
 *     so larger (or more) slices can be displayed. The required tiles are then scheduled to be
 *     loaded asynchronously in case adjacent slices are displayed later. 
 *     For example, loading a 1024x1024 SoOrthoSlice from an 8-bit dataset with 128x128x128
 *     tiles would normally require loading 1024x1024x128 bytes of data (as complete tiles). 
 *     With largeSliceSupport enabled, only 1024x1024 bytes (maximum) of data need to
 *     be loaded (in the worst case where no high resolution data is currently in memory).
 *
 *   - Performance: @BR
 *     - Tile size: @BR
 *       For backward compatibility, the default tile size is still only 64. This is quite
 *       small for modern CPU/GPU hardware. The smaller the tile size, the larger the total
 *       number of tiles that must be managed by VolumeViz. This overhead can be significant,
 *       especially for operations that require reloading the data textures on the GPU, for
 *       example, changing the data range (SoDataRange). For smaller volumes, like 512^3,
 *       it can be efficient to set the tile size large enough to contain the entire volume.
 *       For very large volumes, larger tile sizes are efficient for SoVolumeRender but
 *       somewhat inefficient for slice rendering because complete tiles must be loaded
 *       even though the slice only uses a part of the data. Applications should experiment.
 *       @BR
 *       For volumes stored in LDM file format, the tile size must be specified when the
 *       volume is converted to LDM (see SoConverter and the "-t" option).  For other data
 *       formats the tile size can be specified using the SoVolumeData node's
 *       @I ldmResourceParameters@i field, but only after setting the @I filename@i field
 *       or calling the \if_dotnet SetReader() \else setReader() \endif method.
 *
 *     - Tile cache policy:  
 *       It specifies how the tiles are stored in CPU memory. The selected policy
 *       can significantly impact the data loading performance versus the CPU memory footprint.
 *       See SoLDMResourceParameters::tileCachePolicy for details.
 *
 *     - Compressed textures @BR
 *       For performance reasons, SoOrthoSlice accumulates small textures into a bigger
 *       one. When using compressed RGBA textures (via #SoDataSet's field useCompressedTexture),
 *       this optimization cannot be done.
 *       If you want to favor performance rather than memory usage, you should disable compression
 *       (enabled by default if supported by the graphic card)
 *
 *   - Medical data axes: @BR
 *     The three standard axes (or viewing planes) are Axial, Coronal and Sagittal.
 *     How these axes correspond to the IJK axes in voxel space and the XYZ axes in 3D space depends
 *     on the data set. However DICOM volumes typically have LPS (Left, Posterior, Superior) orientation
 *     and in this case:
 *     - Axial is the Z axis @BR
 *       i.e., toward the head (Superior) is the K / +Z / Axial axis. @BR @BR
 *     - Coronal is the Y axis @BR
 *       i.e., toward the back of the body (Posterior) is the J / +Y / Coronal axis. @BR @BR
 *     - Sagittal is the X axis @BR
 *       i.e., toward the left side of the body (Left) is the I / +X / Sagittal axis.
 *
 * @EXAMPLE
 *   For simple data sets, a basic VolumeViz rendering could be achieved
 *   with only a few nodes: minimally an SoVolumeData node to identify
 *   the data set and one rendering node.  However most data sets need
 *   at least some of the additional nodes shown here in order to get
 *   a correct and useful rendering. Most applications will need
 *   additional nodes to take advantage of region of interest, interaction,
 *   clipping and other VolumeViz features.  Please consider
 *   the code shown here as simply a guideline and a starting point for
 *   exploring the many powerful features available in Open Inventor.
 *   
 *   Note that some of the property nodes (data, material, color map,
 *   etc) will typically be shared by multiple rendering nodes.  In other
 *   words the volume usually only needs to be loaded once, using a single
 *   SoVolumeData node, then multiple slices and/or regions can be rendered
 *   using that data node.
 *
 *   Also note that this example is for a data volume, where each voxel can be
 *   considered a discrete sample from a continuous data field and interpolation
 *   should be used to compute values between voxel centers. If you are
 *   rendering a label volume, then each voxel is an "id" assigning that
 *   voxel to a specific material, object, etc.  In this case, set the
 *   @I interpolation@i field to NEAREST to disable interpolation.
 * \if_cpp
 * \code
 *   // Keeps volume viz separate from geometry
 *   SoSeparator* volSep = new SoSeparator();
 *     root->addChild( volSep );
 *
 *   // Loads volume data
 *   SoVolumeData* volData = new SoVolumeData();
 *     volData->fileName = "$OIVHOME/examples/data/VolumeViz/3DHead.vol";
 *     volSep->addChild( volData );
 *
 *   // Sets range of data values to visualize.
 *   // Not required for 8-bit voxels, critical for larger data types.
 *   // The getMinMax() call may be expensive for non-LDM file formats.
 *   SoDataRange* volRange = new SoDataRange();
 *     if (volData->getDatumSize() > 1) {
 *       double minVal, maxVal;
 *       volData->getMinMax( minVal, maxVal );
 *       volRange->min = minVal;
 *       volRange->max = maxVal;
 *     }
 *     volSep->addChild( volRange );
 *
 *   // Loads opaque intensity ramp
 *   SoTransferFunction* volTF = new SoTransferFunction();
 *     volTF->predefColorMap = SoTransferFunction::INTENSITY;
 *     volSep->addChild( volTF );
 *
 *   // Displays slice at full intensity
 *   SoMaterial* volMat = new SoMaterial();
 *     volMat->diffuseColor.setValue( 1, 1, 1 );
 *     volSep->addChild( volMat );
 *
 *   // Removes tile boundary artifacts while moving. 
 *   SoVolumeShader* volShader = new SoVolumeShader();
 *     volShader->interpolateOnMove = TRUE;
 *     volSep->addChild( volShader );
 *
 *   // Displays a Z axis slice at center of volume
 *   SoOrthoSlice* slice = new SoOrthoSlice();
 *     SbVec3i32 voldim = volData->data.getSize();
 *     slice->axis          = SoOrthoSlice::Z;
 *     slice->sliceNumber   = voldim[2] / 2;
 *     slice->interpolation = SoOrthoSlice::MULTISAMPLE_12;
 *     volSep->addChild( slice );
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   SoSeparator volSep = new SoSeparator();
 *   root.AddChild( volSep );
 *
 *   // Loads volume data
 *   SoVolumeData volData = new SoVolumeData();
 *   volData.fileName.Value = "$OIVNETHOME/src/demos/data/VolumeViz/3DHead.vol";
 *   volSep.AddChild( volData );
 *
 *   // Sets range of data values to visualize.
 *   // Not required for 8-bit voxels, critical for larger data types.
 *   // The getMinMax() call may be expensive for non-LDM file formats.
 *   SoDataRange volRange = new SoDataRange();
 *   if (volData.GetDatumSize() > 1)
 *   {
 *       double minVal, maxVal;
 *       volData.GetMinMax( out minVal, out maxVal);
 *       volRange.min.Value = minVal;
 *       volRange.max.Value = maxVal;
 *   }
 *   volSep.AddChild( volRange );
 *
 *   // Loads opaque intensity ramp
 *   SoTransferFunction volTF = new SoTransferFunction();
 *   volTF.predefColorMap.Value = SoTransferFunction.PredefColorMaps.INTENSITY;
 *   volSep.AddChild( volTF );
 *
 *   // Displays slice at full intensity
 *   SoMaterial volMat = new SoMaterial();
 *   volMat.diffuseColor.SetValue(1, 1, 1);
 *   volSep.AddChild( volMat );
 *
 *   // Removes tile boundary artifacts while moving. 
 *   SoVolumeShader volShader = new SoVolumeShader();
 *   volShader.interpolateOnMove.Value = true;
 *   volSep.AddChild( volShader );
 *
 *   // Displays a Z axis slice at center of volume
 *   SoOrthoSlice slice = new SoOrthoSlice();
 *   SbVec3i32 voldim = volData.data.GetSize();
 *   slice.axis.Value = SoOrthoSlice.AxisType.Z;
 *   slice.sliceNumber.Value = (uint)(voldim[2] / 2);
 *   slice.interpolation.Value = SoVolumeShape.Interpolations.MULTISAMPLE_12;
 *   volSep.AddChild( slice );
 * \endcode
 * \endif
 * \if_java
 * \code
 *   // Keeps volume viz separate from geometry
 *   SoSeparator volSep = new SoSeparator();
 *     root.addChild(volSep);
 *
 *   // Loads volume data
 *   SoVolumeData volData = new SoVolumeData();
 *     volData.fileName.setValue( "$OIVJHOME/data/VolumeViz/3DHead.vol" );
 *     volSep.addChild( volData );
 *
 *   // Sets range of data values to visualize.
 *   // Not required for 8-bit voxels, critical for larger data types.
 *   // The getMinMax() call may be expensive for non-LDM file formats.
 *   SoDataRange volRange = new SoDataRange();
 *     if (volData.getDatumSize() > 1) {
 *       double[] minmax;
 *       minmax = volData.getDoubleMinMax();
 *       volRange.min.setValue( minmax[0] );
 *       volRange.max.setValue( minmax[1] );
 *     }
 *     volSep.addChild( volRange );
 *
 *   // Loads opaque intensity ramp
 *   SoTransferFunction volTF = new SoTransferFunction();
 *     volTF.predefColorMap.setValue( SoTransferFunction.PredefColorMaps.INTENSITY );
 *     volSep.addChild( volTF );
 *
 *   // Displays slice at full intensity
 *   SoMaterial volMat = new SoMaterial();
 *     volMat.diffuseColor.setValue( 1, 1, 1 );
 *     volSep.addChild( volMat );
 *
 *   // Removes tile boundary artifacts while moving.
 *   SoVolumeShader volShader = new SoVolumeShader();
 *     volShader.interpolateOnMove.setValue( true );
 *     volSep.addChild( volShader );
 *
 *   // Displays a Z axis slice at center of volume
 *   SoOrthoSlice slice = new SoOrthoSlice();
 *     SbVec3i32 voldim = volData.data.getSize();
 *     slice.axis.setValue( SoOrthoSlice.AxisType.Z );
 *     slice.sliceNumber.setValue( voldim.getZ() / 2 );
 *     slice.interpolation.setValue( SoVolumeShape.Interpolations.MULTISAMPLE_12 );
 *     volSep.addChild( slice );
 * \endcode
 * \endif
 *
 * @FILE_FORMAT_DEFAULT
 *    OrthoSlice {
 *    @TABLE_FILE_FORMAT
 *       @TR dataSetId       @TD 0
 *       @TR sliceNumber     @TD 0
 *       @TR axis            @TD Z
 *       @TR interpolation   @TD LINEAR
 *       @TR alphaUse        @TD ALPHA_BINARY
 *       @TR useRGBA         @TD FALSE
 *       @TR clipping        @TD FALSE
 *       @TR clippingSide    @TD BACK
 *       @TR alternateRep    @TD NULL
 *       @TR enableBumpMapping @TD FALSE
 *       @TR bumpScale       @TD 1.0
 *       @TR enableBorder    @TD FALSE
 *       @TR borderColor     @TD 0.84 0.43 0.02
 *       @TR borderWidth     @TD 2.0
 *       @TR enableImage     @TD TRUE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws a textured rectangle based on current SoVolumeData, SoTransferFunction,
 *        and SoROI nodes. Sets: SoClipPlaneElement
 *
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses the slice.
 *
 *
 * @SEE_ALSO
 *    SoDataRange,
 *    SoObliqueSlice,
 *    SoOrthoSliceDragger,
 *    SoROI,
 *    SoSlice,
 *    SoTransferFunction,
 *    SoVolumeData
 *
 *
 */
class VOLUMEVIZ_API SoOrthoSlice : public SoSlice
{
  SO_NODE_HEADER( SoOrthoSlice );
  SO_PIMPL_BASE_PUBLIC_HEADER( SoOrthoSlice );

public:

  /**
   * Specifies the SoVolumeData node to use.
   *
   * This is useful when datasets of different dimensions are present in the scene graph.
   * Please see SoMultiDataSeparator for more details.
   *
   * When set to 0, the last SoVolumeData node on state is used.
   * Default is 0.
   *
   * @FIELD_SINCE_OIV 10.11.0
   */
  SoSFInt32 dataSetId;

  /**
   * Slice number.
   */
  SoSFUInt32 sliceNumber;

  /** Slice Axis (see discussion of medical data axes in the class description) */
  enum Axis {
    /** X Axis */
    X,
    /** Y Axis */
    Y,
    /** Z Axis (Default) */
    Z
  };

  /**
   * Slice axis (X, Y, or Z).
   * @useenum{Axis}. Default is Z.
   */
  SoSFEnum axis;

  /** Clipping Side mode */
  enum ClippingSide
  {
    /** Front */
    FRONT,
    /** Back (Default) */
    BACK
  };

  /**
   * Specifies the clipping side.
   * @useenum{ClippingSide}. Default is BACK.
   * Clipping side FRONT means that the clip plane (removes) clips away geometry
   * in the positive direction on the slice axis. BACK means that the clip plane
   * clips away geometry in the negative direction on the slice axis.
   *
   *    @TABLE_0B
   *       @TR These figures show an ortho slice clipping a data volume.
   *
   *           Right:
   *             - #alphaUse = ALPHA_BINARY @BR
   *             - #interpolation = LINEAR
   *
   *           Bottom left:
   *             - #alphaUse = ALPHA_OPAQUE @BR
   *             - #interpolation = LINEAR
   *
   *           Bottom right:
   *             - #alphaUse = ALPHA_BINARY @BR
   *             - #interpolation = NEAREST
   *
   *       @TD @IMAGE binarylinear.jpg
   *       @TR @IMAGE opaquelinear.jpg
   *       @TD @IMAGE binarynearest.jpg
   *    @TABLE_END
   *
   */
  SoSFEnum clippingSide;

  /**
   * Activate/deactivate a clipping plane on a per-slice basis.
   *     Optionally, this node also defines a clipping plane. Similar to SoClipPlane, this
   *     clipping plane affects all subsequent geometry, including SoVolumeRender, but does
   *     not of course affect the ortho slice itself.
   *     The #clippingSide field controls which half-space is clipped. 
   */
  SoSFBool clipping;

  /**
   *  Enables drawing the border. Default is false.
   *
   *  @FIELD_SINCE_OIV 10.10
   */
  SoSFBool   enableBorder;

  /**
   *  Border color.  Default is [0.84, 0.43, 0.02] (orange luminance 55%).
   *
   *  @FIELD_SINCE_OIV 10.10
   */
  SoSFColor  borderColor;

  /**
   *  Border width in pixels. Default is 2.
   *  For more information about borderWidth units: see SoDrawStyle::lineWidth
   *
   *  @FIELD_SINCE_OIV 10.10
   */
  SoSFFloat  borderWidth;

  /**
   *  Enables drawing the slice image. Default is true.
   *  Note that setting this field to false makes the slice image invisible,
   *  but the slice is @I still detectable by picking@i if the border is enabled.
   *
   *  Example of a typical use case : The folling code shows how to use a boolean
   *  engine to disable the image rendering while dragging.
   *  \if_cpp
   *  \code
   *    SoRef<SoBoolOperation> engine = new SoBoolOperation;
   *    engine->a.connectFrom(&dragger->isActive);
   *    orthoslice->enableImage.connectFrom(&engine->inverse);
   *  \endcode
   *  \endif
   *  \if_dotnet
   *  \code
   *    SoBoolOperation engine = new SoBoolOperation();
   *    engine.a.ConnectFrom(dragger.isActive);
   *    orthoslice.enableImage.ConnectFrom(engine.inverse);
   *  \endcode
   *  \endif
   *  \if_java
   *  \code
   *    SoBoolOperation engine = new SoBoolOperation();
   *    engine.a.connectFrom(dragger.isActive);
   *    orthoslice.enableImage.connectFrom(engine.inverse);
   *  \endcode
   *  \endif
   *
   *  Another typical usecase is if you want to display the slice image in a 2D view and also show only the slice position in a 3D view (without the slice image).
   *
   *  @FIELD_SINCE_OIV 10.10
   */
  SoSFBool   enableImage;

  /**
   * Constructor.
   */
  SoOrthoSlice();

  /**
   * Overrides default method on SoShape. It must return TRUE if clip plane
   * is activated.  Used internally, especially for path traversal.  Probably
   * no reason for applications to ever call it.
   */
  virtual SbBool affectsState() const;

  /**
   * @copydoc SoVolumeShape::setRenderProgress
   */
  virtual void setRenderProgress(SoProgressIndicator* ps) override;

SoEXTENDER public:
  /**
   * Compute the bounding box
   */
  void computeBBox(SoAction*, SbBox3f& box, SbVec3f& center);

  /**
   * Implements SoWriteAction for this node.
   * Creates alternateRep subgraph if enabled.
   */
  virtual void write(SoWriteAction* action);

  /**
   * Returns an alternate represention for this node. @BR
   * This is typically a scene graph that gives approximately the same
   * appearance using only core Open Inventor nodes.  For example a
   * slice can be represented using a FaceSet and a Texture2 node.
   */
  virtual SoNode* getAlternateRep(SoAction* action);

  /** @copydoc SoNode::doAction */
  virtual void doAction(SoAction* action);
  /** @copydoc SoNode::GLRender */
  virtual void GLRender(SoGLRenderAction* action);
  /** @copydoc SoNode::callback */
  virtual void callback(SoCallbackAction* action);
  /** @copydoc SoNode::rayPick */
  virtual void rayPick(SoRayPickAction* action);
  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction* action);
  
SoINTERNAL public:

  //@TR enableBorder    @TD FALSE
  //@TR borderColor     @TD 0.84 0.43 0.02
  //@TR borderWidth     @TD 2.0
  //@TR renderSlice     @TD TRUE

  static void initClass();
  static void exitClass();

  /**
   * Handles field change
   */
  virtual void notify( SoNotList* list );

  virtual SoChildList* getChildren() const;

  /**
   * Sets back face culling.
   */
  void setBackFace( SoAction* action, bool backFace );

  /**
   * Returns true if the shape intersect given tile
   */
  virtual bool intersect( SoLDMNodeFrontManager* nfm, const SoLDMTileID& tileId );

  /**
   * Uses by dragger highlighting to disable border drawing
   */
  void preventBorderDrawing( const bool enable );

  /**
   * Used by dragger to set the Orthoslice pickable even if slice is not rendered
   */
  void setForcedPick( const bool enable );


protected:
  virtual void generatePrimitives( SoAction* action );
  virtual void doRendering( SoGLRenderAction* action );

  /**
   * Reads data into instance. Returns FALSE on error.
   * Deletes alternateRep subgraph unless requested not to.
   */
  virtual SbBool readInstance(SoInput* in, unsigned short flags);


SoINTERNAL protected :

  virtual void ldmAction( SoLdmValuationAction* action );

#ifndef HIDDEN_FROM_DOC
protected:
  /** Destructor */
  virtual ~SoOrthoSlice();
#endif // HIDDEN_FROM_DOC
};

#ifdef _WIN32
#pragma warning( pop )
#endif
#endif // _SO_ORTHO_SLICE_

