/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin GRANGE (Aug 2006)
** Modified    : David BEILLOIN (Mar 2011)
**=======================================================================*/
#ifndef  _SO_VOLUME_CLIPPING_GROUP_
#define  _SO_VOLUME_CLIPPING_GROUP_

#include <Inventor/nodes/SoGroup.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/SbBox.h>

class SoGLDepthPeeling;
class SoState;

/**
* @VREXT Clipping a volume with a 3D object.
*
* @ingroup VolumeVizNodes
*
* @DESCRIPTION
* VolumeViz supports clipping a volume against any closed 3D polygonal shape defined
* by standard Open Inventor geometry. This is done using the SoVolumeClippingGroup node.
* Polygon clipping can remove the voxels outside the shape or the voxels inside the shape.
* Polygon clipping can be used in seismic applications to clip outside a fault block or
* geobody, in medical applications to cut away an arbitrary region, in core sample
* applications to apply cylindrical clipping and many other uses.
*
* When this node is in the scene graph before a VolumeViz rendering node,
* all shapes under the SoVolumeClippingGroup will be used to clip the volume.
* These shapes will not be displayed.
*
* The geometry defined under SoVolumeClippingGroup must represent a set of closed 
* surfaces otherwise the result is unpredictable.
*
* The clipping group may include transform nodes, e.g. SoTransform, to scale, rotate
* and translate the geometry.  The transformed geometry will be used for clipping.
*
* In general, do not put draggers or manipulators, e.g. SoTabBoxDragger, in the
* clipping group.  The geometry of the dragger will also be used for clipping and
* the results may be unexpected.  One exception is if the dragger is configured to
* use the clipping shape as "proxy" geometry (see SoInteractionKit::setPartAsPath()).
* A better solution is to put the dragger in the scene graph outside the clipping
* group, add a transform node to the clipping group and connect the fields of the
* dragger to the fields of the transform node (see SoField::connectFrom()).
*
* The clipping result is based on the odd-winding rule, so the result is not simply
* the union or the intersection of the closed surfaces.
* If the goal is to define a clipping shape which is the result of the intersection/union of
* multiple closed surfaces, consider using the SoCSGShape node as a child of SoVolumeClippingGroup.
*
* In some cases the number of passes specified may not be enough to clip correctly with the
* specified geometry. The #numPasses field allows you to increase the clipping quality, for
* example when using shapes with a lot of concavity. To query the maximum allowed number of
* passes, which depends on your graphics card, use the static method
* \if_dotnet GetMaxNumPasses(). \else getMaxNumPasses(). \endif  The
* \if_dotnet SetNotEnoughPassCallback() \else setNotEnoughPassCallback() \endif
* method allows the application to be notified when numPasses is not sufficient.
*
* Standard clipplanes (see SoClipPlane) affect VolumeViz rendering nodes, but the VolumeViz
* specific clipping nodes generally provide better performance and allow much more complex
* clipping. Simple axis-aligned clipping can be done more efficiently using the SoROI node.
* The SoUniformGridClipping or SoUniformGridProjectionClipping nodes clip against height-field
* surfaces (e.g. seismic horizons).  The SoVolumeMask node clips against a boolean mask volume
* on a per-voxel basis.
*
* @B Notes:  @b 
* @UL 
* @LI When using a custom SoVolumeShader with this node and redefining the @I main() @i
* function, you must call @I VVizClipVoxel() @i in the @I main() @i function if you are writing a
* fragment shader. If you don't do this you will get a GLSL compilation error or clipping won't work.
*
* @LI Because this node uses special textures, all texture units between IVVR_FIRST_RESERVED_TEXTURE_UNIT
* (or SoFragmentShader::getMaxTextureImageUnit()-SoShaderProgram::getNumReservedTextures() if this
* variable is not set) and IVVR_FIRST_RESERVED_TEXTURE_UNIT-2 are reserved.
*
* @LI Because this node requires closed geometry, clipping planes (SoClipPlane) are automatically 
*     disabled when computing the clipping shape.
*
* @LI When used inside an SoOffscreenVolumeRender results might be incorrect.
*     In specific cases geometry might be seen as opened.
* @ul
*
* @EXAMPLE
* The following code clips a volume with a cone:
* \if_cpp
*   \code
*   SoVolumeClippingGroup* volClipGroup = new SoVolumeClippingGroup;
*     volClipGroup->addChild( new SoCone );
*
*   SoSeparator* volSep = new SoSeparator();
*     volSep->addChild( volumeData );
*     volSep->addChild( transferFunction );
*     volSep->addChild( volClipGroup );
*     volSep->addChild( volumeRender );
*   root->addChild( volSep );
*   \endcode
* \endif
* \if_dotnet
*   \code
*   SoVolumeClippingGroup volClipGroup = new SoVolumeClippingGroup();
*     volClipGroup.AddChild( new SoCone() );
*
*   SoSeparator volSep = new SoSeparator();
*     volSep.AddChild( volumeData );
*     volSep.AddChild( transferFunction );
*     volSep.AddChild( volClipGroup );
*     volSep.AddChild( volumeRender );
*   root.AddChild( volSep );
*   \endcode
* \endif
* \if_java
* \code
*   SoVolumeClippingGroup volClipGroup = new SoVolumeClippingGroup();
*     volClipGroup.addChild( new SoCone() );
*
*   SoSeparator volSep = new SoSeparator();
*     volSep.addChild( volumeData );
*     volSep.addChild( transferFunction );
*     volSep.addChild( volClipGroup );
*     volSep.addChild( volumeRender );
*   root.addChild( volSep );
* \endcode
* \endif
*
* @TABLE_1B
*      @TR Clipping object @TD Clipping applied to a volume
*      @TR @IMAGE volumeclipping_object.jpg
*                   @TD @IMAGE volumeclipping.jpg
* @TABLE_END
*
* @FILE_FORMAT_DEFAULT
*    VolumeClippingGroup {
*    @TABLE_FILE_FORMAT
*       @TR numPasses      @TD 2
*       @TR clipOutside    @TD TRUE
*    @TABLE_END
*    }
*
* @SEE_ALSO
*  SoVolumeRender,
*  SoUniformGridClipping,
*  SoShaderProgram,
*  SoVolumeRenderingQuality,
*  SoVolumeIsosurface,
*  SoCSGShape,
*  SoPreferences
*
*/
class VOLUMEVIZ_API SoVolumeClippingGroup : public SoGroup {
  SO_NODE_HEADER( SoVolumeClippingGroup );

 public:

  /**
   * Default constructor.
   */
  SoVolumeClippingGroup();

  /**
   * Number of passes used to do the clipping. Increasing this number increases the image
   * quality but decreases performance. Default is 2 and maximum is SoVolumeClippingGroup::getMaxNumPasses().
   */
  SoSFInt32 numPasses;

  /** 
   * If TRUE, voxels inside the clipping object will be drawn,
   * voxels outside will be clipped. Default is TRUE.
   * Note that if the VolumeClippingGroup is empty, setting clipOutside to TRUE
   * will clip everything and setting it to FALSE will clip nothing.
   */
  SoSFBool clipOutside;

  /**
   * Returns the maximum number of passes supported by the hardware.
   */
  static unsigned int getMaxNumPasses();

  /** Callback prototype */
  typedef void SoVolumeClippingGroupCB( SoVolumeClippingGroup* mgr, void* userData );

  /** 
   * Set a callback to be called when there are not enough passes (see #numPasses) 
   * to properly apply the clipping.
   */
  void setNotEnoughPassCallback( SoVolumeClippingGroupCB* f, void* userData );

 SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction* action);
  virtual void doAction(SoAction* action);
  virtual void getBoundingBox(SoGetBoundingBoxAction* action);

  /** @copydoc SoGroup::rayPick */
  virtual void rayPick(SoRayPickAction* action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  //Get the texture unit number where the first depth layer is binded
  int getFirstDepthTextureUnit( SoGLRenderAction* action ) const;

  //Return the number of depth textures pairs
  int getNumPairs() const;

  //Return the real number of depth layer passes used
  int getNumPassUsed() const;

  //TRUE if box is not into the clipping volume
  SbBool isCulled(const SbBox3f& box);

  // This is used by SoVolumeGroupElement just to ensure that
  // the resources won't be allocated-computed-release for each path
  // collected by the SoVolumeGroup node.
  //
  // This method is called by SoVolumeClippingGroup itself. The SoVolumeGroupElement calls the unlock
  // in order to allow releaseResources to do the job.
  void lockResources();

  // Release previously locked resources.
  void unlockResources();

  // Indicates if the resources are locked or not.
  bool areResourcesLocked() const;

  // Release the depth peeling resources if they are not locked.
  // Most of the time it is called by SoVolumeClippingGroupElement, except when we do multi volume 
  // the release is done by SoVolumeGroupElement.
  void releaseResources();

  // Set depth texture in SoVolumeRender
  void setDepthPeelingTextureOnState(SoGLRenderAction* action);

  /** Return BBox of VolumeClippingGroup expressed in world box. This box is computed when VolumeClippingGroup is traversed
   * by a action. */
  inline const SbXfBox3f& getWorldBbox() const { return m_worldBbox; }

  /** Handle field change*/
  virtual void notify(SoNotList* list);

protected:
  virtual ~SoVolumeClippingGroup();

private:

  enum LayersState
  {
    NOT_COMPUTED,
    COMPUTED,
  };

  LayersState m_layersState;

  bool m_isLocked;

  /** struct given to notEnoughLayerCB. */
  struct NotEnoughLayerCBData {
    NotEnoughLayerCBData() : m_notEnoughPassCB(NULL), m_notEnoughPassCBData(NULL), that(NULL) {}
    /** callback specified in setNotEnoughPassCallback */
    SoVolumeClippingGroupCB* m_notEnoughPassCB;
    /** Data specified in setNotEnoughPassCallback */
    void* m_notEnoughPassCBData;
    /** this SoVolumeClippingGroup */
    SoVolumeClippingGroup* that;
  };
  /** userData given to notEnoughLayerCB. */
  NotEnoughLayerCBData m_notEnoughLayerCBData;

  /** Called when GLDepthPeeling detect that their are not enough layers 
   * (see SoGLDepthPeeling::setNotEnoughLayerCallback). This method just call
   * the m_notEnoughPassCB specified in setNotEnoughPassCallback.
   * We have to create a binding because SoGLDepthPeelingCB is internal and 
   * may be modified, whereas SoVolumeClippingGroupCB is public.
   * userData is a NotEnoughLayerCB struct containing m_notEnoughPassCB
   * and m_notEnoughPassCBData.
   * Just see notEnoughLayerCB definition to understand...
   */
  static void notEnoughLayerCB(void* userData, SoGLDepthPeeling* mgr);

  void commonInit();

  SbBool initDepthPeeling(SoGLRenderAction* action, const SbVec2s& vpSize);

  SoGLDepthPeeling* m_glDepthPeeling;

  // update bbox of the clipping volume
  void updateChildBBox();

  // get bbox of the clipping volume
  const SbXfBox3f& getBBox(SoAction*);

  bool m_applyTochild;
  SbXfBox3f m_bbox;
  bool m_bboxCacheClean;

  SbXfBox3f m_worldBbox;

  struct SoVolumeClippingInfos {
    SoVolumeClippingGroup* cg;
    SoGLRenderAction* ra;
  } m_clippingInfos;
  static void renderCB(void* userData, SoGLDepthPeeling* );

  /** true during the depthPeelingPass of the renderCB. In this case, SoGroup::GLRender 
  * must be called intead of VolumeClippingGroup::GLRender */
  SbBool m_depthPeelingPass;

  SbBool m_releaseResources;
};
/*----------------------------------------------------------------------------*/
#endif


