/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thierry Dufour (Aug 2002)
** Modified by : Jerome Hummel (MMM yyyy)
** Modified by : David Beilloin (Apr 2011)
**=======================================================================*/
#ifndef __VOLUME_FACE_SET
#define __VOLUME_FACE_SET

#include <Inventor/nodes/SoFaceSet.h>

#include <VolumeViz/nodes/SoVolumeGeometry.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/SbBox.h>

class SoVolumeGeometryVbo;

/** 
 * @VREXT Polygonal face shape node for volume data.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 *   This node defines a shape which is the intersection of the volume data
 *   defined by an SoVolumeData node and the 3D shape formed by constructing
 *   faces (polygons) from vertices located at the coordinates specified in
 *   the #vertexProperty field (from SoVertexShape) or the current inherited
 *   state. For optimal performance, use of the #vertexProperty field is recommended.
 *
 *   The shape defined by this intersection is textured using the data
 *   from the current SoVolumeData node and SoTransferFunction node.
 *   The #interpolation field controls how the texture is interpolated.
 *   Texture coordinates are automatically computed for each vertex based on its
 *   position relative to the 3D extent of the volume.  If texture coordinates are
 *   specified, they are ignored.
 *
 *   This node can be used to create custom "slices" through the volume, for example
 *   a cylinder slice.  It can be considered a generalization of the SoOrthoSlice,
 *   SoObliqueSlice, SoFenceSlice, etc features. But note:
 *   - Compared to an equivalent standard slice (e.g. SoOrthoSlice), the rendering may
 *     be slightly different due to positioning and interpolation.
 *
 *   - The standard slices (e.g. SoOrthoSlice) have "two-sided lighting" enabled by
 *     default as a convenience.  This is not true for volume geometry nodes (which
 *     have the same default as their corresponding standard geometry node), so the
 *     "back" side of the geometry will not be lighted unless you enable two-sided
 *     lighting using an SoShapeHints node.
 *
 *   When set to FALSE, the #clipGeometry field allows rendering of the
 *   portion of the geometry that does not intersect the volume data. This
 *   portion is not textured and is rendered as a regular SoFaceSet.
 *
 *   This node uses 3D texturing and is not available if the hardware does not
 *   support this feature.
 *
 *   SoVolumeFaceSet uses the coordinates in order, starting with the first one.
 *   Each face has a number of vertices specified by a value in the #numVertices
 *   field. For example, an SoVolumeFaceSet with #numVertices of [3,4,4] would use
 *   coordinates 1, 2, and 3 for the first face, coordinates 4, 5, 6, and 7 for the
 *   second face, and coordinates 8, 9, 10, and 11 for the third. For improved
 *   performance, arrange all the faces with only 3 vertices at the beginning of the
 *   list, then all faces with 4 vertices, and finally all other faces.
 *
 *   The number of values in the #numVertices field indicates the number of
 *   faces in the set.
 *
 *   The coordinates of the face set are transformed by the current cumulative
 *   transformation. The faces are drawn with the current light model and drawing
 *   style.
 *
 *   Treatment of the current material and normal binding is as follows: The PER_PART
 *   and PER_FACE bindings specify a material or normal for each face. The _INDEXED
 *   bindings are equivalent to their non-indexed counterparts. The default material
 *   binding is OVERALL. The default normal binding is PER_VERTEX.
 *
 *   If any normals (or materials) are specified, Open Inventor assumes you have provided
 *   the correct number of them, as indicated by the binding. You will see unexpected
 *   results if you specify fewer normals (or materials) than the shape requires. If
 *   no normals are specified, they will be generated automatically.
 *
 *   @B Notes: @b
 *   @UL
 *    @LI A face set may render or pick incorrectly if any of its polygons are
 *        self-intersecting or non-planar.
 *    @LI The volume size and orientation (like geometry) can be modified by
 *        transformation nodes in the scene graph and this in turn modifies the
 *        appearance of volume rendering nodes like SoVolumeFaceSet.
 *        However the same transformation must be applied to the volume data node
 *        and all volume rendering nodes associated with that volume. So effectively
 *        any transformation nodes that affect the volume must be placed @B before @b
 *        the volume data node.
 *    @LI For performance concerns, an instance of SoVolumeFaceSet should not be used
 *        multiple times in the scene graph.
 *        In order to have the best trade-off between performances and memory footprint
 *        use a single SoVertexProperty node shared by several instances of SoVolumeFaceSet.
 *    @LI Composition with Multiple Data: @BR
 *        It is possible to compose datasets that have different dimensions,
 *        tile sizes and transformations. @BR
 *        In order to help fetch the correct data values in custom shaders,
 *        texture coordinates conversion functions are provided in the
 *        @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 *        For instance,
 *        \code
 *        vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 *        \endcode
 *        can be used to convert texture coordinates related to one dataset to
 *        texture coordinates related to another dataset. @BR
 *        The conversion is based solely on the transformations applied to each
 *        dataset, which are defined by their model matrix and their extent. @BR
 *        Please note that the model matrix of a dataset is defined by to the
 *        SoTransformation nodes that are placed @B before @b the SoDataSet node in
 *        the order of the traversal.
 *   @ul
 *
 * @FILE_FORMAT_DEFAULT
 *    VolumeFaceSet {
 *    @TABLE_FILE_FORMAT
 *       @TR vertexProperty   @TD NULL
 *       @TR startIndex       @TD 0
 *       @TR numVertices      @TD -1
 *       @TR clipGeometry     @TD TRUE
 *       @TR offset           @TD 0
 *       @TR interpolation    @TD LINEAR
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    @B SoGLRenderAction @b @BR
 *        Draws faces based on the current coordinates, normals, materials, drawing
 *        style, and so on.
 *
 *    @B SoRayPickAction @b @BR
 *        Picks faces based on the current coordinates and transformation. Details about
 *        the intersection are returned in an SoFaceDetail.
 *
 *    @B SoGetBoundingBoxAction @b @BR
 *        Computes the bounding box that encloses all vertices of the face set with the
 *        current transformation applied to them. Sets the center to the average of the
 *        coordinates of all vertices.
 *
 *    @B SoCallbackAction @b @BR
 *        If any triangle callbacks are registered with the action, they will be invoked
 *        for each successive triangle generated from each face in the set.
 *
 *
 * @SEE_ALSO
 *    SoVolumeData,
 *    SoTransferFunction,
 *    SoROI,
 *    SoCoordinate3,
 *    SoDrawStyle,
 *    SoVolumeIndexedFaceSet,
 *    SoVolumeIndexedTriangleStripSet,
 *    SoVolumeTriangleStripSet,
 *    SoFaceDetail,
 *    SoVertexProperty
 *
 *
 */
class VOLUMEVIZ_API SoVolumeFaceSet : public SoFaceSet, public SoVolumeGeometry
{
  SO_NODE_HEADER(SoVolumeFaceSet);

public :

  /** Constructor. */
  SoVolumeFaceSet();

  /**
   * Specifies to clip the geometry at the volume boundaries. @BR
   * The SoROI node also affects the clipping process. Default is TRUE.
   */
  SoSFBool clipGeometry;

  /**
   * Sets an offset value used for computing the texture coordinates.  @BR
   * The texture coordinate for each vertex will be taken from a point @B offset @b
   * units away from the vertex.
   * The offset is measured in the direction of the vertex normal.
   * By default the offset is equal to 0 (no offset). When using this feature,
   * a vertex normal must exist for every vertex. If vertex normals are not
   * supplied, Open Inventor will compute vertex normals in the usual way except
   * that the crease angle will be fixed at PI in order to assure that a vertex
   * normal is computed at every vertex.
   */
  SoSFFloat offset;

  /**
   * @copydoc SoVolumeShape::interpolation
   */
  SoSFEnum interpolation;

#if SoDEPRECATED_BEGIN(9000)

  enum SoDEPRECATED_ENUM(9000,"Use the enumeration SoVolumeShape::Interpolation.")
  Interpolation {
    /** @copydoc SoVolumeShape::NEAREST */
    NEAREST = SoVolumeShape::NEAREST,
    /** @copydoc SoVolumeShape::LINEAR */
    LINEAR = SoVolumeShape::LINEAR
  };

#endif /** @DEPRECATED_END */


SoEXTENDER public:
  /** @copydoc SoNode::initClass() */
  static void initClass();

  /** @copydoc SoNode::exitClass() */
  static void exitClass();

  /** @copydoc SoFaceSet::GLRender() */
  virtual void GLRender(SoGLRenderAction *action);

  /** @copydoc SoFaceSet::computeBBox() */
  virtual void computeBBox(SoAction *action, SbBox3f &box, SbVec3f &center);

  /** @copydoc SoFaceSet::computeBBox() */
  virtual void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);

SoINTERNAL public:

  /** @copydoc SoFaceSet::notify() */
  virtual void notify(SoNotList *list);

#if SoDEPRECATED_BEGIN(9000)

  /** Return interpolation value according to the SoInteractiveComplexityElement */
  SoDEPRECATED_METHOD(9000,"SoVolumeShape::Interpolation not supported anymore.")
  virtual SoVolumeShape::Interpolation getInteractiveInterpolationValue(SoState* state) const;

#endif /** @DEPRECATED_END */

protected:
  /** @copydoc SoVolumeRender::shapeRender */
  virtual void shapeRender(SoGLRenderAction* action);

  /** @copydoc SoVolumeRender::shapeGeneratePrimitives */
  virtual void shapeGeneratePrimitives(SoAction* action);

  /** @copydoc SoFaceSet::checkShapeStyle */
  virtual void checkShapeStyle(SoState *state,const SoShapeStyleElement *shapeStyle);

  /** @copydoc SoFaceSet::GLRenderBoundingBox */
  virtual void GLRenderBoundingBox(SoGLRenderAction* action);

  /** @copydoc SoVolumeRender::getBBox */
  virtual void getBBox(SoAction *action, SbBox3f &box, SbVec3f &center);

  /** @copydoc SoVolumeRender::getBBox */
  virtual void getBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);

  /** @copydoc SoVolumeGeometry::getNumVertices */
  virtual int getNumVertices(SoState* state);

  /** @copydoc SoVolumeGeometry::getIndexCoord */
  virtual int getIndexCoord(int index) const;

  /** Destructor. */
  virtual ~SoVolumeFaceSet();

  using SoVolumeGeometry::doRendering;

private:

  SoDEPRECATED_FIELD(8500,"Use the SoShapeHints::useVBO field to enable or disable use of Vextex Buffer Objects for rendering.")
  SoSFBool useVbo;

};

#endif //__VOLUME_FACE_SET


