/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin GRANGE (Aug 2005)
**=======================================================================*/


#ifndef _SO_ISOSURFACE_H_
#define _SO_ISOSURFACE_H_


#include <Inventor/SbBasic.h>
#include <Inventor/caches/SoNodeDependencies.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoSubNode.h>
#include <Inventor/fields/SoMFNode.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeShader.h>


/*----------------------------------------------------------------------------*/
class SoMaterial;
class SoDataRange;
class SoIsosurfaceTexture;
class SoMaterial;
class SoVolumeDataDrawStyle;

/**
 *
 * @VREXT Isosurface property node
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 * This property node causes subsequent SoVolumeRender nodes to draw isosurfaces instead of
 * classical volume rendering of the volume data defined by the current SoVolumeData.
 *
 * This is functionally equivalent to using an SoVolumeDataDrawStyle node with the style
 * field set to ISOSURFACE, but is much simpler when you only need to display an iso surface.
 *
 * The isosurfaces are only rendered, not geometrically computed.  So it is not possible to
 * directly get the geometry corresponding to the rendered isosurface. (The MeshViz
 * extension includes algorithms to extract an isosurface as geometry, see MoMeshIsosurface.)
 *
 * The isovalues to display are specified in the #isovalues field.
 *
 * You may not use both SoVolumeShader and SoVolumeIsosurface with the same
 * SoVolumeRender node. However since Open Inventor 7.1 it is possible to use
 * both SoVolumeRenderingQuality and SoVolumeIsosurface with SoVolumeRender.
 *
 * Remember this is a shader node. The effect will usually be undesirable if it is
 * applied to standard geometry (polygons, lines, etc).  Therefore applications
 * should generally keep the volume visualization nodes and standard geometry
 * nodes separate in the scene graph (i.e. under different SoSeparator nodes).
 *
 * The multiple fields SoMaterial::diffuseColor, SoMaterial::transparency, SoMaterial::specularColor, and SoMaterial::shininess of the current
 * SoMaterial specify the material of each isosurface. All isosurfaces share the first
 * SoMaterial::specularColor and SoMaterial::shininess.
 * The first material will be for the first isosurface, the second material for the second
 * isosurface, and so on.
 *
 * Surfaces are lighted with the first directional light found in the scene graph.
 * No other lights affect the isosurface.
 * Note: The isosurfaces are @I always@i lighted, regardless of the value of the
 * SoVolumeRenderingQuality lighting and deferredLighting fields.
 *
 * The isosurface displays the boundary between 2 different values, similar to the marching
 * cubes algorithms. In other words, a voxel is part of the isosurface if the ray entry and
 * exit values for the voxel are different and include the specified isosurface value in
 * their range. If you render, for example, the isosurface 100 of a volume data containing 
 * the value 100 everywhere, nothing will be rendered.
 *
 * Because of linear interpolation, basic isosurface may not render correctly on segmented dataset like label field. In this case, you should use
 * the #SoVolumeRenderingQuality::segmentedInterpolation or #SoVolumeRenderingQuality::colorInterpolation fields.
 * Default Linear interpolation may also generate rough result even for scalar datasets. For
 * a smoother surface, you may have to use cubic interpolation. Please refer to
 * #SoVolumeShape::interpolation for details.
 *
 * Because this node is derived from SoVolumeShader, IVVR_FIRST_RESERVED_TEXTURE_UNIT applies
 * to it. See SoVolumeShader for more information.
 *
 *     Note: The volume size and orientation (like geometry) can be modified by
 *     transformation nodes in the scene graph and this in turn modifies the
 *     appearance of volume rendering nodes like SoVolumeIsosurface.
 *     However the same transformation must be applied to the volume data node
 *     and all volume rendering nodes associated with that volume.  So effectively
 *     any transformation nodes that affect the volume must be placed @B before @b
 *     the volume data node.
 *
 * @FILE_FORMAT_DEFAULT
 *    VolumeIsosurface {
 *    @TABLE_FILE_FORMAT
 *       @TR isovalues          @TD []
 *    @TABLE_END
 *    }
 * @ACTION_BEHAVIOR
 * SoGLRenderAction @BR
 * Sets volume isosurface parameters in the traversal state.
 *
 * @SEE_ALSO
 *  SoVolumeRender,
 *  SoVolumeShader,
 *  SoVolumeRenderingQuality,
 *  SoPreferences
 *
 *
 */
class VOLUMEVIZ_API SoVolumeIsosurface : public SoVolumeShader {

  SO_NODE_HEADER(SoVolumeIsosurface);

public:
  /**
   * Constructor
   */
  SoVolumeIsosurface();

  /**
   * Specifies the list of isovalues to display
   */
  SoMFFloat isovalues;

#ifndef HIDDEN_FROM_DOC
SoINTERNAL public:
  virtual void notify(SoNotList* list);

  virtual void GLRender(SoGLRenderAction *action);

  // Initializes the classes.
  static void initClass() ;
  static void exitClass() ;

  /** Install textures needed for shaders */
  virtual void installTextures(SoGLRenderAction* action);

  /** Return true if the fake isosurface drawstyle is on state */
  bool isFakeDrawStyleOnState(SoState* state) const;

  /** Reserve needed texture units */
  virtual void allocateTextureUnit(SoGLRenderAction *action) const;

SoEXTENDER public:
  virtual void rayPick(SoRayPickAction *action);

protected:
  virtual ~SoVolumeIsosurface();

  /** Create a shader suitable for shadowmap generation pass */
  virtual SoShaderProgram* generateShadowShader() const;

  /** 
  * Return the fragment filename according to state 
  * Can be the main or the FRAGMENT_COMPUTE_COLOR in case of
  * raytracing
  */
  virtual SbString getRenderingModeFragmentName(SoState* state);

#endif // HIDDEN_FROM_DOC

private:
  /**
   * Set needed rendering modes
   */
  void handleVolumeRenderingQuality(SoState* state);

  /** Update isovalues of the shader used for shadowmap pass */
  void updateShadowShaderIsoValue();

  /** Proxy draw style used to assign a iso style during ray pick action*/
  SoVolumeDataDrawStyle* m_volumeDataDrawStyle;

  bool m_needUpdate;
};
/*----------------------------------------------------------------------------*/

#endif /*_SO_ISOSURFACE_H_*/


