/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef SOVOLUMEMASK_H
#define SOVOLUMEMASK_H

#include <VolumeViz/nodes/SoVolumeData.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/fields/SoSFVec4f.h>
#include <Inventor/fields/SoMFVec4f.h>


class SoVolumeMaskReader;
class SoCpuBufferUniform;
class SoCpuBufferBasicProperty;

SO_PIMPL_PUBLIC_DECLARATION(SoVolumeMask)

/**
* @VREXT Volume mask.
*
* @ingroup VolumeVizNodes
*
* @DESCRIPTION
*   This node specifies an arbitrary voxel-based region of a volume data set
*   (SoVolumeData). A volume mask is itself a data volume consisting of binary 
*   values. Zero values are "false" and non-zero values are "true".
*   A large number of masks may be defined simultaneously, using an
*   SoVolumeMask node for each one, and mask regions may overlap if desired.
*
*   SoVolumeMask is derived from SoVolumeData.  Like SoVolumeData it accesses
*   its data through an SoVolumeReader class.  I.e. the data may be in a file
*   or in memory (or any data source using a custom volume reader). The mask 
*   volume can be any data type, but unsigned byte is currently the most 
*   memory efficient representation.
*
*   SoVolumeMask nodes must follow these rules:
*   - Each mask volume must have exactly the same voxel dimensions and tile
*     size value as the data volume it is applied to.
*   - All SoVolumeMask nodes must be children of an SoVolumeMaskGroup node.
*   - The SoVolumeData node and the SoVolumeMaskGroup nodes must be children
*     of an SoMultiDataSeparator node.
*   - The SoVolumeData and associated SoVolumeMask nodes must each have a
*     different value assigned to their dataSetId field (see SoDataSet).
*     Note that SoVolumeData and SoVolumeMask both have a default value of
*     1 for the dataSetId field, so some ids must be explicitly set.
*   - Currently SoVolumeMask nodes only affect volume rendering (SoVolumeRender).
*     They do not affect slice rendering or other VolumeViz primitives.
*
*   SoVolumeMask can be used to clip the data volume on a per-voxel basis.  
*   But the mask mechanism is much more powerful than that.  Each region can 
*   have its own transfer function (color map) using SoTransferFunction nodes.
*   Each region can also have its own draw style (volume rendering, isosurface
*   or boundary) using SoVolumeDataDrawStyle nodes.  Each region, including the
*   original unmasked volume, is only visible if there exists a transfer 
*   function (SoTransferFunction) with the same id value.
*
*   Mask regions:
*
*   A single SoVolumeMask node effectively defines two regions:
*   - False voxels (where the mask voxel = 0), and
*   - True voxels (where is mask voxel != 0).
*
*   Because intersecting regions are allowed, two SoVolumeMask nodes define 
*   (up to) four regions:
*   - False (where all masks are 0),
*   - True only in mask 1,
*   - True only in mask 2, and
*   - True in more than one mask (intersection region).
*
*   Each additional mask defines one additional region (voxels that are true
*   only in that mask).
*
*   In addition, if there exists a transfer function with the same id as the
*   original volume (SoVolumeData node), the color value from that transfer
*   function is combined with the color value from the mask region's transfer
*   function.
*
*   Transfer functions:
*
*   A separate transfer function can be specified for each of these regions 
*   using SoTransferFunction nodes. The transfer function nodes may be placed
*   under the SoVolumeMaskGroup node, but this is not required.
*
*   NOTE: It is not currently possible to specify a different data range for each mask.
*
*   - A transfer function is associated with a specific mask (i.e. with the voxels
*     that are true only in that mask), by setting the SoTransferFunction
*     node's @I transferFunctionId @i equal to the mask's @I dataSetId @i.
*     If there is no transfer function with the same id as the mask, then
*     that mask is ignored.
*
*   - A transfer function is associated with all "false" voxels (voxels that
*     do not belong to any mask) by setting its @I transferFunctionId @i equal to zero.
*     If there is no transfer function with id = 0, then "false" voxels are
*     not rendered.
*
*   - A transfer function is associated with all "intersection" voxels
*     (voxels that belong to more than one mask) by setting its @I transferFunctionId @i
*     equal to SoVolumeMaskGroup::TRANSFERFUNCTION_INTERSECTION_ID.
*     If there is no transfer function with this id value, then intersection
*     voxels (if any) are rendered using the default GRAY color map.
*
*   Draw Styles:
*
*   A separate draw style can be specified for each mask region and for the
*   intersection voxels (but not for false voxels) using SoVolumeDataDrawStyle
*   nodes:
*
*   - Unlike transfer functions, a draw style is associated with a mask (voxels
*     that are true only in that mask) by traversal order.  In other words, each
*     mask uses the draw style that is current when the mask is traversed.  So 
*     you need to put SoVolumeDataDrawStyle nodes under the VolumeMaskGroup, 
*     alternating draw style, mask, draw style, mask.
*
*   - A draw style is associated with all "intersection" voxels (voxels that
*     belong to more than one mask) by placing it before the SoVolumeMaskGroup
*     node in the scene graph.
*
*   Data Range:
*
*   SoVolumeMask currently does not support separate data ranges for each
*   mask region. The same SoDataRange will be applied on all regions.
*
* Note that the inherited undefinedValue field is ignored. Setting it has no effect.
*
* @FILE_FORMAT_DEFAULT
*    VolumeMask {
*    @TABLE_FILE_FORMAT
*       @TR allocateResourceOnRender   @TD FALSE
*       @TR data                       @TD NODATA 0 0 0 UBYTE 8
*       @TR dataRGBA                   @TD FALSE
*       @TR dataSetId                  @TD 1
*       @TR dataTransform              @TD NULL
*       @TR extent                     @TD -1 -1 -1 1 1 1
*       @TR fileName                   @TD ""
*       @TR texturePrecision           @TD 0
*       @TR undefinedValue             @TD NaN
*       @TR useCompressedTexture       @TD TRUE
*       @TR useExtendedData            @TD FALSE
*       @TR usePalettedTexture         @TD TRUE
*       @TR useSharedPalettedTexture   @TD TRUE
*    @TABLE_END
*    }
*
* @SEE_ALSO
*    SoVolumeDataDrawStyle,
*    SoVolumeRender,
*    SoOrthoSlice,
*    SoObliqueSlice,
*    SoVolumeReader,
*    SoVolumeSkin,
*    SoVolumeMask,
*    SoVolumeMaskGroup,
*    SoMultiDataSeparator
*
*
*/
class VOLUMEVIZ_API SoVolumeMask : public SoVolumeData
{
  SO_NODE_HEADER(SoVolumeMask);

  SO_PIMPL_PUBLIC_HEADER( SoVolumeMask );

public:
  /**
   * Constructor.
   */
  SoVolumeMask();

  /** 
   * Replace the contents of a subvolume with the given data.
   * Returns 0 if successful.
   *
   * - The subvolume is specified in voxel/cell coordinates.
   * - The buffer size (in bytes) must match the subvolume size (in bytes) exactly.
   * - Only an SoVolumeMask with data size of 1 (ie. byte or unsigned byte) can be edited. 
   *   This implies that dataType of userData must be byte or unsigned byte.
   * - @B warning @b: userData must be an SoCpuBufferBitSet.
   *
   * Call #startEditing() before calling this method. 
   */
  virtual int editSubVolume( const SbBox3i32& subVolume, SoBufferObject* userData );

 /**
  * Replace the contents of a subvolume with the specified value.
  * Returns 0 if successful.
  *
  * - The subvolume is specified in voxel/cell coordinates.
  * - Only an SoVolumeMask with data size of 1 (ie. byte or unsigned byte) can be edited. 
  * - All  zero values are set to zero and all non-zero values are set to 1.
  *
  * Call #startEditing() before calling this method. 
  */
  virtual int editSubVolume( const SbBox3i32& subVolume, const double& value );

 /** 
  * Replace all voxels intersecting the polygons or lines defined by the surfaceShape
  * and given thickness with the specified value. Returns 0 if successful.
  *
  * - @I surfaceShape @i is defined in the same 3D space as the dataSet.
  * - @I thickness @i is defined in voxels. The surfaceShape can be deleted after calling this function.
  * - Only an SoVolumeMask with data size of 1 (ie. byte or unsigned byte) can be edited. 
  * - All zero values are set to zero and all non-zero values are set to 1.
  *
  * Call #startEditing() before calling this method. 
  */
  virtual int editSurfaceShape( const SoNode* surfaceShape, const float& thickness, const double& newValue );

 /** 
  * Replace all voxels inside the given shape with the specified value.
  * Returns 0 if successful.
  *
  * The geometry defined under @I solidShape @i must represent a list of closed 
  * surfaces otherwise the result is unpredictable. solidShape must not
  * contain any lines or open polyhedrons (polyhedron with shapes). 
  * The result is based on the odd-winding rule, so the result is not necessarily
  * the union or the intersection of the closed surfaces.
  * If you want to voxelize lines or single polygons, see also the #editSurfaceShape method.
  *
  * If the goal is to define a shape which is the results of the intersection/union of
  * multiple closed surfaces, see the SoCSGShape node.
  *
  * - solidShape is defined in the same 3D space as the dataSet. The solidShape can
  *   be deleted after calling this function.
  * - Only VolumeMask with data size of 1 (ie. byte or unsigned byte) can be edited. 
  * - All zero values are set to zero and all non-zero values are set to 1.
  *
  * Call #startEditing() before calling this method. 
  */
  virtual int editSolidShape( const SoNode* solidShape, const double& value );

  /**
  * Save editing done on the volume mask data.
  * Returns true if save was successful.
  */
  bool saveEditing(SbString filename, bool recomputeLowerResolution = TRUE, const std::vector<char*> = std::vector<char*>(), SaveEditingCB callback = NULL);

  /**
  * Set default value of the volume mask.  Default is 0.
  */
  void setDefaultValue( const bool defValue );

  /**
  * Returns default value of the volume mask.
  */
  bool getDefaultValue() const;

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  virtual SoLDMReader* getAppropriateLDMReader(const SbString& pathname);
  /** Generate the final texture mask */
  virtual SoBufferObject* getTransformedTile(const SoLDMTileID& tile, const SoLDM::DataSetIdPair& pair,
                                             SoState * state = NULL, const bool useExtendedData=false);

SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction* action);

  inline virtual void setIsReadChar(bool flag);

  // Returns SoVolumeData node that this mask applies to
  virtual SoVolumeData* getVolumeData();

  // Note: Since OIV 9.0, the border parameter is no longer used.
  virtual void setProperties( SbVec3i32 dimension, SbVec3i32 tileSize, int border );

protected:
 ~SoVolumeMask();
 void readerChanged();

};

void
SoVolumeMask::setIsReadChar(bool flag)
{
  m_dataCharRead = flag;
}

#endif


