/**************************************************************************************
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **************************************************************************************/

#pragma once

#include <Inventor/SbPImpl.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoSubNode.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>

class SoGLRenderAction;
class SoAction;
class SoNotList;
class SoVRImageSpaceEffects;

SO_PIMPL_PUBLIC_DECLARATION(SoVolumeRenderingPhysicalQuality)

/**
 @VREXT Volume rendering physical quality property node

 @ingroup VolumeVizNodes

 @DESCRIPTION
 This node is intended to be used when one wants to achieve "out of the box" high quality
 volume rendering. It works the same way as a SoVolumeRenderingQuality node, in the sense
 that it causes subsequent SoVolumeRender nodes to be drawn with different rendering
 effects and/or levels of quality, the difference being that the SoVolumeRenderingQuality
 fields are not exposed in this node and are internally pre-selected to provide the highest
 possible quality:

 The SoVolumeRenderingQuality fields that are internally activated are:
   - SoVolumeRenderingQuality::preIntegrated = TRUE
   - SoVolumeRenderingQuality::ambientOcclusion = TRUE
   - SoVolumeRenderingQuality::deferredLighting = TRUE

 The presence of this node in the scene graph also forces subsequent SoVolumeRender nodes
 to be drawn with SoVolumeRender::samplingAlignment set to SoVolumeRender::BOUNDARY_ALIGNED.

 Note that, unlike SoVolumeRenderingQuality, this node does not support customization of
 VolumeViz shaders.

 The fields exposed in this node provide control over effects that are not present
 in the SoVolumeRenderingQuality node. These effects include:
    - Physically Based Lighting
    - Ray-Traced Shadows
    - Depth of Field

# Physically Based Lighting

 The Physically Based Lighting effect provides lighting using an advanced BRDF model.
 It can be seen as an enhanced version of the regular deferred lighting.
 In particular, it shows the same behavior regarding the current SoPhysicalMaterial or SoMaterial on the state and has the
 same limitations. See SoVolumeRenderingQuality::deferredLighting for a detailed list of limitations.
 In addition to regular deferred lighting, this effect supports HDR environment mapping and several
 cubemap textures are provided (see #environmentMap).
 The field #toneMapping is provided to select the Tone Mapping algorithm used to remap HDR values to
 low range values. A predefined material that will be applied to the shape can also be selected using
 the field #predefinedMaterial. Any material (SoPhysicalMaterial or SoMaterial) set between this node and the SoVolumeRender node in the
 scenegraph will override this predefined material. Note that, just like with regular deferred lighting,
 the material properties will affect the lighting, and the @ref SoLightModel-base-color-def "base color" will be mixed with
 the voxel colors using a component-wise multiplication. Finally, the field #shadingStyle allows to
 switch between this type of lighting and regular deferred lighting.

# Ray-Traced Shadows
 The Ray-Traced Shadows effect computes shadows inside the volume using a ray-marching based algorithm.
 This effect is activated and parameterized just like regular shadows using the SoShadowGroup node.
 The SoShadowGroup fields taken into account for this effect are SoShadowGroup#isActive,
 SoShadowGroup#intensity and SoShadowGroup#quality.
 This type of shadows supports SoROI clipping, but does not support SoClipPlane, SoVolumeClippingGroup,
 SoUniformGridClipping or SoUniformGridProjectionClipping.
 Note that this effect has a huge impact on performance.
 The field #shadowsStyle allows to switch between this type of shadows and regular shadows.

# Depth of Field
 Depth Of Field is a "cinematic" effect that adds blur on regions far from the focal point of the camera.
 The focal point is typically the center of the scene bounding box, causing regions close to the camera
 and far from the camera to be blurred.
 The focal point is specified by the field SoCamera::focalDistance of the current camera.
 Only SoVolumeRender nodes are affected by the blurring.
 The field #enableDepthOfField is used to switch it on and off, while the #blurFactor field
 controls the amount of blur induced by this effect.

 @FILE_FORMAT_DEFAULT
   VolumeRenderingPhysicalQuality {
   @TABLE_FILE_FORMAT
     @TR #shadowsStyle         @TD #RAYTRACED
     @TR #shadingStyle         @TD #PHYSICALLY_BASED
     @TR #predefinedMaterial   @TD #SEMI_GLOSSY
     @TR #environmentMap       @TD #UFFIZI
     @TR #toneMapping          @TD #MEDIUM
     @TR #enableDepthOfField   @TD TRUE
     @TR #blurFactor           @TD 0.01
   @TABLE_END
   }

 @SEE_ALSO
  SoVolumeRenderingQuality,
  SoShadowGroup

 @NODE_SINCE_OIV 9.8
*/
class SoDEPRECATED_CLASS(2025100, "Use SoPhysicalMaterial, SoEnvironmentMap, SoCamera::exposureMode and SoCamera::blur instead.") VOLUMEVIZ_API SoVolumeRenderingPhysicalQuality : public SoNode
{
  SO_NODE_HEADER(SoVolumeRenderingPhysicalQuality);
  SO_PIMPL_PUBLIC_HEADER(SoVolumeRenderingPhysicalQuality);

public:
  SoVolumeRenderingPhysicalQuality();

  // Shadows Fields

  /**
   * Specifies which kind of shadows to compute for the volume.
   * This field allows to enable RayTraced shadows, which have a better quality
   * than the standard shadowmap-based shadows, but take longer to compute.
   * This field is taken into account only if the associated SoVolumeRender node is
   * inside a SoShadowGroup and the field SoShadowGroup#isActive is set to TRUE.
   * @useenum{ShadowsStyle}. Default is #RAYTRACED.
   */
  SoSFEnum shadowsStyle;


  // Physically Based Lighting Fields

  /**
   * Specifies which shading style is applied to the volume.
   * @useenum{ShadingStyle}. Default is #PHYSICALLY_BASED.
   */
  SoSFEnum shadingStyle;

  /**
   * Specifies a material from a list of predefined ones.
   * This field is taken into account only if #shadingStyle is set to #PHYSICALLY_BASED.
   * The selected material will be applied on the object.
   * The environment map and tone mapping algorithm are part of the predefined material properties.
   * The value #CUSTOM_MATERIAL can also be selected, in which case the properties of the applied material
   * will be retrieved from the current material on the state.
   * @useenum{PredefinedMaterial}. Default is #SEMI_GLOSSY.
   */
  SoSFEnum predefinedMaterial;

  /**
   * Specifies which Environment map is used.
   * This field is taken into account only if #shadingStyle is set to #PHYSICALLY_BASED
   * and if #predefinedMaterial is set to #CUSTOM_MATERIAL.
   * The available environment maps are HDR cubemap textures that will be
   * used to display reflexions of the environment on the object.
   * This behaviour can be disabled by selecting the #NO_ENVIRONMENT field value.
   * @useenum{EnvironmentMap}. Default is #UFFIZI.
   */
  SoSFEnum environmentMap;

  /**
   * Specifies which tone mapping algorithm to apply.
   * This algorithm is used to remap HDR color values to low range values.
   * This field is taken into account only if #shadingStyle is set to #PHYSICALLY_BASED
   * and if #predefinedMaterial is set to #CUSTOM_MATERIAL.
   * @useenum{ToneMapping}. Default is #MEDIUM.
   */
  SoSFEnum toneMapping;


  // Depth Of Field fields

  /**
   * Enables a 'Depth of Field' effect on the volume.
   *
   * This effect blurs everything outside of the depth of field area,
   * but keeps everything inside sharp.
   * This effect is particularly useful to highlight a specific zone of a volume.
   *
   * The focal point is controlled by the field SoCamera::focalDistance of the
   * current camera. Only VolumeRender nodes are affected by the blurring.
   * The amount of blurring applied is controlled by the #blurFactor field.
   * Default is TRUE.
   */
  SoSFBool enableDepthOfField;

  /**
   * This value controls the intensity of the blurring for the Depth Of Field effect.
   * The physical value associated with this parameter is the camera's lens aperture.
   * Bigger values induce a shallower depth of field.
   * Default is 0.01.
   */
  SoSFFloat blurFactor;


  // Shadows Enums

  /**
   * List of available types of shadows.
   */
  enum ShadowsStyle
  {
    /** The standard shadowmap-based shadows. See SoShadowGroup. */
    SHADOWMAP,

    /**
     * Shadows using a ray-marching based algorithm.
     * This type provides high quality shadows, but can have a big impact on performance.
     * It supports SoROI clipping, but does not support SoClipPlane, SoVolumeClippingGroup,
     * SoUniformGridClipping or SoUniformGridProjectionClipping.
     */
    RAYTRACED,
  };


  // Physically Based Lighting Enums

  /**
   * List of available shading styles.
   */
  enum ShadingStyle
  {
    /**
     * Phong shading.
     * This is the regular lighting provided by the field SoVolumeRenderingQuality::deferredLighting.
     */
    PHONG,

    /**
     * Physically-Based shading.
     * This is the advanced lighting that provides:
     *   - A complex BRDF model
     *   - Image-based lighting
     *   - Tone mapping.
     *
     * This shading style can be seen as an enhanced version of the regular deferred lighting.
     * In particular, it shows the same behavior regarding the current SoMaterial on the state and has the
     * same limitations. See SoVolumeRenderingQuality::deferredLighting for a detailed list of limitations.
     */
    PHYSICALLY_BASED,
  };

  /**
   * List of available Environment Maps for Physically Based Lighting.
   */
  enum EnvironmentMap
  {
    /**
    * @IMAGE envmap_sky.png
    */
    SKY,
    /**
    * @IMAGE envmap_grove.png
    */
    GROVE,
    /**
    * @IMAGE envmap_stpeter.png
    */
    STPETER,
    /**
    * @IMAGE envmap_grace.png
    */
    GRACE,
    /**
    * @IMAGE envmap_uffizi.png
    */
    UFFIZI,
    /**
    * @IMAGE envmap_stlazarus.png
    */
    STLAZARUS,

    /** When this value is selected, image-based lighting is disabled. */
    NO_ENVIRONMENT,
  };

  /**
   * List of available Tone Mapping algorithms for Physically Based Lighting.
   */
  enum ToneMapping
  {
    /** Pale Tone Mapping */
    PALE,

    /** Medium Tone Mapping */
    MEDIUM,

    /** High-contrast Tone Mapping */
    HIGH_CONTRAST,
  };

  /**
   * List of available predefined materials.
   */
  enum PredefinedMaterial
  {
    /**
     * ambient: (0.5, 0.5, 0.5)
     * diffuse: (0.8, 0.8, 0.8)
     * specular: (0.25, 0.25, 0.25)
     * shininess: 0.4
     * Environment Map: #GROVE
     * Tone Mapping: #PALE
     */
    MATTE,

    /**
     * ambient: (0.5, 0.5, 0.5)
     * diffuse: (0.8, 0.8, 0.8)
     * specular: (0.06, 0.06, 0.06)
     * shininess: 0.7
     * Environment Map: #STLAZARUS
     * Tone Mapping: #PALE
     */
    SEMI_GLOSSY,

    /**
     * ambient: (0.5, 0.5, 0.5)
     * diffuse: (1.0, 1.0, 1.0)
     * specular: (0.25, 0.25, 0.25)
     * shininess: 0.9
     * Environment Map: #STPETER
     * Tone Mapping: #MEDIUM
     */
    GLOSSY,

    /**
     * When this value is selected, the applied material will get its properties
     * from the current material on the state.
     */
    CUSTOM_MATERIAL,
  };

SoEXTENDER public:
  void GLRender( SoGLRenderAction* action );
  virtual void doAction( SoAction* action );

SoINTERNAL public:
  // Initializes the classes.
  static void initClass();
  static void exitClass();
  virtual void notify( SoNotList* list );

protected:
  virtual ~SoVolumeRenderingPhysicalQuality();

private:
  void commonConstructor();

#ifndef HIDDEN_FROM_DOC
  friend class SoVRImageSpaceEffects;
#endif // HIDDEN_FROM_DOC
};
