/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin GRANGE (Aug 2005)
**=======================================================================*/


#ifndef _SO_VOLUME_SHADER_H_
#define _SO_VOLUME_SHADER_H_


#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderObject.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoTextureUnit.h>
#include <VolumeViz/nodes/SoVolumeRender.h>

#include <Inventor/STL/cassert>
#include <Inventor/STL/string>
#include <Inventor/STL/vector>
#include <Inventor/STL/map>

/*----------------------------------------------------------------------------*/

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable : 4251)
#endif

class SoVolumeData;
class SoIsosurfaceTexture;
class SoVolumeRenderingQuality;
class SoVolumeDataDrawStyle;
class SoInteractiveComplexity;
class SoNodeDependencies;
class SoGroup;
class SoTexture;

/** 
 * @VREXT Shader node for volume rendering.
 *
 * @ingroup VolumeVizNodes
 *
 * @DESCRIPTION
 * This node manages the VolumeViz GLSL shader pipeline. It is derived from 
 * SoShaderProgram and behaves in a similar way. It allows you to supply custom
 * shaders for all VolumeViz rendering shapes (SoVolumeRender, SoOrthoSlice, etc.).
 *
 * @B Note@b: GLSL is the only shading language supported by this node.
 *
 * SoVolumeShader fields provide different pre-implemented rendering effect options,
 * but the application is free to redefine some stages of the VolumeViz shader pipeline
 * by inserting GLSL shader functions in the #shaderObject field (inherited from
 * SoShaderProgram). 
 * The shaderObject multi-field contains only application redefined shaders.
 * The position of a shader in the multi-field explicitly specifies the pipeline shader stage
 * to redefine. Customizable stages are described below in the #ShaderPosition enum.
 * For example, a seismic application could implement co-blending of multiple volumes by supplying
 * a replacement for the VVizComputeFragmentColor() function in the FRAGMENT_COMPUTE_COLOR
 * position of the shaderObject field.
 *
 * @B Note@b: The advanced rendering options, e.g. lighting, are provided by the
 * SoVolumeRenderingQuality node (a subclass of SoVolumeShader). Generally if
 * an application wants to redefine a stage of the shader pipeline but still be
 * able to use these advanced options, it should create an SoVolumeRenderingQuality
 * node and set the replacement shader functions in the shaderObject field of that node.
 *
 * VolumeViz provides a shader pipeline API composed of GLSL functions that are
 * automatically loaded.  This allows the application to modify existing effects
 * or add new effects with minimum new codes. The VolumeViz GLSL shader pipeline
 * API is described in the \ref VolumeVizShaders document.
 *
 * Use the #forVolumeOnly field to specify if the shader is to be used for volume
 * rendering or for non-volume rendering (slice, volume geometry, etc).
 * In some cases it may be possible to use the same shader source for both volume 
 * and non-volume rendering. However the application must still create two shader
 * nodes, one with the #forVolumeOnly field set to TRUE and one with it set to FALSE, 
 * even if both nodes load the same shader source file. (This is necessary because 
 * the shader source must be compiled with different parameters for the different cases.)
 * 
 * No more than one SoVolumeShader (or derived class) can be used with one volume visualization node,
 * e.g. SoVolumeRender. Since SoVolumeIsosurface and SoVolumeRenderingQuality are
 * derived from this class, only one (or none) of these three nodes can be used at
 * the same time.  Exception: Since Open Inventor 7.1 it is possible to use
 * both SoVolumeRenderingQuality and SoVolumeIsosurface with SoVolumeRender.
 *
 * Remember that this is an SoShaderProgram node. The effect will usually be undesirable
 * if it is applied to non-VolumeViz geometry (polygons, lines, etc).  Therefore applications
 * should generally keep the volume visualization nodes and standard geometry
 * nodes separate in the scene graph (i.e. under different SoSeparator nodes).
 *
 * Reserved texture units: @BR @BR
 * Because some rendering methods need to create and use special textures, some texture
 * units must be reserved for internal use.  The application can specify which texture
 * units VolumeViz should use by setting environment variables (see SoPreferences).
 * The texture units between OIV_FIRST_RESERVED_TEXTURE_UNIT and
 * OIV_FIRST_RESERVED_TEXTURE_UNIT+SoShaderProgram::getNumReservedTextures()-1
 * inclusive are reserved for internal VolumeViz use.
 * If OIV_FIRST_RESERVED_TEXTURE_UNIT is not set, its default value is
 * SoFragmentShader::getMaxTextureImageUnit() - SoShaderProgram::getNumReservedTextures().
 * Note: The value returned by SoShaderProgram::getNumReservedTextures() may change between
 * VolumeViz versions.  The total number of available texture units depends on the graphics
 * hardware.
 *
 * @B Composition with Multiple Data: @b
 * @anchor TexCoordConvShaderAPI @BR
 * When compositing multiple datasets that have different dimensions or extents
 * using a custom shader, it is necessary to convert texture coordinates from
 * one dataset to another in order to fetch the correct data values. @BR
 * For this purpose, texture coordinates conversion functions are provided in
 * the @I@B VolumeViz/vvizStructure.h@b@i shader include. @BR
 * For instance,
 * \code
 * vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);
 * \endcode
 * can be used to convert texture coordinates related to one dataset to
 * texture coordinates related to another dataset. @BR
 * The conversion is based solely on the transformations applied to each
 * dataset, which are defined by their model matrix and their extent. @BR
 * Please note that the model matrix of a dataset is defined by to the
 * SoTransformation nodes that are placed @B before @b the SoDataSet node in
 * the order of the traversal.
 *
 * @B Limitations: @b
 * @UL
 *  @LI Only graphics cards supporting the GLSL language can use this node.
 *
 *  @LI Shader filenames beginning with "vviz" are reserved. @BR
 *      Filenames set in a public slot with this prefix will be ignored.
 *
 *  @LI Fragment shaders must not use the GLSL @B discard@b keyword when volume
 *      rendering is using the ray casting algorithm.
 *      If discard is used, the rendering will be incorrect. If a fragment
 *      should not affect the frame buffer, set it to completely transparent.
 *      If you are discarding fragments in the #VVizGetData() function (i.e. before
 *      assigning color), then you must reserve one entry in the color map to be
 *      completely transparent and return the appropriate data value. The easiest
 *      way to do this is to make the first color map entry transparent, either
 *      explicitly or by setting the SoTransferFunction node's @I minValue@i field
 *      to 1. Because the VVizGetData() works with normalized data values in the 
 *      range 0..1, returning 0 will select the transparent entry regardless of 
 *      the actual data range.
 * @ul
 *
 * @B Note: @b
 *   Since the GLSL specification doesn't currently allow the use of any include directive,
 *   Open Inventor provides this service through a comment directive. This provides greater 
 *   flexibility in implementing complex GLSL shaders. Included files are loaded using SoInput
 *   and use the same search path order. 
 *
 *   \code
 *     //!oiv_include <VolumeViz/vvizCombine_frag.h>
 *   \endcode
 *
 * The VolumeViz shader API is described in \ref VolumeVizShaders. @BR@BR
 *
 * Available vertex program functions are described in \ref VolumeVizVertexShaders. @BR@BR
 *
 * Available fragment program functions are described in \ref VolumeVizFragmentShaders. @BR@BR
 *
 * Available constants, macros and data structures are described in \ref VolumeVizShadersData. @BR@BR
 *
 * @EXAMPLE
 * Load a fragment shader in the COMPUTE_COLOR slot.
 * - Various important nodes are omitted here for clarity.
 * - This example could be simplified slightly using the #setFragmentShader() convenience method.
 * \if_cpp
 * - This example could also be simplified slightly using SoFragmentShader's addShaderParameter1i() method.
 * \endif
 *
 *
 * \if_cpp
 *   \code
 *   // Create an SoVolumeData node
 *   SoVolumeData* volData = new SoVolumeData();
 *     volData->dataSetId = 1;
 *     volSep->addChild( volData );
 *
 *   // Create an integer uniform parameter for the dataSetId
 *   SoShaderParameter1i* paramTex1 = new SoShaderParameter1i();
 *     paramTex1->name = "dataId1";
 *     paramTex1->value = volData->dataSetId.getValue();
 *
 *   // Create a fragment shader, load the source file and add the parameter
 *   SoFragmentShader* fragmentShader = new SoFragmentShader();
 *     fragmentShader->sourceProgram = SHADER_FILENAME;
 *     fragmentShader->parameter.set1Value( 0, paramTex1 );
 *
 *   // Create a shader program and add the fragment shader
 *   SoVolumeShader* volShader = new SoVolumeShader();
 *     int shaderPosition = SoVolumeShader::FRAGMENT_COMPUTE_COLOR;
 *     volShader->shaderObject.set1Value( shaderPosition, fragmentShader );
 *     volSep->addChild( volShader );
 * \endcode
 * \endif
 * \if_dotnet
 *   \code
 *   // Create an SoVolumeData node
 *   SoVolumeData volData = new SoVolumeData();
 *     volData.dataSetId.Value = 1;
 *     volSep.AddChild(volData);
 *
 *   // Create an integer uniform parameter for the dataSetId
 *   SoShaderParameter1i paramTex1 = new SoShaderParameter1i();
 *     paramTex1.name.Value = "dataId1";
 *     paramTex1.value.Value = volData.dataSetId.Value;
 *
 *   // Create a fragment shader, load the source file and add the parameter
 *   SoFragmentShader fragmentShader = new SoFragmentShader();
 *     fragmentShader.sourceProgram.Value = SHADER_FILENAME;
 *     fragmentShader.parameter[0] = paramTex1;
 *
 *   // Create a shader program and add the fragment shader
 *   SoVolumeShader volShader = new SoVolumeShader();
 *     int shaderPosition = (int)SoVolumeShader.ShaderPositions.FRAGMENT_COMPUTE_COLOR;
 *     volShader.shaderObject[shaderPosition] = fragmentShader;
 *     volSep.AddChild(volShader);
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *   // Create an SoVolumeData node
 *   SoVolumeData volData = new SoVolumeData();
 *     volData.dataSetId.setValue( 1 );
 *     volSep.addChild( volData );
 *
 *   // Create an integer uniform parameter for the dataSetId
 *   SoShaderParameter1i paramTex1 = new SoShaderParameter1i();
 *     paramTex1.name.setValue( "dataId1" );
 *     paramTex1.value.setValue( volData.dataSetId.getValue() );
 *
 *   // Create a fragment shader, load the source file and add the parameter
 *   SoFragmentShader fragmentShader = new SoFragmentShader();
 *     fragmentShader.sourceProgram.setValue( SHADER_FILENAME );
 *     fragmentShader.parameter.set1Value( 0, paramTex1 );
 *
 *   // Create a shader program and add the fragment shader
 *   SoVolumeShader volShader = new SoVolumeShader();
 *     int shaderPosition = SoVolumeShader.ShaderPositions.FRAGMENT_COMPUTE_COLOR.getValue();
 *     volShader.shaderObject.set1Value( shaderPosition, fragmentShader );
 *     volSep.addChild( volShader );
 *   \endcode
 * \endif
 *
 * @FILE_FORMAT_DEFAULT
 *    SoShaderProgram {
 *      @TABLE_FILE_FORMAT
 *        @TR shaderObject      @TD [
 *        @TR                   @TD   GEOMETRY_MAIN shader
 *        @TR                   @TD   DATA_COMBINE_FUNCTION shader,
 *        @TR                   @TD   GET_DAT_FUNCTION shader,
 *        @TR                   @TD   FRAGMENT_COMPUTE_COLOR shader,
 *        @TR                   @TD   VERTEX_MAIN shader,
 *        @TR                   @TD   VERTEX_MAIN shader,
 *        @TR                   @TD   FRAGMENT_MAIN shader,
 *        @TR                   @TD   ... (reserved)
 *        @TR                   @TD   CUSTOM_SHADER shader
 *        @TR                   @TD   ... (free for apps)
 *        @TR                   @TD  ]
 *        @TR forVolumeOnly     @TD FALSE
 *        @TR raycasting        @TD TRUE
 *        @TR interpolateOnMove @TD TRUE
 *      @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *  SoShaderProgram,
 *  SoVolumeRenderingQuality,
 *  SoMultiDataSeparator,
 *  SoVolumeIsosurface,
 *  SoPreferences
 *
 */
class VOLUMEVIZ_API SoVolumeShader : public SoShaderProgram {

  SO_NODE_HEADER(SoVolumeShader);

public:
  /**
   * Constructor
   */
  SoVolumeShader();

  /**
   * Specifies the position of the shader pipeline stages in the field
   * #shaderObject. In other words, use these enumeration values for
   * the first parameter of the set1Value() method when setting a custom
   * shader in the SoVolumeShader node.
   */
  enum ShaderPosition {
    /**
    * The main geometry program used for rendering.
    *
    * Notes: Defining a custom GEOMETRY_MAIN slot is only supported for compatibility.
    * It is not compatible with the volume rendering.
    * @BR
    */
    GEOMETRY_MAIN =0,

    /**
    * This shader is used for GPU multi-data composition.
    *
    * It must contain an application defined data combining function whose prototype is:
    * \code
    * VVIZ_DATATYPE VVizCombineData(in vec3 dataCoord);
    * \endcode
    *
    * The subtraction of two volumes can be written in GLSL like this:
    * \code
    *   // !oiv_include <VolumeViz/vvizGetData_frag.h>
    *   // !oiv_include <VolumeViz/vvizCombineData_frag.h>
    *
    *   uniform VVizDataSetId dataId1;
    *   uniform VVizDataSetId dataId2;
    *
    *   VVIZ_DATATYPE VVizCombineData(vec3 tcoord)
    *   {
    *     VVIZ_DATATYPE d1 = VVizGetData(dataId1, tcoord);
    *     VVIZ_DATATYPE d2 = VVizGetData(dataId2, tcoord);
    *     return d1-d2;
    *   }
    * \endcode
    *
    * @TABLE_1B
    * @TR First volume @TD Second volume @TD Subtraction
    *   @TR @IMAGE iso_compose_vol1.jpg
    *   @TD @IMAGE iso_compose_vol2.jpg
    *   @TD @IMAGE iso_compose.jpg
    * @TABLE_END
    *
    * NOTE: On the GPU, voxel values are always returned as a normalized value in the
    * range 0..1. If the actual voxel value is needed, the shader function must compute
    * that value using the current data range (see SoDataRange). The application must
    * pass the data range to the shader function as a uniform parameter.
    *
    * See \ref VolumeVizFragmentShaders for more details.
    * @BR
    */
    DATA_COMBINE_FUNCTION,

    /** 
    * This shader is used to access datasets.
    *
    * It must contain an application defined data accessor function whose prototype is:
    * \code
    * VVIZ_DATATYPE VVizGetData(in VVizDataSetId dataset, in vec3 dataCoord);
    * \endcode
    *
    * The default implementation is defined as follows:
    * \code
    * // !oiv_include <VolumeViz/vvizGetData_frag.h>
    *
    * // Default 3D DATA interpolation
    * VVIZ_DATATYPE VVizGetData(VVizDataSetId tex, vec3 tcoord)
    * {
    *   return VVizGetRawData(tex, tcoord);
    * }
    * \endcode
    *
    * This function can be used to apply filters on each data volume.
    * The following example shows how to make a smoothing filter which will remove some noise:
    * \code
    * // !oiv_include <VolumeViz/vvizGetData_frag.h>
    * VVIZ_DATATYPE VVizGetData(in VVizDataSetId dataset, vec3 tcoord)
    * {
    *   vec3 voxelDim = VVizGetVoxelDimensions(dataset);
    *   vec3 du = vec3(voxelDim[0], 0., 0.);
    *   vec3 dv = vec3(0., voxelDim[1], 0.);
    *   vec3 dw = vec3(0., 0., voxelDim[2]);
    *
    *  return 1./6.*(VVizGetRawData(dataset, tcoord-du).w+VVizGetRawData(dataset, tcoord+du).w+
    *                VVizGetRawData(dataset, tcoord-dv).w+VVizGetRawData(dataset, tcoord+dv).w+
    *                VVizGetRawData(dataset, tcoord-dw).w+VVizGetRawData(dataset, tcoord+dw).w);
    *
    * }
    * \endcode
    *
    * @TABLE_1B
    * @TR Non filtered data @TD Filtered data
    *   @TR @IMAGE iso_nofilter.jpg
    *   @TD @IMAGE iso_filter.jpg
    * @TABLE_END
    *
    * NOTE: On the GPU, voxel values are always returned as a normalized value in the
    * range 0..1. If the actual voxel value is needed, the shader function must compute
    * that value using the current data range (see SoDataRange). The application must
    * pass the data range to the shader function as a uniform parameter.
    *
    * See \ref VolumeVizFragmentShaders for more details.
    * @BR
    */
    GET_DATA_FUNCTION,

    /**
    * This shader is used to compute the current fragment color.
    *
    * Note: VVizComputePreIntegrated should @I only@i be called when SoVolumeRendering::renderMode
    * is set to VOLUME_RENDERING (the default).  It is not appropriate for MIN, MAX, etc composition.
    *
    * It must contain an application defined compute fragment color function whose prototype is:
    *
    * \code
    * // For an SoVolumeRender node
    * vec4 VVizComputeFragmentColor(in VVizDataSetId dataset, in vec3 rayDir, inout VVizVoxelInfo voxelInfoFront, in VVizVoxelInfo voxelInfoBack, in int mask);
    * \endcode
    * \code
    * // For all other VolumeViz shape nodes (SoOrthoSlice, SoObliqueSlice, SoVolumeSkin ...)
    * vec4 VVizComputeFragmentColor(in VVIZ_DATATYPE vox, in vec3 dataCoord);
    * \endcode
    * 
    * This function can be used to do co-blending on multiple data volumes.
    * The following example shows how to blend two volumes:
    *
    * \code
    *   // !oiv_include <VolumeViz/vvizGetData_frag.h>
    *   // !oiv_include <VolumeViz/vvizTransferFunction_frag.h>
    *
    *   uniform VVizDataSetId data1;
    *   uniform VVizDataSetId data2;
    *
    *   vec4 blend(in vec3 texCoord)
    *   {
    *     VVIZ_DATATYPE index1 = VVizGetData(data1, texCoord);
    *     vec4 data1Color = VVizTransferFunction(index1, 0);
    *   
    *     VVIZ_DATATYPE index2 = VVizGetData(data2, texCoord);
    *     vec4 data2Color = VVizTransferFunction(index2, 1);
    *
    *     // Color modulated by intensity from volume2
    *     data2Color.rgb *= data1Color.r;
    *     data2Color.a   *= data1Color.a;
    *     return res;
    *   }
    *
    *   // Implement VVizComputeFragmentColor for slice nodes
    *   vec4 VVizComputeFragmentColor(in VVIZ_DATATYPE vox, in vec3 texCoord)
    *   {
    *    return blend(texCoord);
    *   }
    *
    *   // Implement VVizComputeFragmentColor for SoVolumeRender
    *   vec4 VVizComputeFragmentColor(in VVizDataSetId data, in vec3 rayDir, inout VVizVoxelInfo voxelInfoFront, in VVizVoxelInfo voxelInfoBack, in int maskId)
    *   {
    *     return blend(voxelInfoFront.texCoord);
    *   }
    *
    * \endcode
    *
    * @B Composition of multiple independent volumes@b:
    *
    * The following code gives an example of how to blend two independent volumes (different dimensions or extents). @BR
    * The main difference with the previous example is that this one must transform the texture coordinates
    * passed as parameter to the correct space before calling VVizGetData().
    *
    * \code
    * // !oiv_include <VolumeViz/vvizStructure.h>
    * // !oiv_include <VolumeViz/vvizGetData_frag.h>
    * // !oiv_include <VolumeViz/vvizTransferFunction_frag.h>
    *
    * uniform VVizDataSetId DataSet1;
    * uniform VVizDataSetId DataSet2;
    *
    * vec4 alphaBlend(in vec4 dst, in vec4 src)
    * {
    *   vec4 res = src.a * (1.0 - dst.a) * vec4(src.rgb, 1.0) + dst.a * vec4(dst.rgb, 1.0);
    *   res.rgb = (res.a != 0.0) ? (res.rgb / res.a) : vec3(0.0);
    *   return res;
    * }
    *
    * vec4 blend(in VVizDataSetId data, in vec3 texCoord)
    * {
    *   vec4 outputColor = vec4(0.0);
    *
    *   vec3 data1Coord = VVizTextureToTextureVec(data, DataSet1, texCoord);
    *   if (!VVizIsOutsideTexture(data1Coord))
    *   {
    *     VVIZ_DATATYPE index1 = VVizGetData(DataSet1, data1Coord);
    *     vec4 data1Color = VVizTransferFunction(index1, DataSet1);
    *     outputColor = alphaBlend(outputColor, data1Color);
    *   }
    *
    *   vec3 data2Coord = VVizTextureToTextureVec(data, DataSet2, texCoord);
    *   if (!VVizIsOutsideTexture(data2Coord))
    *   {
    *     VVIZ_DATATYPE index2 = VVizGetData(DataSet2, data1Coord);
    *     vec4 data2Color = VVizTransferFunction(index2, DataSet2);
    *     outputColor = alphaBlend(outputColor, data2Color);
    *   }
    *
    *   return outputColor;
    * }
    *
    * // Implement VVizComputeFragmentColor for slice nodes
    * vec4 VVizComputeFragmentColor(in VVIZ_DATATYPE vox, in vec3 texCoord)
    * {
    *   return blend(VVizGetDefaultDataSet(), texCoord);
    * }
    *
    * // Implement VVizComputeFragmentColor for SoVolumeRender
    * vec4 VVizComputeFragmentColor(in VVizDataSetId data, in vec3 rayDir, inout VVizVoxelInfo voxelInfoFront, in VVizVoxelInfo voxelInfoBack, in int maskId)
    * {
    *   return blend(data, voxelInfoFront.texCoord);
    * }
    * \endcode
    *
    * See \ref VolumeVizFragmentShaders for more details.
    *
    * @ENUM_SINCE_OIV 9.0
    * @BR
    */
    FRAGMENT_COMPUTE_COLOR,

    /** 
    * Main vertex shader used for rendering.
    *
    * This stage should not be redefined unless you know exactly what you are doing.
    * If the goal of your application is to compute specific varying attributes 
    * to pass to the fragment shader you should use the shaderPosition slot named 
    * VERTEX_POSTPROCESSING.
    * 
    * Notes: Defining a custom VERTEX_MAIN slot is only supported for compatibility.
    * It is not compatible with the raycasting volume rendering technique which is now
    * the default for SoVolumeRender.
    *
    * See \ref VolumeVizVertexShaders for more details.
    */
    VERTEX_MAIN,

    /**
     * Main fragment shader used for rendering.
     *
     * This stage should not be redefined unless you know exactly what you are doing.
     * If the goal of your application is to blend colors or add effects to the existing pipeline
     * you should use the shaderPosition slot named FRAGMENT_COMPUTE_COLOR.
     * 
     * Notes: Defining a custom FRAGMENT_MAIN slot is only supported for compatibility.
     * It is not compatible with the raycasting volume rendering technique.
     *
     * See \ref VolumeVizFragmentShaders for more details.
     */
    FRAGMENT_MAIN,

    /** 
     * Method called at the end of the VolumeViz vertex shader stage. @BR
     * This shader allows the application to generate custom varying values at the vertex shader
     * stage for use in the fragment shader stage.
     *
     * See \ref VolumeVizVertexShaders for more details.
     *
     * @ENUM_SINCE_OIV 9.0
     */
    VERTEX_POSTPROCESSING,

    /**
     * This method can be used to implement a custom clipping algorithm.
     * Note that when using this method, it may not always be possible for VolumeViz to correctly manage slicing artifacts.
     * You have to use SoVolumeRender::samplingAlignement = BOUNDARY_ALIGNED to avoid slicing artifacts.
     *
     * When applying a custom clipping function with SoVolumeRenderingQuality::voxelizedRendering = TRUE,
     * the voxels will be hollow.
     *
     * @ENUM_SINCE_OIV 9.7
     */
    CLIPPING_FUNCTION,

    /**
     * Shader function used to modify the position of vertices during
     * HeightField rendering.
     *
     * This shader slot must be set with an SoTessellationEvaluationShader.
     *
     * The shader must contain a function whose prototype is:
     * \code
     * vec3 VVizTessVertexShift( in vec3 position, in vec2 texCoord );
     * \endcode
     *
     * The input position represents the 3D position of a vertex on the
     * HeightField surface in model space: X and Y coordinates are mapped to
     * [-1, 1] and the Z coordinate corresponds to the data set value, either
     * normalized to [0, 1] for unsigned integer data types, [-1, 1] for signed
     * integer datatypes, or used "as is" for floating point data types.
     *
     * texCoord represents the 2D texture coordinate associated with the
     * position.
     *
     * The position returned by the function is expected to be the modified
     * vertex position and must also be set in model space.
     *
     * A vertex shift using a shift texture can be written in GLSL like this:
     * \code
     * // !oiv_include <VolumeViz/vvizVertexShift_tess.h>
     *
     * uniform sampler2D shiftTexture;
     *
     * vec3
     * VVizTessVertexShift( in vec3 position, in vec2 texCoord )
     * {
     *   return position + texture( shiftTexture, texCoord ).xyz;
     * }
     * \endcode
     *
     * Notes & Limitations:
     * - This shader is available only for HeightField rendering. It is ignored
     *   during volume rendering and slice rendering.
     * - The default shader returns the input position unmodified.
     * - Although the shader must be an SoTessellationEvaluationShader, the
     *   GLSL function VVizTessVertexShift() will also be called internally
     *   during the tessellation control stage and the fragment stage, so
     *   the shader must only use the GLSL API that is common to those 3 stages.
     * - Bounding Box, Tile Loading & Picking:
     *   Because this shader slot allows to modify the geometry of the rendered
     *   object, a discrepancy may arise that can lead to a misaligned bounding
     *   box, defective tile loading and incorrect picking values. These issues
     *   can be avoided by deriving the SoHeightFieldGeometry class and
     *   overriding the methods voxelToXYZ() and XYZToVoxel(). The voxelToXYZ()
     *   method is used by the bounding box computation and the tile loading and would need
     *   to re-implement the vertex shift transformation. The
     *   XYZToVoxel(SbVec3f) method is used for picking and would need to
     *   implement the inverse vertex shift transformation in order to be
     *   correct.
     *
     * See \ref VolumeVizTessellationShaders
     * @BR
     *
     * @ENUM_SINCE_OIV 10.8
     */
    TESS_VERTEX_SHIFT,

    /** 
     * This position and all subsequent positions CUSTOM_SHADER+x are freely available for user-defined shaders.
     *
     * @ENUM_SINCE_OIV 9.0
     */
    CUSTOM_SHADER = 64
 };

  /**
  * Set to TRUE if the shader should be called for volume rendering (SoVolumeRender).
  * Set to FALSE if it should be called for other VolumeViz shapes (SoOrthoSlice, 
  * SoObliqueSlice, SoVolumeSkin, volume geometry, etc).
  * Default is FALSE.
  *
  * TRUE means that if the shader uses texture coordinates, they will be 3D texture
  * coordinates. FALSE means they will be 2D texture coordinates. 
  *
  * In some cases it may be possible to use the same shader source for both volume 
  * and non-volume rendering. However the application must still create two shader
  * nodes, one with TRUE and one with FALSE, even if both nodes load the same shader
  * source file. (This is necessary because the shader source must be compiled with
  * different parameters for different cases.)
  */
  SoSFBool forVolumeOnly;


  /**
  * When set to FALSE, interpolation between LDM tiles (across the tile boundary)
  * is not done when rendering in interactive mode.
  * This increases the interactive rendering frame rate at the cost of some rendering 
  * artifacts at the tile boundaries (seen as horizontal and vertical "lines" in
  * the rendered image) and a small lag when switching between still and interactive.
  * If your rendering shaders do not need to access voxel's neighbor (for lighting 
  * or gradient computation for instance), you should set this field to TRUE as the
  * cost of interpolation is not significant in this case.
  *
  * Default is TRUE.
  *
  * @FIELD_SINCE_OIV 9.0
  */
  SoSFBool interpolateOnMove;

  /**
   * Returns TRUE if SoVolumeShader is supported by the current graphics board.
   * When using a debug build of Open Inventor, some "no context available"
   * warning messages may be generated. You can ignore them or see
   * SoGLExtension for an example of using SoGLContext to avoid them.
   */
  static SbBool isSupported(SoState* state=NULL) ;

  /**
   * Creates a fragment shader with the given filename and
   * add it at the given pos. Return value is the new fragment shader.
   */
  virtual SoFragmentShader* setFragmentShader(int pos, const SbString& filenameOrSource,
                                              SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /**
   * Creates a vertex shader with the given filename and
   * adds it at the given pos.
   * @return the new vertex shader.
   */
  virtual SoVertexShader* setVertexShader(int pos, const SbString& filenameOrSource,
                                          SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

#ifndef HIDDEN_FROM_DOC
SoINTERNAL public:

  /**
   * Returns current custom shader at the given pipeline position.
   * Returns NULL if no custom shader available.
   */
  template<typename T>
  T* getCustomShader(const ShaderPosition pos)
  {
    if (shaderObject.getNum() > pos )
      return static_cast<T*>(shaderObject[pos]);
    return NULL;
  }

  /**
   * setup public shader stage stagePos to defaultShader if not yet in hiddenShaderObjectList objName.
   */
  template<typename T>
  T* setupPublicShaderStage(const ShaderPosition stagePos, const char* hiddenName, T* defaultShader)
  {
    T* fp = getCustomShader<T>(stagePos);
    // There is no custom shader for this stage
    if ( !fp )
      return setupPrivateShaderStage<T>(hiddenName,defaultShader);
    else
      removeHiddenShaderObject(hiddenName);
    return fp;
  }


  /**
   * setup public shader stage stagePos to shaderSource if not yet in hiddenShaderObjectList objName.
   */
  template<typename T>
  T* setupPublicShaderStage(const ShaderPosition stagePos, const char* hiddenName, const SbString& shaderSource)
  {
    T* fp = getCustomShader<T>(stagePos);
    // There is no custom shader for this stage
    if ( !fp )
      return setupPrivateShaderStage<T>(hiddenName, shaderSource);
    else
      removeHiddenShaderObject(hiddenName);
    return fp;
  }

  /**
   * setup private shader stage hiddenName to shaderSource if needed.
   * Does not override if already exists.
   */
  template<typename T>
  T* setupPrivateShaderStageNoOverride(const char* hiddenName, const SbString& shaderSource)
  {
    T* fp = (T*) getHiddenShaderObject(hiddenName);
    if (fp == nullptr)
    {
      fp = new T;
      fp->sourceProgram.setValue(shaderSource);
      setHiddenShaderObject(hiddenName, fp);
    }
    return fp;
  }

  /**
   * setup public shader stage stagePos to shaderSource if not yet in hiddenShaderObjectList objName.
   * Does not override if hidden already exists.
   */
  template<typename T>
  T* setupPublicShaderStageNoOverride(const ShaderPosition stagePos, const char* hiddenName, const SbString& shaderSource)
  {
    T* fp = getCustomShader<T>(stagePos);
    if (fp == nullptr)
      return setupPrivateShaderStageNoOverride<T>(hiddenName, shaderSource);
    else
      removeHiddenShaderObject(hiddenName);
    return fp;
  }

  /**
   * Returns true if frag is a default fragment main
   */
  bool isInternalFragment(SoFragmentShader* frag) const;

  /**
   * Returns true if obj is a default shader object
   */
  bool isInternalShaderObject(SoShaderObject* obj) const;

  virtual void doAction(SoAction *action);

  /**
   * GLrender action.
   */
  virtual void GLRender( SoGLRenderAction *action );

  /**
   * Installs the ShaderProgramElement
   */
  void installShaderProgram( SoGLRenderAction *action );

  virtual void doRendering(SoGLRenderAction *action);

  /** register class. */
  static void initClass() ;

  /** unregister class. */
  static void exitClass() ;

  /**
   * Returns true if this shader is only for volume (ie: not for orthoslice,
   * obliqueslice and volume skin)
   */
  SbBool isForVolumeOnly() { return forVolumeOnly.getValue(); }

  /**
   * Sends uniform parameters allowing shader to compute texcoords
   * relative to the entire volume (instead of per-tile).
   */
  static void setTexCoordVolUniform(SoState*state, SoVolumeData* vd, SoNode* node);

  static int getFirstUsedTextureUnit(SoGLRenderAction* action);

  /** Installs textures needed for shaders */
  virtual void installTextures(SoGLRenderAction* ) {}

  /** Returns TRUE if render mode technique is raycasting  */
  virtual SbBool isRaycastingEnabled(SoState* state) const;

  /** Returns TRUE if volumeGroup special handling is activated */
  virtual SbBool isVolumeGroupEnabled(SoState* state) const;

  /** Reserves needed texture units */
  virtual void allocateTextureUnit(SoGLRenderAction *) const {}
  
  /**
  * Indicates if GL_EXT_texture_array is supported by your graphics board.
  */
  static SbBool isTextureArrayEXTSupported(SoState* state);

  /**
   * returns whether IVVR_FORCE_RAYCASTING is TRUE or not (TRUE by default).
   * Must be called after SoVolumeShader::initClass().
   */
  static bool isRaycastingDefault()
  { return s_forceRaycasting; };


#if SoDEPRECATED_BEGIN(9630)
  SoDEPRECATED_METHOD(9630, "Do nothing")
  void createRenderModeShaders(){}
#endif  /** @DEPRECATED_END */

  /** Returns true if interpolation across tile boundaries must be enabled */
  virtual bool isInterpolationActive(SoGLRenderAction* action);

  /** Gets the value to use for VVizUseOrthoCamera shader parameter */
  bool mustUseOrthoCamera( SoState* state );

protected:

  /** Keeps shader node name up to date when changed */
  virtual void notify(SoNotList *list);

  /**
   * Installs noise texture in the jitter texture unit
   */
  void installJitteringTexture(SoGLRenderAction* action);

  /**
   * Uninstalls the jitter texture
   */
  void uninstallJitteringTexture();

  ~SoVolumeShader();

  // Clean a specified path:
  // - converts the path to unix path.
  // - removes the drive is the path is a windows path.
  static SbString cleanUpFilePath( const SbString& string );

  /** Creates a shader suitable for shadowmap generation pass */
  virtual SoShaderProgram* generateShadowShader() const;

  /**
   * Checks that all shaders slots that are not currently used fallback to a
   * well defined default shader.
   */
  void updateShaderSlots( SoState* state );

  /**
   * Default threshold used when computing gradient
   */
  static const float DEFAULT_GRADIENT_THRESHOLD;

  static const float DEFAULT_EDGE_THRESHOLD;

  static const float DEFAULT_BOUNDARY_THRESHOLD;
  static const float DEFAULT_BOUNDARY_INTENSITY;

#endif //HIDEN_FROM_DOC

#if SoDEPRECATED_BEGIN(9620)

SoINTERNAL protected:
  SoDEPRECATED_MEMBER(9620, "Only raycasting mode is supported")
  SoSFBool raycasting;
#endif  /** @DEPRECATED_END */

private:

  static const size_t JITTER_TEX_SIZE;

  /**
   * Creates texture used for jittering
   */
  void createJitterTex();

  /**
   * Needed for backward compatibility with SoVolumeIsosurface
   * Resets drawstyle to volume rendering if a SoVolumeIsosurface node is on the state
   * */
  bool hasToHandleFakeSoVolumeIsosurfaceStyle(SoState* state);
  void handleFakeSoVolumeIsosurfaceStyle(SoGLRenderAction* action);

  SoRef<SoGroup> m_jitterTextureGroup;
  SoRef<SoTextureUnit> m_jitterTextureUnit;
  SoRef<SoTexture2> m_jitterTexture;

  /** Proxy draw style used for compatibility with SoVolumeIsosurface */
  SoVolumeDataDrawStyle* m_volumeDataDrawStyle;

  static int s_texture_array_EXT_extensionID;
  static bool s_forceRaycasting;
};

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /*_SO_VOLUME_SHADER_H_*/


