/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/


#ifndef SO_VOLUME_SHAPE_H
#define SO_VOLUME_SHAPE_H

#include <Inventor/nodes/SoShape.h>

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFUInt32.h>
#include <Inventor/fields/SoSFNode.h>

#include <Inventor/STL/vector>

#include <Inventor/sys/SoGLType.h>

#include <Inventor/SbViewportRegion.h>

#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoLdmShape.h>

class SoState;
class SoVolumeData;
class SoVolumeRenderingQuality;
class SoVolumeShader;
class SoVolumeShaderARB;
class SoVolumeState;
class SoProgressIndicator;

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif


/**
*
* @VREXT Abstract base class for slices, SoVolumeSkin, SoHeightFieldRender,
* and SoVolumeRender nodes.
*
* @ingroup VolumeVizNodes
*
* @DESCRIPTION
*
*   This class defines common properties for the SoOrthoSlice, SoObliqueSlice,
*   SoFenceSlice, SoVolumeSkin, SoHeightFieldRender and SoVolumeRender nodes.
*
*   The #interpolation field controls how the texture is interpolated.
*
*   The #composition field is deprecated and should not be used.  Instead
*   specify the composition mode for volume rendering using the SoVolumeRender node.
*
* @SEE_ALSO
*    SoObliqueSlice,
*    SoOrthoSlice,
*    SoFenceSlice,
*    SoVolumeRender,
*    SoVolumeSkin,
*    SoHeightFieldRender
*
*
*/
class VOLUMEVIZ_API SoVolumeShape : public SoLdmShape {
  SO_NODE_ABSTRACT_HEADER(SoVolumeShape);

public:

  /** Interpolation mode.
   *  Used with #interpolation field.  Also used with interpolation field in
   *  volume geometry nodes, e.g. SoVolumeIndexedFaceSet::interpolation 
   */
  enum Interpolation {
    /**
     * OpenGL "nearest neighbor" interpolation
     */
    NEAREST,
    /** OpenGL linear interpolation (Default).
     *  This implies bi-linear interpolation for 2D textures
     *  (e.g. SoOrthoSlice, SoObliqueSlice, SoVolumeSkin, SoFenceSlice) and
     *  tri-linear interpolation for 3D textures (SoVolumeRender, SoVolumeGeometry).
     */
    LINEAR,
    /**
     * Tri-linear interpolation for SoObliqueSlice.
     * Tri-linear interpolation is used when extracting the 2D texture that
     * will be applied to the slice, providing better image quality.  
     * Same as LINEAR for other nodes.
     * @BR @BR
     * @ENUM_SINCE_OIV 6.0
     */
    TRILINEAR,
    /**
     * Multi-sample interpolation for slice shapes.
     * Interpolates data values using 12 samples around the voxel. This provides much higher image
     * quality and specifically avoids rendering artifacts due to bi-linear interpolation.
     * Only supported for 2D texture shapes (e.g. SoOrthoSlice, SoObliqueSlice, SoVolumeSkin, SoFenceSlice).
     * Same as LINEAR for other nodes.
     * @BR @BR
     * @ENUM_SINCE_OIV 7.0
     */
    MULTISAMPLE_12,
    /**
     * Cubic interpolation.
     * This implies bi-cubic interpolation for 2D textures
     * (e.g. SoOrthoSlice, SoObliqueSlice, SoVolumeSkin, SoFenceSlice) and
     * tri-cubic interpolation for 3D textures (SoVolumeRender, SoVolumeGeometry).
     * It gives smoother results, especially for volume rendering.
     *      @TABLE_1B
     *         @TR Linear interpolation @TD Cubic interpolation
     *         @TR @IMAGE volume_linear.jpg
     *         @TD @IMAGE volume_tricubic.jpg
     *      @TABLE_END
     *
     * @warning Heavy GPU usage. Primarily useful for still image rendering. @BR
     *
     * It is recomended to use SoInteractiveComplexity node to switch bewteen LINEAR interpolation
     * while moving and CUBIC interpolation for Still frame:
     *   \if_cpp
     *     \code
     *     SoInteractiveComplexity* icplx = new SoInteractiveComplexity;
     *     icplx->fieldSettings.set1Value( 0, "SoVolumeRender interpolation LINEAR CUBIC" );
     *     root->addChild(icplx);
     *     \endcode
     *   \endif
     *   \if_dotnet
     *     \code
     *     SoInteractiveComplexity icplx = new SoInteractiveComplexity();
     *     icplx.fieldSettings[0] = "SoVolumeRender interpolation LINEAR CUBIC";
     *     root.AddChild(icplx);
     *     \endcode
     *   \endif
     *   \if_java
     *     \code
     *     SoInteractiveComplexity icplx = new SoInteractiveComplexity();
     *     icplx.fieldSettings.set1Value( 0, "SoVolumeRender interpolation LINEAR CUBIC" );
     *     root.addChild(icplx);
     *     \endcode
     *   \endif
     *
     * @ENUM_SINCE_OIV 9.3
     */
    CUBIC
  };

  /**
  * Interpolation mode. 
  * @useenum{SoVolumeShape::Interpolation}. Default is LINEAR.
  * NOTE: In most cases on modern graphics boards, indexed textures are used,
  * so this refers to interpolation of volume data values.
  */
  SoSFEnum interpolation;

  /**
  * Set an application defined SoProgressIndicator object which will raise an event
  * before and after the rendering task, before and after each subtask (in
  * this case: Texture creation and Geometry creation) and after each step in the subtasks
  * which represents in most cases individual tiles of data.
  *
  * - #SoProgressIndicator::onBeginTask / #SoProgressIndicator::onEndTask will be raised only one time per frame : "Rendering".
  * - #SoProgressIndicator::onBeginSubTask / #SoProgressIndicator::onEndSubTask will be raised 2 times per frame : "Loading Texture" and "Create Geometry".
  * - #SoProgressIndicator::onEndStep will be raised several times per subtask.
  *
  * @B Track loading progression @b with #SoProgressIndicator::onEndStep event of SubTask "Loading Texture":
  *  - NumSteps represents here the number of tiles to load to reach the "stable rendering state".
  *  - NumStepsDone represents here the number of tiles currently loaded since the last NumSteps reset.
  *  - The two numbers are reset when the viewer is moving the camera.
  *
  * \if_cpp
  * The example VolRend (examples/source/VolumeViz/VolRend) implements a loading progress bar in class EventRaisedProgressBar.
  * \endif
  *
  * @B Limitations @b:
  *   - if SoLDMResourceParameters::maxMainMemory (CPU Mem) is lower than SoLDMResourceParameters::maxTexMemory / SoLDMResourceParameters::max2DTexMemory (GPU mem/GPU mem for slices) the number of steps to reach the "stable rendering state" will be bigger than expected (the progress bar will stop without reaching the end).
  *   - if screen resolution culling is activated (see #SoLDMGlobalResourceParameters::setScreenResolutionCulling) the number of steps to reach the "stable rendering state" might be lower than expected
  *     (progress bar reaches the end but loading might continue).
  *   - Not yet supported by SoFenceSlice and SoHeighFieldRender.
  *
  * If set to NULL no events will be raised. Default is NULL.
  *
  * @M_SINCE 2023.2
  */
  virtual void setRenderProgress(SoProgressIndicator* ps);

#if SoDEPRECATED_BEGIN(9100)

  /**
   * Composition mode
   */
  enum SoDEPRECATED_ENUM(9100,"See SoVolumeRender::RenderMode enum.")
  Composition {
    /** Max intensity */
    MAX_INTENSITY = 0,
    /** Min intensity */
    MIN_INTENSITY,
    /** Sum intensity */
    SUM_INTENSITY,
    /** Alpha blending (Default) */
    ALPHA_BLENDING,
    COMPOSITION_LAST
  };

  /**
  * Specifies color composition mode.
  * @useenum{Composition}. Default is ALPHA_BLENDING.
  *
  * ALPHA_BLENDING blends the R, G, and B components for each pixel based on the
  * their alpha values.
  *
  * SUM_INTENSITY draws the sum of the R, G, and B components for each pixel.
  *
  * MAX_INTENSITY draws the maximum R, G, and B components for each pixel.
  *
  * MIN_INTENSITY draws the minimum R, G, and B components for each pixel.
  *
  * @B Note: @b
  * In all modes, blending is done against the background color of the scene (or
  * the object behind the volume). This means that, using MIN_INTENSITY for example,
  * if the background color is black then 0,0,0 is already the minimum RGB value
  * and the volume will not be visible.  Generally when using MIN_INTENSITY the
  * background color should be set to white (1,1,1) and when using MAX_INTENSITY
  * or SUM_INTENSITY the background color should be set to black (0,0,0).  To set
  * the background color see the setBackgroundColor method of the appropriate
  * RenderArea class, for example SoWinRenderArea on Windows.
  *
  * @FIELD_SINCE_OIV 5.0.3
  */
  SoDEPRECATED_MEMBER_NOWARN(9100,"See SoVolumeRender::renderMode field.")
  SoSFEnum composition;

#endif /** @DEPRECATED_END */

SoEXTENDER public:

  /** @copydoc SoLdmShape::GLRender */
  virtual void GLRender(SoGLRenderAction *action);

  /** @copydoc SoLdmShape::computeBBox */
  virtual void computeBBox(SoAction *action, SbBox3f &box, SbVec3f &center);

  /** @copydoc SoLdmShape::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

  // ScaleViz specific : force rendering on all oiru in depth compositing mode
  // As VolumeViz shape nodes are considered as transparent (alpha)
  virtual int32_t getRenderUnitID() const;

SoINTERNAL public:

  /** @copydoc SoLdmShape::initClass */
  static void initClass();

  /** @copydoc SoLdmShape::initClass */
  static void exitClass();

  /**
   * Set all VViz uniforms.
   */
  void setVVizUniforms( SoGLRenderAction* action,
                       const float *sliceTangent, const float *sliceBinormal,
                       float scale, float sliceSpacing,
                       const SbVec3f& scaleFactor,
                       const SbVec3f& lowResScaleFactor, SbBool doTexGen = TRUE,
                       const SbMatrix* virtToPageTableMatrix = NULL, const SbMatrix* shapeToVirtMatrix = NULL);

  /**
   * Set axis along which slices are drawn
   */
  void setVVizZAxis( SoGLRenderAction* action, const SbVec3f& zAxis );

  /**
   * Install the colormap according to its type.
   * Return TRUE if an external shader is used.
   * textype is not modified for volume
   */
  virtual bool enableColorMap(SoState *state,
                              SoTransferFunction::GLColorMapType colorMapInstalled,
                              int &texType);

  /**
   * Return the type of installed colormap
   */
  SoTransferFunction::GLColorMapType getInstalledColorMap() const;

  /**
   * Return SoTexture::NEAREST or SoTexture::LINEAR according to interpolation field.
   */
  SoTexture::Filter getTexFilter() const;

  /**
   * Activate blending according to composition field.
   */
  virtual void enableBlending(SoState *state);

  /**
   * Called when user stops or starts moving.
   */
  virtual void onUserInteractionChange(SoState* state, bool stopMoving);

  /**
   * Return interpolation value accoriding to the SoInteractiveComplexityElement
   */
  SoVolumeShape::Interpolation getInteractiveInterpolationValue(SoState* state) const;

  SoVolumeState* getVolumeState() const
  {
    return m_volumeState;
  }

protected:
  SoVolumeState* m_volumeState;

  SoVolumeShape();
  virtual ~SoVolumeShape();

  /**
   * True if the shape is transparent.
   */
  virtual bool isTransparent(SoState *state);

  /**
   * Delay rendering as needed by VViz.
   */
  virtual void delayRendering( SoGLRenderAction* action );

  /**
   * @return true if the shape is going to be delayed
   */
  bool isDelayed();

  /**
   * Return the volume shader in the state, NULL otherwise.
   */
  SoVolumeShader* getVolumeShader(SoState *state);

  /**
   * Blending will use (1, 1-alpha) instead of (alpha, 1-alpha)
   * for alpha channel composition.
   */
  void useSeparateAlphaBlend(bool flag);

  /**
   * Returns the supported openGL depth internal format.
   */
  GLenum getDepthInternalFormat();

  /**
   * Return true if there is user interaction.
   */
  bool isMoving();

  /**
   * Do the actual rendering.
   */
  virtual void doRendering(SoGLRenderAction* action) = 0;

  static void onUserInteractionChangeCb(SoState* state, SoNode* node, bool moving);

  SoProgressIndicator* m_renderProgress;

#ifndef HIDDEN_FROM_DOC
  friend class SoSliceInterface;
  friend class SoVolumeState;
#endif

SoINTERNAL protected :

  /** Register ldm geometry to octree */
  virtual void ldmAction( SoLdmValuationAction* action );
};

#if defined(_WIN32)
#pragma warning( pop )
#pragma warning(disable:4251)
#endif

#endif


