/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : M. Heck (Jul 2003)
**=======================================================================*/
#ifndef  _SO_VR_VOX_FILE_READER_
#define  _SO_VR_VOX_FILE_READER_

#include <LDM/readers/SoVolumeReader.h>
#include <VolumeViz/nodes/SoVolumeData.h>


/**
 * @VREXT VOX file reader.
 * 
 * @ingroup VolumeVizReaders
 * 
 * @DESCRIPTION
 * Reader for TeraRecon ".vox" data files (vox1999 format).
 * 
 *   VOX is a volume interchange format defined by TeraRecon Inc.
 *   (www.terarecon.com). The VolumeViz reader can load "Vox1999a"
 *   files containing 8- or 16-bit voxels (first volume only).
 * 
 *   The following attributes are supported:
 *    - "Volume Size" which describes the size of the volume in voxels.
 *    - "Voxel Size" which describes the size of voxel data (only 8-bit and 16-bit are supported).
 *    - "Endian" which describes the endianness of the data (both little-endian and big-endian modes are supported).
 *    - "Volume Scale" which describes the volume of one voxel.
 *    - "Volume Position" which describes the origin position of the first voxel.
 *
 *   Since Open Inventor 10.3, the following custom attributes are supported when specified in the "Field" descriptor:
 *    - Scale: Define a scaling to apply to data values.
 *    - Offset: Define an offset to apply to data values.
 *
 *   If Scale and Offset are specified and are not exactly 1 and 0 respectively:
 *   - The actual data value will be : newValue = Scale * fileValue + Offset.
 *   - The volume data type is automatically changed to SIGNED_SHORT. @BR
 *     This means that, even if the file is specified as unsigned 8 bit, the data will be loaded as signed 16 bit values.
 *
 *   Other attributes have no effect on the reader and loaded data.
 * 
 */
class VOLUMEVIZ_API SoVRVoxFileReader : public SoVolumeReader 
{
  SO_FIELDCONTAINER_HEADER(SoVRVoxFileReader);

 public:
  /** Constructor. */
  SoVRVoxFileReader();

  /**
  * Specifies the path of the file. Returns 0 for success. Any other return
  * value indicates failure.
  */
  virtual int setFilename( const SbString& filename );

  /**
   * @copydoc SoVolumeReader::getDataChar(SbBox3f&,SoDataSet::DataType&,SbVec3i32&)
   */
  virtual ReadError getDataChar( SbBox3f &size, SoDataSet::DataType &type,  SbVec3i32 &dim ) override;

  /**
   * @copydoc SoVolumeReader::getSubSlice(const SbBox2i32&,int,void*)
   */
  virtual void getSubSlice( const SbBox2i32& subSlice, int sliceNumber, void * data );

  /** 
   * This method is optional. It returns the number of significant bits of the volume data.
   *
   * If it is not implemented, the default return value is 0, meaning the number of bits
   * depends on the data type. See the last parameter of SoVolumeData::setVolumeData().
   * This method is called immediately after #getDataChar().
   */
  virtual int getNumSignificantBits() 
  { return m_numSigBits; }

  /**
   * Returns the reader type.
   */
  virtual SoVolumeReader::ReaderType getReaderType ()
  {return VOX;};

 private:

  /**
   * Convert inBuffer of inDataType using inEndianness to outBuffer of outDataType using current machine endianess.
   * inEndianness should be 'B' for big endian or 'L' for little endian.
   */
  void convertFromTo(void* inBuffer, void* outBuffer, size_t numElements, SoDataSet::DataType inDataType, SoDataSet::DataType outDataType, char inEndianness);

  /**
   * Convert inBuffer using inEndianness to outBuffer of outDataType using current machine endianess.
   * inEndianness should be 'B' for big endian or 'L' for little endian.
   */
  template<typename FROM_TYPE>
  void convertFromTo(FROM_TYPE* inBuffer, void* outBuffer, size_t numElements, SoDataSet::DataType outDataType, char inEndianness);

  /**
   * Convert inBuffer using inEndianness to outBuffer using current machine endianess.
   * inEndianness should be 'B' for big endian or 'L' for little endian.
   */
  template<typename FROM_TYPE, typename TO_TYPE>
  void convertFromTo(FROM_TYPE* inBuffer, TO_TYPE* outBuffer, size_t numElements, char inEndianness);

  SbBox3f   m_size;
  SbVec3i32 m_dim;

  // File may store data as int8 but this volume may be seen as int16.
  // For. when file is uint8 but offset is -2000, actual data may be in [-2000, 0] so have
  // to be encode in signed 16 bits.
  SoDataSet::DataType m_fileDataType;
  SoDataSet::DataType m_volumeDataType;

  int m_offsetToData;

  // Specific data from vox file
  SbString  m_volumeTitle;
  SbString  m_volumeCopyright;
  int       m_volumeCount;
  SbVec3f   m_volumeScale;
  SbVec3f   m_volumePosition;
  char      m_endian;
  SbMatrix  m_volumeMatrix;

  // Scale and offset to rescale data:
  // outData = scale * inData + offset.
  double m_voxelScale;
  double m_voxelOffset;

  int       m_numSigBits;

  SbString m_userFilename;

  // If true, apply scale and offset to data in the getSubSlice. Linked to LDM_VOX_READER_APPLY_SCALE_OFFSET.
  static bool s_applyScaleOffset;
};

#endif // _SO_VR_VOX_FILE_READER_


