#ifndef OIV_SHADER_STATE_H
#define OIV_SHADER_STATE_H

/**
 * @ingroup InventorShaderState
 * @file
 * This file contains fonction to acces built-in GLSL uniforms
 */


/*****************************************************************************/
// Constants

/**
 * Get the maximum number of lights sets by Open Inventor.
 */
int OivMaxLights();

/**
 * Get the maximum number of clipping planes sets by Open Inventor.
 */
int OivMaxClipPlanes();

/**
 * Get the maximum number of texture coordinates sets by Open Inventor.
 */
int OivMaxTextureCoordinates();

/**
 * Get the maximum number of texture units sets by Open Inventor.
 */
int OivMaxTextureUnits();

/*****************************************************************************/
// Transforms

/**
 * Get transformation of the model (M).
 * @return the model matrix.
 */
mat4 OivModelMatrix();

/**
 * Get inverse transformation of the model (M^{-1}).
 * @return the inverse of the model matrix.
 */
mat4 OivModelMatrixInverse();

/**
 * Get transformation of the view (V).
 * @return the view matrix.
 */
mat4 OivViewMatrix();

/**
 * Get inverse transformation of the view (V^{-1}).
 * @return the inverse of the view matrix.
 */
mat4 OivViewMatrixInverse();

/**
 * Get the combination of model and view transformations (MV),
 * i.e., MV = V * M.
 * @return the model-view matrix.
 */
mat4 OivModelViewMatrix();

/**
 * Get the inverse of the model-view matrix,
 * @return the inverse of the model-view matrix.
 */
mat4 OivModelViewMatrixInverse();

/**
 * Get the normal matrix (N),
 * i.e., N = tr(MV^{-1}).
 * @return the normal matrix.
 */
mat3 OivNormalMatrix();

/**
 * Get the projection matrix (P).
 * @return the projection matrix.
 */
mat4 OivProjectionMatrix();

/**
 * Get the inverse of the projection matrix (P^{-1}).
 * @return the inverse projection matrix.
 */
mat4 OivProjectionMatrixInverse();

/**
 * Get the combination of projection and model-view matrices (MVP),
 * i.e., MVP = P * V * M.
 * @return the model-view-projection matrix
 */
mat4 OivModelViewProjectionMatrix();

/**
 * Get texture matrices.
 * @param unit specifies which texture unit between 0 and OivMaxTextureUnit()-1.
 * @return the texture matrix for a given unit.
 */
mat4 OivTextureMatrix(int unit);

/**
 * Get the origin of the viewport.
 * @return width and height of the viewport origin packed in vec2.
 */
vec2 OivViewportOrigin();

/**
 * Get the size of the viewport.
 * @return width and height of the viewport packed in vec2.
 */
vec2 OivViewportSize();

/**
 * Get the near/far view volume of the camera
 */
vec2 OivViewVolumeSize();

/**
 * Get the near/far depth range
 */
vec2 OivDepthRange();

#define OIV_PROJECTION_TYPE_ORTHO       0
#define OIV_PROJECTION_TYPE_PERSPECTIVE 1

/**
* Get the projection type of the camera (OIV_PROJECTION_TYPE_ORTHO or OIV_PROJECTION_TYPE_PERSPECTIVE)
*/
int  OivViewProjectionType();

/*****************************************************************************/
// Lighting

/**
 * Ask if lighting is enabled.
 * @return true if lighting is enabled, false otherwise.
 */
bool OivHasLighting();

/**
 * Ask if two-sided lighting is enabled.
 * @return true if two-sided lighting is enabled, false otherwise
 */
bool OivHasTwoSidedLighting();

/**
 * Ask if per-pixel lighting is enabled.
 * @return true if per-pixel lighting is enabled, false otherwise
 */
bool OivHasPerPixelLighting();

/**
 * Ask if color material is enabled (for front faces).
 * @return true if color material is active for front faces, false otherwise
 */
bool OivHasColorMaterial();

/**
 * Ask if color material is enabled for front faces.
 * @return true if color material is active for front faces, false otherwise
 */
bool OivHasFrontColorMaterial();

/**
 * Ask if color material is enabled for back faces.
 * @return true if color material is active for back faces, false otherwise
 */
bool OivHasBackColorMaterial();

/**
 * Gets the global ambient color value
 * @return The global ambient color value
 */
vec4 OivGlobalAmbient();

/**
 * Ask if the light is enabled.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return true if light with a given id is enabled, false otherwise.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
bool OivLightSourceEnabled(int id);

#define OIV_LIGHT_TYPE_NONE        0
#define OIV_LIGHT_TYPE_POINT       1
#define OIV_LIGHT_TYPE_SPOT        2
#define OIV_LIGHT_TYPE_DIRECTIONAL 3
#define OIV_LIGHT_TYPE_AREA        4

/**
 * Get the type of the light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the type of the light (OIV_LIGHT_TYPE_NONE, OIV_LIGHT_TYPE_POINT, OIV_LIGHT_TYPE_SPOT or OIV_LIGHT_TYPE_DIRECTIONAL).
 */
int OivLightSourceType(int id);

/**
 * Get the light position.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light position for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourcePosition(int id);

/**
 * Get the light spot direction.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot direction for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec3 OivLightSourceSpotDirection(int id);

/**
 * Get the light spot exponent.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot exponent for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceSpotExponent(int id);

/**
 * Get the light spot cutoff.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot cutoff for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceSpotCutoff(int id);

/**
 * Get the light spot cutoff cosine.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light spot cutoff cosine for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceSpotCosCutoff(int id);

/**
 * Get the light ambient color.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light ambient color for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourceAmbient(int id);

/**
 * Get the light diffuse color.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light diffuse color for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourceDiffuse(int id);

/**
 * Get the light specular color.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light specular color for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
vec4 OivLightSourceSpecular(int id);

/**
 * Get the light constant attenuation value.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light constant attenuation value for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceConstantAttenuation(int id);

/**
 * Get the light linear attenuation value.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light linear attenuation value for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceLinearAttenuation(int id);

/**
 * Get the light quadratic attenuation value.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return the light quadratic attenuation value for a given light id.
 *
 * Note that this parameter is relevant only if lighting is enabled (i.e. OivHasLighting() returns true).
 */
float OivLightSourceQuadraticAttenuation(int id);

int OivMaxLightId();

/*****************************************************************************/
// Material

/**
 * Get front material ambient color.
 * @return ambient color for front material.
 */
vec4 OivFrontMaterialAmbient();

/**
 * Get front material diffuse color.
 * @return diffuse color for front material.
 */
vec4 OivFrontMaterialDiffuse();

/**
 * Get front material specular color.
 * @return specular color for front material.
 */
vec4 OivFrontMaterialSpecular();

/**
 * Get front material emissive color.
 * @return emissive color for front material.
 */
vec4 OivFrontMaterialEmissive();

/**
 * Get front material shininess value.
 * @return shininess value for front material.
 */
float OivFrontMaterialShininess();

/**
 * Get front material specular factor value.
 * @return specular factor value for front material.
 */
float OivFrontMaterialSpecularFactor();

/**
 * Get front material roughness value.
 * @return roughness value for front material.
 */
float OivFrontMaterialRoughness();

/**
 * Get front material metallic value.
 * @return metallic value for front material.
 */
float OivFrontMaterialMetallic();

/**
 * Get back material ambient color.
 * @return ambient color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialAmbient();

/**
 * Get back material diffuse color.
 * @return diffuse color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialDiffuse();

/**
 * Get back material specular color.
 * @return specular color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialSpecular();

/**
 * Get back material emissive color.
 * @return emissive color for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackMaterialEmissive();

/**
 * Get back material shininess value.
 * @return shininess value for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
float OivBackMaterialShininess();

/**
 * Get back material specular factor value.
 * @return specular factor value for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
float OivBackMaterialSpecularFactor();

/**
 * Get back material roughness value.
 * @return roughness value for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
float OivBackMaterialRoughness();

/**
 * Get back material metallic value.
 * @return metallic value for back material.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
float OivBackMaterialMetallic();

/**
 * Get sum of emissive component of front material and product
 * between front material ambient and global ambient value.
 * @return sum of emissive component of front material and product
 * between front material ambient and global ambient value.
 */
vec4 OivFrontSceneColor();

/**
 * Get product of ambient component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of ambient component between back material and id'th light.
 */
vec4 OivFrontLightProductAmbient(int id);

/**
 * Get product of diffuse component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of diffuse component between back material and id'th light.
 */
vec4 OivFrontLightProductDiffuse(int id);

/**
 * Get product of specular component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of specular component between back material and id'th light.
 */
vec4 OivFrontLightProductSpecular(int id);

/**
 * Get product of ambient component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of ambient component between back material and id'th light.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackLightProductAmbient(int id);

/**
 * Get product of diffuse component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of diffuse component between back material and id'th light.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackLightProductDiffuse(int id);

/**
 * Get product of specular component between back material and id'th light.
 * @param id specifies which light id to ask, must be between 0 and OivMaxLights()-1.
 * @return product of specular component between back material and id'th light.
 *
 * Note that this parameter is relevant only if two side lighting is enabled (i.e. OivHasTwoSidedLighting() returns true).
 */
vec4 OivBackLightProductSpecular(int id);

/*****************************************************************************/
// Clip planes

/**
 * Ask if clip plane is enabled.
 * @param id specifies which clip plane id to ask, must be between 0 and OivMaxClipPlanes()-1.
 * @return true if clip plane with a given id is enabled, false otherwise.
 */
bool OivClipPlaneEnabled(int id);

/**
 * Get the clip plane coordinates.
 * @param id specifies which clip plane id to ask, must be between 0 and OivMaxClipPlanes()-1.
 * @return the clip plane .
 */
vec4 OivClipPlane(int id);


/*****************************************************************************/
// Fog parameters

/**
 * Get the fog color.
 * @return fog color.
 */
vec4 OivFogColor();


/**
 * Get fog density used in exponential fog equations.
 * @return fog density.
 */
float OivFogDensity();


/**
 * Get the near distance used in the linear fog equation.
 * @return fog start.
 */
float OivFogStart();


/**
 * Get the far distance used in the linear fog equation.
 * @return fog end.
 */
float OivFogEnd();


/**
 * Get the fog scale derived from start and end parameters : 1.0 / (end - start)
 * @return fog scale.
 */
float OivFogScale();

/**
 * Get the fog type
 * @return fog type (FOG_TYPE_NONE, FOG_TYPE_EXP, FOG_TYPE_EXP2 or FOG_TYPE_LINEAR).
 */
int OivFogType();

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneS(int i);

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneT(int i);

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneR(int i);

/**
* Get the user-defined planes for texture coordinate generation
* @return user-defined plane
*/
vec4 OivObjectPlaneQ(int i);

#if defined(VULKAN)
#define OivFrontFacing() (!gl_FrontFacing)
#else
#define OivFrontFacing() gl_FrontFacing
#endif

#endif /* OIV_SHADER_STATE_H */
