/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef VVIZ_STRUCTURE_H
#define VVIZ_STRUCTURE_H

/**
 * @ingroup VolumeVizShadersData
 * Returns volume dimensions of a dataset in voxels.
 * 
 * @param dataset The dataset to retrieve information from.
 * @return dataset Dimension.
 */
vec3 VVizGetVolumeDimensions(in VVizDataSetId dataset);

/**
 * @ingroup VolumeVizShadersData
 * Returns volume dataset tile dimensions in voxels.
 * 
 * @param dataset The dataset to retrieve information from.
 * @return dataset Tile dimension.
 */
vec3 VVizGetTileDimensions(in VVizDataSetId dataset);

/**
 * @ingroup VolumeVizShadersData
 * Returns volume dataset uniform voxel dimensions.
 * 
 * @param dataset The dataset to retrieve information from.
 * @return dataset Voxel uniform dimension.
 */
vec3 VVizGetVoxelDimensions(in VVizDataSetId dataset);

/**
 * @ingroup VolumeVizShadersData
 * Defines internal VolumeViz tile info data storage.
 *
 * See VVizGetTileResolution, VVizGetTileInfo
 */
#define VVizTileInfo vec4

/**
 * @ingroup VolumeVizShadersData 
 * Returns VolumeViz tile resolution info. 
 *
 * @param tileInfo The tileInfo to retrieve information from.
 * @return The LDM resolution level of this tile (zero is full resolution).
 */
int VVizGetTileResolution(in VVizTileInfo tileInfo);

/**
 * @ingroup VolumeVizShadersData
 * Returns VolumeViz tile info for a specified voxel
 *
 * @param dataset The dataset to retrieve information from.
 * @param dataCoord Uniform data coordinates in volume space.
 * @return The tileInfo of the voxel at position dataCoord.
 */
VVizTileInfo VVizGetTileInfo(in VVizDataSetId dataset, in vec3 dataCoord);

/**
 * @ingroup VolumeVizShadersData
 * Returns VolumeViz tile IJK box info for a specified voxel.
 *
 * @param dataset The dataset to which the tileinfo belongs.
 * @param dataCoord Uniform data coordinates in volume space.
 * @param tileIJKMin ijk min of the tile in voxels.
 * @param tileIJKMax ijk max of the tile in voxels.
 */
void VVizGetTileIJKBox(in VVizDataSetId dataset, in vec3 dataCoord, out vec3 tileIJKMin, out vec3 tileIJKMax);

/**
 * @ingroup VolumeVizShadersData
 * Returns the default DataSet ID.
 */
VVizDataSetId VVizGetDefaultDataSet();

/**
 * @ingroup VolumeVizShadersData
 * Returns true if the given data is undefined
 */
bool VVizIsUndefined( VVIZ_DATATYPE data );


/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the input dataset's texture space to object space.
 * Object space corresponds to the space defined by the extent of the dataset.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param texCoord The texture coordinate vector in homogeneous coordinates.
 * @return The converted vector in object space in homogeneous coordinates.
 */
vec4 VVizTextureToObject(in VVizDataSetId dataset, in vec4 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the input dataset's texture space to object space.
 * Object space corresponds to the space defined by the extent of the dataset.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param texCoord The 3D texture coordinate vector, assumed to be a position vector.
 * @return The converted 3D position vector in object space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by dividing the X, Y and Z components by W.
 */
vec3 VVizTextureToObjectVec(in VVizDataSetId dataset, in vec3 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the input dataset's texture space to object space.
 * Object space corresponds to the space defined by the extent of the dataset.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param texCoord The 3D texture coordinate vector, assumed to be a direction vector.
 * @return The converted 3D direction vector in object space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by ignoring the W component.
 */
vec3 VVizTextureToObjectDir(in VVizDataSetId dataset, in vec3 texCoord);


/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from object space to the input dataset's texture space.
 * Object space corresponds to the space defined by the extent of the dataset.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param objectCoord The object space vector, in homogeneous coordinates.
 * @return The converted vector in texture space, in homogeneous coordinates.
 */
vec4 VVizObjectToTexture(in VVizDataSetId dataset, in vec4 objectCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from object space to the input dataset's texture space.
 * Object space corresponds to the space defined by the extent of the dataset.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param objectCoord The 3D object space vector, assumed to be a position vector.
 * @return The converted 3D position vector in the dataset's texture space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by dividing the X, Y and Z components by W.
 */
vec3 VVizObjectToTextureVec(in VVizDataSetId dataset, in vec3 objectCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from object space to the input dataset's texture space.
 * Object space corresponds to the space defined by the extent of the dataset.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param objectCoord The 3D object space vector, assumed to be a direction vector.
 * @return The converted 3D direction vector in the dataset's texture space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by ignoring the W component.
 */
vec3 VVizObjectToTextureDir(in VVizDataSetId dataset, in vec3 objectCoord);


/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the input dataset's texture space to world space.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param texCoord The texture coordinate vector in homogeneous coordinates.
 * @return The converted vector in world space in homogeneous coordinates.
 */
vec4 VVizTextureToWorld(in VVizDataSetId dataset, in vec4 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the input dataset's texture space to world space.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param texCoord The 3D texture coordinate vector, assumed to be a position vector.
 * @return The converted 3D position vector in world space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by dividing the X, Y and Z components by W.
 */
vec3 VVizTextureToWorldVec(in VVizDataSetId dataset, in vec3 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the input dataset's texture space to world space.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param texCoord The 3D texture coordinate vector, assumed to be a direction vector.
 * @return The converted 3D direction vector in world space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by ignoring the W component.
 */
vec3 VVizTextureToWorldDir(in VVizDataSetId dataset, in vec3 texCoord);


/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from world space to the input dataset's texture space.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param worldCoord The world space vector, in homogeneous coordinates.
 * @return The converted vector in texture space, in homogeneous coordinates.
 */
vec4 VVizWorldToTexture(in VVizDataSetId dataset, in vec4 worldCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from world space to the input dataset's texture space.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param worldCoord The 3D world space vector, assumed to be a position vector.
 * @return The converted 3D position vector in the dataset's texture space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by dividing the X, Y and Z components by W.
 */
vec3 VVizWorldToTextureVec(in VVizDataSetId dataset, in vec3 worldCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from world space to the input dataset's texture space.
 *
 * @param dataset The dataset to which the texture space belongs.
 * @param worldCoord The 3D world space vector, assumed to be a direction vector.
 * @return The converted 3D direction vector in the dataset's texture space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by ignoring the W component.
 */
vec3 VVizWorldToTextureDir(in VVizDataSetId dataset, in vec3 worldCoord);


/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the texture space of one dataset to the texture space of another dataset.
 *
 * @param datasetSrc The dataset to which the initial texture space belongs.
 * @param datasetDst The dataset to which the final texture space belongs.
 * @param texCoord The texture coordinate vector in the initial texture space, in homogeneous coordinates.
 * @return The converted vector in the destination texture space, in homogeneous coordinates.
 */
vec4 VVizTextureToTexture(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec4 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the texture space of one dataset to the texture space of another dataset.
 *
 * @param datasetSrc The dataset to which the initial texture space belongs.
 * @param datasetDst The dataset to which the final texture space belongs.
 * @param texCoord The 3D texture coordinate vector, assumed to be a position vector.
 * @return The converted 3D position vector in the destination texture space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by dividing the X, Y and Z components by W.
 */
vec3 VVizTextureToTextureVec(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Converts the input vector from the texture space of one dataset to the texture space of another dataset.
 *
 * @param datasetSrc The dataset to which the initial texture space belongs.
 * @param datasetDst The dataset to which the final texture space belongs.
 * @param texCoord The 3D texture coordinate vector, assumed to be a direction vector.
 * @return The converted 3D direction vector in the destination texture space.
 *         The intermediate homogeneous (vec4) value is converted to 3D by ignoring the W component.
 */
vec3 VVizTextureToTextureDir(in VVizDataSetId datasetSrc, in VVizDataSetId datasetDst, in vec3 texCoord);

/**
 * @ingroup VolumeVizShadersData
 * Defines voxel state info used with the VolumeViz Shader Library API.
 */
struct VVizVoxelInfo
{
  /**
   * Uniform data coordinates in volume space.
   */
  vec3 texCoord;
  /** 
   * Voxel value at texCoord position.
   * It is filled by DATA_COMBINE_FUNCTION stage.
   */
  VVIZ_DATATYPE value;

  float undefinedFactor;

  // PRIVATE members
  // Texture coordinate in Physical Space
  vec3 texCoordPhys;
  // Physical tile information
  // - tileInfo.xyz  = tile's origin coordinate
  // - tileInfo.w    = tile's ldm resolution
  VVizTileInfo tileInfo;
  //Used by VVizEdgeImageSpace and accumulated along the ray
  // - nprData.x = viewspace Z
  // - nprData.y = normal factor
  vec4 nprData;

  // coordinates of sample in volume space.
  // most of the time, equal to texCoord but may be different in case of projection
  // or in voxelized rendering.
  // Difference with texCoord is that texCoord represent where we will access data whereas
  // sample coord represent where is the current sample of raycast.
  vec3 sampleCoord;
};


/**
 * 3D 8 Neighbors storage:
 * values[0] is (0, 0, 0)
 * values[1] is (1, 0, 0)
 * values[2] is (0, 1, 0)
 * values[3] is (1, 1, 0)
 * values[4-7] same but z=1.
 */
struct VVizNeighborsGeneral
{
  vec3 texCoords[8];
  VVIZ_DATATYPE values[8];
  // Coeffs used for linear interpolation (ie: fractionnal part of texcoords)
  vec3 coeffs;
};

#endif
