/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/

#include <Inventor/ViewerComponents/MFC/RenderArea.h>

#include <Inventor/SbViewportRegion.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/ViewerComponents/SoRawStereoParameters.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/components/SoWinGLGraphicDevice.h>
#include <Inventor/components/SoWinGLGraphicConfig.h>

BEGIN_MESSAGE_MAP( RenderArea, CWnd )
ON_WM_PAINT()
ON_WM_ERASEBKGND( )
ON_WM_SIZE()
END_MESSAGE_MAP()

//------------------------------------------------------------------------------
RenderArea::RenderArea()
  : m_renderAreaCore( NULL )
  , m_sceneGraph( NULL )
  , m_color( 0, 0, 0, 0 )
  , m_depth( 1.0f )
  , m_size( 0, 0 )
  , m_glRenderAction( NULL )
  , m_stereo( false )
  , m_clearPolicy( COLORBUFFER_AND_DEPTHBUFFER )
  , m_antialiasingMode( SoSceneManager::NO_ANTIALIASING )
  , m_antialiasingQuality( 0 )
  , m_stillAntialiasingQuality( 0 )
  , m_stillAntialiasingDelay( 0 )
{
}

//------------------------------------------------------------------------------
RenderArea::~RenderArea()
{
  uninitializeGL();

  // remove the ref added in setSceneGraph
  if ( m_sceneGraph != NULL )
    m_sceneGraph->unref();
}

//------------------------------------------------------------------------------
BOOL
RenderArea::PreCreateWindow( CREATESTRUCT& cs )
{
  if ( !CWnd::PreCreateWindow( cs ) )
    return FALSE;

  cs.dwExStyle |= WS_EX_CLIENTEDGE;
  cs.style &= ~WS_BORDER;
  cs.lpszClass = AfxRegisterWndClass( CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS,
                                      ::LoadCursor( NULL, IDC_ARROW ),
                                      reinterpret_cast<HBRUSH>( COLOR_WINDOW + 1 ),
                                      NULL );

  return TRUE;
}

//------------------------------------------------------------------------------
void
RenderArea::setSceneGraph( SoNode* sceneGraph )
{
  // unref previous scene graph
  if ( m_sceneGraph != NULL )
    m_sceneGraph->unref();

  m_sceneGraph = sceneGraph;

  // keep an additional ref to the scene graph, because the scene renderer
  // may be created later.
  if ( m_sceneGraph != NULL )
    m_sceneGraph->ref();

  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setSceneGraph( m_sceneGraph );
}

//------------------------------------------------------------------------------
SoNode*
RenderArea::getSceneGraph() const
{
  return m_sceneGraph;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearPolicy( SiRenderArea::ClearPolicy policy )
{
  m_clearPolicy = policy;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setClearPolicy( m_clearPolicy );
}

//------------------------------------------------------------------------------
SiRenderArea::ClearPolicy
RenderArea::getClearPolicy() const
{
  return m_clearPolicy;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearColor( const SbColorRGBA& color )
{
  m_color = color;
  if (m_renderAreaCore.ptr() != NULL)
    m_renderAreaCore->setClearColor( m_color );
}

//------------------------------------------------------------------------------
SbColorRGBA
RenderArea::getClearColor() const
{
  return m_color;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearDepth( float depth )
{
  m_depth = depth;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setClearDepth( m_depth );
}

//------------------------------------------------------------------------------
float
RenderArea::getClearDepth() const
{
  return m_depth;
}

//------------------------------------------------------------------------------
void
RenderArea::setSize( const SbVec2i32& size )
{
  m_size = size;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setSize( m_size );
}

//------------------------------------------------------------------------------
SbVec2i32
RenderArea::getSize() const
{
  return m_size;
}

//------------------------------------------------------------------------------
void
RenderArea::setGLRenderAction( SoGLRenderAction* glAction )
{
  m_glRenderAction = glAction;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setGLRenderAction( glAction );
}

//------------------------------------------------------------------------------
SoGLRenderAction*
RenderArea::getGLRenderAction() const
{
  return m_glRenderAction;
}

//------------------------------------------------------------------------------
void
RenderArea::setAntialiasingMode( SoSceneManager::AntialiasingMode mode )
{
  m_antialiasingMode = mode;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setAntialiasingMode( mode );
}

//------------------------------------------------------------------------------
SoSceneManager::AntialiasingMode
RenderArea::getAntialiasingMode() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_antialiasingMode;
  return m_renderAreaCore.ptr()->getAntialiasingMode();
}

//------------------------------------------------------------------------------
void
RenderArea::setAntialiasingQuality( float quality )
{
  m_antialiasingQuality = quality;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setAntialiasingQuality( quality );
}

//------------------------------------------------------------------------------
float
RenderArea::getAntialiasingQuality() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_antialiasingQuality;
  return m_renderAreaCore.ptr()->getAntialiasingQuality();
}

//------------------------------------------------------------------------------
void
RenderArea::setStillSuperSamplingQuality( float quality )
{
  m_stillAntialiasingQuality = quality;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setStillSuperSamplingQuality( quality );
}

//------------------------------------------------------------------------------
float
RenderArea::getStillSuperSamplingQuality() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_stillAntialiasingQuality;
  return m_renderAreaCore.ptr()->getStillSuperSamplingQuality();
}

//------------------------------------------------------------------------------
void
RenderArea::setStillSuperSamplingDelay( unsigned int delay )
{
  m_stillAntialiasingDelay = delay;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setStillSuperSamplingDelay( delay );
}

//------------------------------------------------------------------------------
unsigned int
RenderArea::getStillSuperSamplingDelay() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_stillAntialiasingDelay;
  return m_renderAreaCore.ptr()->getStillSuperSamplingDelay();
}

//------------------------------------------------------------------------------
SbEventHandler<SiRenderArea::RenderEventArg&>&
RenderArea::onStartRender()
{
  return m_renderAreaCore->onStartRender();
}

//------------------------------------------------------------------------------
void
RenderArea::initializeGL( HDC deviceContext, bool enableStereo )
{
  // Initialise OpenGL

  // Get a DC for the OpenGL render window
  m_hDC = deviceContext;

  // Get the best graphic configuration
  SoGLGraphicDevice device( m_hDC );
  SoWinGLGraphicDevice winDevice( m_hDC, SoGLGraphicDevice::WINDOW );
  SoGraphicConfig* gc = device.getBestGraphicConfig( winDevice.getDefaultGraphicConfigTemplate() );
  PIXELFORMATDESCRIPTOR pfd = *( (SoWinGLGraphicConfig*) gc )->getPixelFormatDescriptor();

  if ( enableStereo )
    pfd.dwFlags |= PFD_STEREO;

  int nMyPixelFormatID = ChoosePixelFormat( m_hDC, &pfd );

  SetPixelFormat( m_hDC, nMyPixelFormatID, &pfd );

  // Create new SoGLContext
  m_glContext = new SoGLContext( m_hDC, &pfd, true );
  m_glContext->bind();

  // Create the render area core associated with this openGL context
  m_renderAreaCore = new SoRenderAreaCore( m_glContext.ptr() );
  m_renderAreaCore->setSceneGraph( m_sceneGraph );
  setClearPolicy( m_clearPolicy );
  setClearDepth( m_depth );
  setSize( m_size );
  activateStereo( m_stereo );
  setClearColor( m_color );
  if ( m_glRenderAction != NULL )
    m_renderAreaCore->setGLRenderAction( m_glRenderAction );
  m_renderAreaCore->setAntialiasingMode( m_antialiasingMode );
  m_renderAreaCore->setAntialiasingQuality( m_antialiasingQuality );
  m_renderAreaCore->setStillSuperSamplingDelay( m_stillAntialiasingDelay );
  m_renderAreaCore->setStillSuperSamplingQuality( m_stillAntialiasingQuality );
}

//------------------------------------------------------------------------------
void
RenderArea::uninitializeGL()
{
  // Tell windows we don't want to use OpenGL anymore
  wglMakeCurrent( m_hDC, NULL );
  wglDeleteContext( m_hRC );
  m_hDC = NULL;
  m_hRC = NULL;

  if ( m_glContext.ptr() )
  {
    // Release openGL context
    m_glContext->unbind();
    m_glContext = NULL;
  }
}

//------------------------------------------------------------------------------
void
RenderArea::OnPaint()
{
  CPaintDC dc( this );

  if ( m_renderAreaCore.ptr() == NULL )
    return;

  // Render the scene graph.
  const bool renderOK = render();

  // Check if the rendering is made well
  // If not : don't swap the buffer because we can have actifacts in the buffer
  if ( renderOK )
    SwapBuffers( m_hDC );
}

//------------------------------------------------------------------------------
BOOL RenderArea::OnEraseBkgnd(CDC* pDC) 
{
  // Dot not clear the background when resizing
  return TRUE;
}

//------------------------------------------------------------------------------
void
RenderArea::OnSize( UINT nType, int cx, int cy )
{
  CWnd::OnSize( nType, cx, cy );

  // Store width/height to transform mouse location
  m_height = cy;
  m_width = cx;

  if ( m_renderAreaCore.ptr() != NULL )
  {
    m_renderAreaCore->setSize( SbVec2i32( cx, cy ) );
    OnPaint();
  }
}

//------------------------------------------------------------------------------
void
RenderArea::activateStereo( bool activated )
{
  m_stereo = activated;
  if ( m_renderAreaCore.ptr() == NULL )
    return;

  if ( activated )
  {
    // activate raw stereo
    if ( !isRawStereoAvailable() )
    {
      MessageBox( _T( "Stereo buffers are not enabled." ), _T( "Warning" ), MB_ICONWARNING | MB_OK );
    }
    else
    {
      SoRawStereoParameters* params = new SoRawStereoParameters();
      setStereoParameters( params );
      delete params;
      m_renderAreaCore->activateStereo( true );
    }
  }
  else
  {
    m_renderAreaCore->activateStereo( false );
  }
}

//------------------------------------------------------------------------------
bool
RenderArea::isStereoActivated() const
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->isStereoActivated();

  SoDebugError::postWarning("RenderArea::isStereoActivated",
    "Rendering area is not initialized, stereo status cannot be retrieved.");
  return false;
}

//------------------------------------------------------------------------------
void
RenderArea::setStereoParameters(SoStereoParameters* parameters)
{
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setStereoParameters( parameters );
  else
    SoDebugError::postWarning("RenderArea::setStereoParameters",
      "Rendering area is not initialized, stereo parameters cannot be set.");
}

//------------------------------------------------------------------------------
SoStereoParameters*
RenderArea::getStereoParameters() const
{
  if ( m_renderAreaCore.ptr() != NULL )
    return m_renderAreaCore->getStereoParameters();

  SoDebugError::postWarning("RenderArea::getStereoParameters",
    "Rendering area is not initialized, stereo parameters cannot be retrieved.");
  return NULL;
}

//------------------------------------------------------------------------------
bool
RenderArea::isRawStereoAvailable()
{
  // Get pixel format descriptor from device context
  PIXELFORMATDESCRIPTOR pfd;
  DescribePixelFormat( m_hDC, GetPixelFormat(m_hDC), sizeof(PIXELFORMATDESCRIPTOR), &pfd );

  return (pfd.dwFlags & PFD_STEREO);
}

SoRenderAreaCore::RenderStatus
RenderArea::render()
{
  return m_renderAreaCore->render();
}
