
#include <QUrl>
#include <QFileDialog>
#include <QLayout>
#include <QMimeData>
#include <QProgressDialog>
#include <QMessageBox>

#include <Inventor/SoPreferences.h>
#include <Inventor/caches/SoCache.h>
#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

#include <Inventor/ViewerComponents/Qt/Viewers/ViewerExaminer.h>

#include "PreferencesDialog.h"

#include "ui_ViewerExaminer.h"

/**
 * This class shows how to display the progression of the loading of an IV file
 * using the SoInput callbacks.
 */
class QtInput : public SoInput
{
public:
  /**
   * Constructor.
   * It creates and shows the QProgressDialog.
   */
  QtInput( QWidget* parent ) : SoInput()
  {
    m_dialog = new QProgressDialog( "Loading file...", "Abort", 0, 100, parent );
    m_dialog->show();
  }

  /**
   * Destructor.
   * It hides and destroy the QProgressDialog.
   */
  ~QtInput()
  {
    m_dialog->hide();
    delete m_dialog;
  }

  /**
   * This function is called by SoInput when the progression
   * changes.
   */
  virtual void updateReadPercent( double readPercentage )
  {
    m_dialog->setValue( int(readPercentage) );
  }

private:
  /** Instance of the QProgressDialog used to display the loading progression */
  QProgressDialog* m_dialog;
};


//------------------------------------------------------------------------------
ViewerExaminer::ViewerExaminer( QWidget* parent )
  : QWidget( parent )
  , m_sceneGraph(NULL)
{
  m_ui = new Ui::ViewerExaminer();
  m_ui->setupUi( this );

  setAcceptDrops( true );

  // build an examiner render area with stereo support
  m_renderArea = new RenderAreaExaminer( this );

  if ( !m_renderArea->isRawStereoAvailable() )
    qWarning("Could not enable stereo buffers");

  // add the render area
  m_ui->m_mainHorizLayout->addWidget( m_renderArea->getContainerWidget() );

  connect( m_ui->m_viewAllButton, SIGNAL( clicked() ), (ViewerExaminer*)this, SLOT( viewAll() ) );

  // predefined views YZ, XZ, XY
  connect( m_ui->m_yzViewButton, SIGNAL( clicked() ), this, SLOT( setYZView() ) );
  connect( m_ui->m_xzViewButton, SIGNAL( clicked() ), this, SLOT( setXZView() ) );
  connect( m_ui->m_xyViewButton, SIGNAL( clicked() ), this, SLOT( setXYView() ) );

  // Connect with function to save camera
  connect( m_ui->m_saveCameraButton, SIGNAL( clicked() ), this, SLOT( saveCamera() ) );

  // Connect with function to restore camera
  connect( m_ui->m_restoreCameraButton, SIGNAL( clicked() ), this, SLOT( restoreCamera() ) );

  // Connect with function to perspective camera
  connect( m_ui->m_viewPerspectButton, SIGNAL( clicked() ), this, SLOT( setCameraPerspective() ) );

  // Connect with function to orthographic camera
  connect( m_ui->m_viewOrthoButton, SIGNAL( clicked() ), this, SLOT( setCameraOrtho() ) );

  // Connect with function to activate/deactivate seek mode
  connect( m_ui->m_seekButton, SIGNAL( clicked() ), this, SLOT( setSeekMode() ) );

  // Connect with function to open preferences dialog
  connect( m_ui->m_preferencesButton, SIGNAL( clicked() ), this, SLOT( openPreferences() ) );
  
}

//------------------------------------------------------------------------------
ViewerExaminer::~ViewerExaminer()
{
  delete m_renderArea;
  delete m_ui;
}

//------------------------------------------------------------------------------
void
ViewerExaminer::loadFile( QString filename )
{
  SbString fname;

  // We do this trick according to the documentation of SbString
  // in order to conserve Unicode filenames.
  fname.fromUtf16( filename.utf16() );

  // We use our nice class to display a progress dialog.
  QtInput input( this );

  // We try to open the file
  if (input.openFile( fname ))
  {
    // We read the content.
    m_sceneGraph = SoDB::readAll( &input );
    m_renderArea->setSceneGraph(m_sceneGraph);
    m_renderArea->viewAll(SbViewportRegion());
  }
  else
    QMessageBox::critical(this, tr("QtRenderAreaGuiIntegration"),
                   tr("Error reading the IV file."),
                   QMessageBox::Ok,
                   QMessageBox::Ok);
}

//------------------------------------------------------------------------------
void
ViewerExaminer::setSceneGraph( SoNode* sceneGraph )
{
  m_sceneGraph = sceneGraph;
  m_renderArea->setSceneGraph(m_sceneGraph);
}

//------------------------------------------------------------------------------
RenderAreaExaminer*
ViewerExaminer::getRenderArea()
{
  return m_renderArea;
}

//------------------------------------------------------------------------------
void
ViewerExaminer::viewAll()
{
  m_renderArea->viewAll(SbViewportRegion());
}

void
ViewerExaminer::setYZView()
{
  viewAxis( SbVec3f(-1.f, 0.f, 0.f), SbVec3f(0, 0, 1) );
  viewAll();
}

void
ViewerExaminer::setXZView()
{
  viewAxis( SbVec3f(0.f, -1.f, 0.f), SbVec3f(1, 0, 0) );
  viewAll();
}

//------------------------------------------------------------------------------
void
ViewerExaminer::setXYView()
{
  viewAxis( SbVec3f(0.f, 0.f, -1.f), SbVec3f(0, 1, 0) );
  viewAll();
}

//------------------------------------------------------------------------------
void
ViewerExaminer::saveCamera()
{
  m_renderArea->getSceneInteractor()->getCameraInteractor()->pushCamera();
}

//------------------------------------------------------------------------------
void
ViewerExaminer::restoreCamera()
{
  m_renderArea->getSceneInteractor()->getCameraInteractor()->popCamera();
}

//------------------------------------------------------------------------------
void
ViewerExaminer::setCameraPerspective()
{
  setCameraType( SoPerspectiveCamera::getClassTypeId() );
  if ( isStereoSupported() && isRawStereoAvailable() )
  {
    // enable stereo
    PreferencesDialog::getInstance( this )->enableStereoPreferences(true);
  }
}

//------------------------------------------------------------------------------
void
ViewerExaminer::setCameraOrtho()
{
  setCameraType( SoOrthographicCamera::getClassTypeId() );
  // disable stereo
  PreferencesDialog::getInstance( this )->enableStereoPreferences(false);
}

//------------------------------------------------------------------------------
void
ViewerExaminer::setSeekMode()
{
  m_renderArea->getSceneInteractor()->setSeekMode( true );
}

//------------------------------------------------------------------------------
void
ViewerExaminer::openPreferences()
{
  PreferencesDialog* dialog = PreferencesDialog::getInstance( this );
  if ( !isStereoSupported() || !isRawStereoAvailable() )
  {
    // disable stereo
    PreferencesDialog::getInstance( this )->enableStereoPreferences(false);
  }
  dialog->show();
  dialog->raise();
}

void
ViewerExaminer::setNavigationMode( SceneExaminer::NavigationMode mode )
{
  m_renderArea->setNavigationMode( mode );
}

SceneExaminer::NavigationMode
ViewerExaminer::getNavigationMode()
{
  return m_renderArea->getNavigationMode();
}

RenderAreaInteractive::ClippingPlanesAdjustMode
ViewerExaminer::getClippingPlanesAdjustMode()
{
  return m_renderArea->getClippingPlanesAdjustMode();
}

void
ViewerExaminer::setClippingPlanesAdjustMode(RenderAreaInteractive::ClippingPlanesAdjustMode mode)
{
  m_renderArea->setClippingPlanesAdjustMode(mode);
}

//------------------------------------------------------------------------------
void
ViewerExaminer::viewAxis( const SbVec3f& direction, const SbVec3f& up )
{
  m_renderArea->viewAxis( direction, up );
}

//------------------------------------------------------------------------------
void
ViewerExaminer::setCameraType(SoType type)
{
  // How to set the camera type
  if ( type == SoOrthographicCamera::getClassTypeId() )
    m_renderArea->getSceneInteractor()->setCameraMode(SceneInteractor::ORTHOGRAPHIC);
  else
    m_renderArea->getSceneInteractor()->setCameraMode(SceneInteractor::PERSPECTIVE);
}

void
ViewerExaminer::activateStereo( bool activated )
{
  m_renderArea->activateStereo( activated );
}

bool
ViewerExaminer::isRawStereoAvailable()
{
  return m_renderArea->isRawStereoAvailable();
}

bool
ViewerExaminer::isStereoSupported()
{
  return m_renderArea->isStereoSupported();
}

void
ViewerExaminer::setStereoCameraOffset( float offset )
{
  m_renderArea->setStereoCameraOffset( offset );
}

void
ViewerExaminer::setStereoCameraBalance( float balance )
{
  m_renderArea->setStereoCameraBalance( balance );
}
