#include <Inventor/ViewerComponents/Win/RenderArea.h>

#include <Inventor/SbViewportRegion.h>
#include <Inventor/ViewerComponents/SoRenderAreaCore.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/components/SoWinGLGraphicDevice.h>
#include <Inventor/components/SoWinGLGraphicConfig.h>

std::map<HWND, RenderArea*> RenderArea::m_hWndToRenderAreaMap;

//------------------------------------------------------------------------------
LRESULT CALLBACK WndProc( HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam )
{
  RenderArea* renderArea = RenderArea::getRenderArea( hWnd );

  if ( renderArea != NULL )
    renderArea->processEvents( uMsg, wParam, lParam );

  return DefWindowProc( hWnd, uMsg, wParam, lParam );
}

//------------------------------------------------------------------------------
RenderArea*
RenderArea::getRenderArea( HWND hWnd )
{
  if ( m_hWndToRenderAreaMap.count( hWnd ) == 0 )
    return NULL;

  return m_hWndToRenderAreaMap[hWnd];
}

//------------------------------------------------------------------------------
RenderArea::RenderArea()
  : m_renderAreaCore( NULL )
  , m_sceneGraph( NULL )
  , m_color( 0, 0, 0, 0 )
  , m_depth( 1.0f )
  , m_size( 0, 0 )
  , m_glRenderAction( NULL )
  , m_hInstance( NULL )
  , m_hDC( NULL )
  , m_hWnd( NULL )
  , m_parentHWND( NULL )
  , m_active( TRUE )
  , m_width( 0 )
  , m_height( 0 )
  , m_clearPolicy( COLORBUFFER_AND_DEPTHBUFFER )
  , m_antialiasingMode( SoSceneManager::NO_ANTIALIASING )
  , m_antialiasingQuality( 0 )
  , m_stillAntialiasingQuality( 0 )
  , m_stillAntialiasingDelay( 0 )
{
}

//------------------------------------------------------------------------------
RenderArea::RenderArea(HWND parentHWND)
  : m_renderAreaCore( NULL )
  , m_sceneGraph( NULL )
  , m_color( 0, 0, 0, 0 )
  , m_depth( 1.0f )
  , m_size( 0, 0 )
  , m_glRenderAction( NULL )
  , m_hInstance( NULL )
  , m_hDC( NULL )
  , m_hWnd( NULL )
  , m_parentHWND( parentHWND )
  , m_active( TRUE )
  , m_width( 0 )
  , m_height( 0 )
  , m_clearPolicy( COLORBUFFER_AND_DEPTHBUFFER )

{
}

//------------------------------------------------------------------------------
RenderArea::~RenderArea()
{
  uninitializeGL();

  // remove the ref added in setSceneGraph
  if ( m_sceneGraph != NULL )
    m_sceneGraph->unref();
}

//------------------------------------------------------------------------------
void
RenderArea::setSceneGraph( SoNode* sceneGraph )
{
  // unref previous scene graph
  if ( m_sceneGraph != NULL )
    m_sceneGraph->unref();

  m_sceneGraph = sceneGraph;

  // keep an additional ref to the scene graph, because the scene renderer
  // may be created later.
  if ( m_sceneGraph != NULL )
    m_sceneGraph->ref();

  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setSceneGraph( m_sceneGraph );
}

//------------------------------------------------------------------------------
SoNode*
RenderArea::getSceneGraph() const
{
  return m_sceneGraph;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearPolicy( SiRenderArea::ClearPolicy policy )
{
  m_clearPolicy = policy;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setClearPolicy( m_clearPolicy );
}

//------------------------------------------------------------------------------
SiRenderArea::ClearPolicy
RenderArea::getClearPolicy() const
{
  return m_clearPolicy;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearColor( const SbColorRGBA& color )
{
  m_color = color;
  if (m_renderAreaCore.ptr() != NULL)
    m_renderAreaCore->setClearColor( m_color );
}

//------------------------------------------------------------------------------
SbColorRGBA
RenderArea::getClearColor() const
{
  return m_color;
}

//------------------------------------------------------------------------------
void
RenderArea::setClearDepth( float depth )
{
  m_depth = depth;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setClearDepth( m_depth );
}

//------------------------------------------------------------------------------
float
RenderArea::getClearDepth() const
{
  return m_depth;
}

//------------------------------------------------------------------------------
void
RenderArea::setSize( const SbVec2i32& size )
{
  m_size = size;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setSize( m_size );
}

//------------------------------------------------------------------------------
SbVec2i32
RenderArea::getSize() const
{
  return m_size;
}

//------------------------------------------------------------------------------
void
RenderArea::setGLRenderAction( SoGLRenderAction* glAction )
{
  m_glRenderAction = glAction;
  if ( m_renderAreaCore.ptr() != NULL )
    m_renderAreaCore->setGLRenderAction( glAction );
}

//------------------------------------------------------------------------------
SoGLRenderAction*
RenderArea::getGLRenderAction() const
{
  return m_glRenderAction;
}

//------------------------------------------------------------------------------
void
RenderArea::setAntialiasingMode( SoSceneManager::AntialiasingMode mode )
{
  m_antialiasingMode = mode;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setAntialiasingMode( mode );
}

//------------------------------------------------------------------------------
SoSceneManager::AntialiasingMode
RenderArea::getAntialiasingMode() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_antialiasingMode;
  return m_renderAreaCore.ptr()->getAntialiasingMode();
}

//------------------------------------------------------------------------------
void
RenderArea::setAntialiasingQuality( float quality )
{
  m_antialiasingQuality = quality;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setAntialiasingQuality( quality );
}

//------------------------------------------------------------------------------
float
RenderArea::getAntialiasingQuality() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_antialiasingQuality;
  return m_renderAreaCore.ptr()->getAntialiasingQuality();
}

//------------------------------------------------------------------------------
void
RenderArea::setStillSuperSamplingQuality( float quality )
{
  m_stillAntialiasingQuality = quality;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setStillSuperSamplingQuality( quality );
}

//------------------------------------------------------------------------------
float
RenderArea::getStillSuperSamplingQuality() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_stillAntialiasingQuality;
  return m_renderAreaCore.ptr()->getStillSuperSamplingQuality();
}

//------------------------------------------------------------------------------
void
RenderArea::setStillSuperSamplingDelay( unsigned int delay )
{
  m_stillAntialiasingDelay = delay;
  if ( m_renderAreaCore.ptr() != nullptr )
    m_renderAreaCore->setStillSuperSamplingDelay( delay );
}

//------------------------------------------------------------------------------
unsigned int
RenderArea::getStillSuperSamplingDelay() const
{
  if ( m_renderAreaCore.ptr() == nullptr )
    return m_stillAntialiasingDelay;
  return m_renderAreaCore.ptr()->getStillSuperSamplingDelay();
}

//------------------------------------------------------------------------------
SbEventHandler<SiRenderArea::RenderEventArg&>&
RenderArea::onStartRender()
{
  return m_renderAreaCore->onStartRender();
}

//------------------------------------------------------------------------------
SoRenderAreaCore::RenderStatus
RenderArea::render()
{
  return m_renderAreaCore->render();
}

//------------------------------------------------------------------------------
void
RenderArea::initializeGL( int width, int height )
{
  GLuint PixelFormat;
  WNDCLASS wc;
  DWORD dwExStyle;
  DWORD dwStyle;

  // Specify window rectangle
  RECT WindowRect;
  WindowRect.left = (long) 0;
  WindowRect.right = (long) width;
  WindowRect.top = (long) 0;
  WindowRect.bottom = (long) height;

  // Register the window class
  m_hInstance = GetModuleHandle( NULL );
  wc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
  wc.lpfnWndProc = (WNDPROC) WndProc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = m_hInstance;
  wc.hIcon = LoadIcon( NULL, IDI_WINLOGO );
  wc.hCursor = NULL;
  wc.hbrBackground = NULL;
  wc.lpszMenuName = NULL;
  wc.lpszClassName = "OpenGL";

  if ( !RegisterClass( &wc ) )
  {
    MessageBox( NULL, "Failed To Register The Window Class.", "ERROR", MB_OK | MB_ICONEXCLAMATION );
    return;
  }

  // Set size and style to the window
  dwExStyle = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
  dwStyle = WS_OVERLAPPEDWINDOW;

  AdjustWindowRectEx( &WindowRect, dwStyle, FALSE, dwExStyle );

  // Create the window
  if ( !( m_hWnd = CreateWindowEx( dwExStyle,
                                   "OpenGL",
                                   "RenderArea",
                                   dwStyle | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
                                   0,
                                   0,
                                   WindowRect.right - WindowRect.left,
                                   WindowRect.bottom - WindowRect.top,
                                   m_parentHWND,
                                   NULL,
                                   m_hInstance,
                                   NULL ) ) )
  {
    MessageBox( NULL, "Window Creation Error.", "ERROR", MB_OK | MB_ICONEXCLAMATION );
    return;
  }

  // Associate HWND to this render area
  m_hWndToRenderAreaMap[m_hWnd] = this;

  // Catch touch events on this HWND
  RegisterTouchWindow( m_hWnd, 0 );

  if ( !( m_hDC = GetDC( m_hWnd ) ) )
  {
    MessageBox( NULL, "Can't Create A GL Device Context.", "ERROR", MB_OK | MB_ICONEXCLAMATION );
    return;
  }

  // Get the best graphic configuration
  SoGLGraphicDevice device( m_hDC );
  SoWinGLGraphicDevice winDevice( m_hDC, SoGLGraphicDevice::WINDOW );
  PIXELFORMATDESCRIPTOR pfd;
  SoGraphicConfig* gc = device.getBestGraphicConfig( winDevice.getDefaultGraphicConfigTemplate() );
  pfd = *( (SoWinGLGraphicConfig*) gc )->getPixelFormatDescriptor();

  // Set the pixel format
  if ( !( PixelFormat = ChoosePixelFormat( m_hDC, &pfd ) ) )
  {
    MessageBox( NULL, "Can't Find A Suitable PixelFormat.", "ERROR", MB_OK | MB_ICONEXCLAMATION );
    return;
  }

  if ( !SetPixelFormat( m_hDC, PixelFormat, &pfd ) )
  {
    MessageBox( NULL, "Can't Set The PixelFormat.", "ERROR", MB_OK | MB_ICONEXCLAMATION );
    return;
  }

  // Create new SoGLContext
  SoRef<SoGLContext> glContext = new SoGLContext( m_hDC, &pfd, true );

  // Create the render area core associated with this openGL context
  m_renderAreaCore = new SoRenderAreaCore( glContext.ptr() );
  m_renderAreaCore->setSceneGraph( m_sceneGraph );
  // Set all properties kept while m_renderAreaCore was not initialized
  if ( m_size == SbVec2i32( 0, 0 ) )
    m_size = SbVec2i32( width, height );
  setClearPolicy( m_clearPolicy );
  setClearDepth( m_depth );
  setSize( m_size );
  setClearColor( m_color );
  m_renderAreaCore->setAntialiasingMode( m_antialiasingMode );
  m_renderAreaCore->setAntialiasingQuality( m_antialiasingQuality );
  m_renderAreaCore->setStillSuperSamplingDelay( m_stillAntialiasingDelay );
  m_renderAreaCore->setStillSuperSamplingQuality( m_stillAntialiasingQuality );

  // Display window
  ShowWindow( m_hWnd, SW_SHOW );
  SetForegroundWindow( m_hWnd );
  SetFocus( m_hWnd );

  // Resize OpenGL rendering
  resizeGL( width, height );
  if ( m_glRenderAction != NULL )
    m_renderAreaCore->setGLRenderAction( m_glRenderAction );
}

//------------------------------------------------------------------------------
void
RenderArea::uninitializeGL()
{
  m_renderAreaCore = NULL;
}

//------------------------------------------------------------------------------
void
RenderArea::paintGL()
{
  // Render the scene graph.
  const bool renderOK = render();

  // Check if the rendering is made well
  // If not : don't swap the buffer because we can have actifacts in the buffer
  if ( renderOK )
    SwapBuffers( m_hDC );
}

//------------------------------------------------------------------------------
void
RenderArea::resizeGL( int width, int height )
{
  // Store width/height to transform mouse location
  m_width = width;
  m_height = height;

  if ( m_renderAreaCore.ptr() != NULL )
  {
    // Update the viewport of the render area
    SbViewportRegion viewportRegion( width, height );
    m_renderAreaCore->setSize(SbVec2i32( width, height ));
    paintGL();
  }
}

//------------------------------------------------------------------------------
void
RenderArea::processEvents( UINT uMsg, WPARAM wParam, LPARAM lParam )
{
  switch ( uMsg )
  {
  case WM_ACTIVATE:
  {
    // Window activated or not
    if ( !HIWORD( wParam ) )
    {
      m_active = TRUE;
    }
    else
    {
      m_active = FALSE;
    }
    break;
  }

  case WM_DESTROY:
  {
    uninitializeGL();
    // Send message to quit the application
    PostQuitMessage( 0 );
    break;
  }

  case WM_SIZE:
  {
    // Size changed so inform OpenGL
    resizeGL( LOWORD( lParam ), HIWORD( lParam ) );
    break;
  }
  }
}
