#pragma once

#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <Inventor/SbViewportRegion.h>
#include "Animator.h"

/**
 * Orbit Animation.
 *
 * This class allows animating a camera by moving it around the current focus point.
 * This point is specified by the #setStartPoint function.
 * The speed of the rotation is computed with the last 3 rotations passed to #setRotation
 */
class OrbitAnimator : public Animator
{
public:
  OrbitAnimator( SoCameraInteractor* cameraInteractor );

  /**
   * Adds a screen point of projection to the samples
   * This point is used to compute the direction and speed of the rotation.
   * @param screenPosition an on-screen position of the cursor that this animator will translate into a rotation
   */
  void addScreenPoint( const SbVec2f& screenPosition );

  /**
   * Sets the original point of animation, and stops any existing rotation by re-initializing the samples to an empty
   * rotation.
   * @param initialOrbitPosition initializes the Sphere Sheet projection
   */
  void setStartPoint( const SbVec2f& initialOrbitPosition );

  /**
  * Returns the original point of animation
  */
  SbVec2f startPoint() const;

  /**
  * Sets the scene graph.
  */
  void setSceneGraph(SoNode* sceneGraph);

  /**
  * Sets the viewport region.
  */
  void setViewportRegion(const SbViewportRegion& vpRegion);

protected:
  /**
   * overload of Animator::animate(), modifies the orbit of the camera interactor
   */
  void animate();

private:
  // a camera interactor to modify the camera (in this case, orbit)
  SoCameraInteractor* m_cameraInteractor;
  // the rotation by which the camera will orbit at each animation cycle
  SbRotation m_rotation;
  // a projector to transform screen coordinates into world spherical coordinates
  SbSphereSheetProjector m_sphereProjector;
  // a list of samples used to compute the direction and speed of the rotation
  // this list only keeps a finite amount of samples, and overwrites the oldest one
  std::vector< SbRotation > m_rotationSamples;
  // the current sample, used to keep track of which one should be overwritten next
  size_t m_currentRotationSampleIndex;
  // recompute the rotation from the samples
  void updateRotation();

  SbVec2f m_startPoint;

  SoNode* m_sceneGraph;

  SbViewportRegion m_vpRegion;

  const bool m_isAutoClippingPlanes;
};
