#include <Inventor/ViewerComponents/nodes/SceneExaminer.h>

#include <cmath>
#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/touch/events/SoTouchEvent.h>
#include <Inventor/gestures/events/SoScaleGestureEvent.h>
#include <Inventor/gestures/events/SoRotateGestureEvent.h>
#include <Inventor/gestures/events/SoDoubleTapGestureEvent.h>
#include <Inventor/touch/SoTouchManager.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/sensors/SoAlarmSensor.h>
#include <Inventor/sensors/SoOneShotSensor.h>

#include "AnimatorListener.h"
#include "OrbitAnimator.h"
#include "SeekAnimator.h"
#include <Inventor/ViewerComponents/nodes/SiInteractionModeListener.h>
#include <algorithm>

class BaseInteraction
{
public:
  virtual void beginAction( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action ) = 0;
  virtual void beginAction( SoKeyboardEvent* keyEvent, SoHandleEventAction* action ) = 0;

  virtual void doAction( SoLocation2Event* mouseEvent, SoHandleEventAction* action ) = 0;
  virtual void doAction( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action ) = 0;
  virtual void doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action ) = 0;
  virtual void doAction( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action ) = 0;
  virtual void doAction( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action ) = 0;
  virtual void doAction( SoDoubleTapGestureEvent* doubleTapEvent, SoHandleEventAction* action ) = 0;
  virtual void doAction( SoLongTapGestureEvent* longTapEvent, SoHandleEventAction* action ) = 0;

  virtual bool propagateEvents() const { return false; };

protected:
  BaseInteraction() {};
  virtual ~BaseInteraction() {};

};

class NavigationInteraction : public BaseInteraction
{
  friend class SeekAnimatorListener;
  friend class OrbitAnimatorListener;

public:
  NavigationInteraction( SoCameraInteractor* cameraInteractor );
  NavigationInteraction( const NavigationInteraction& copyFrom );
  virtual ~NavigationInteraction();

  virtual void beginAction( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void beginAction( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );

  virtual void doAction( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action );
  virtual void doAction( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action );
  virtual void doAction( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action );
  virtual void doAction( SoDoubleTapGestureEvent* doubleTapEvent, SoHandleEventAction* action );
  virtual void doAction( SoLongTapGestureEvent* longTapEvent, SoHandleEventAction* action );

  void reset();
  void setCameraInteractor( SoCameraInteractor* cameraInteractor );
  void setSeekMode( bool onOrOff );

  void addInteractionModeListener( SiInteractionModeListener* listener );
  void removeInteractionModeListener( SiInteractionModeListener* listener );

  bool m_isZoomEnabled;
  bool m_isPanEnabled;
  bool m_isOrbitEnabled;
  bool m_isRotateEnabled;
  bool m_isSeekEnabled;
  bool m_isSpinEnabled;

protected:
  virtual void beginAction( SoHandleEventAction* action, bool ctrlDown, bool shiftDown ) = 0;

  void beginPan( SoHandleEventAction* action );
  void beginOrbit( SoHandleEventAction* action );
  void doPan( SoEvent* anyEvent, SoHandleEventAction* action );
  void doOrbit( SoEvent* anyEvent, SoHandleEventAction* action );
  void doDolly( SoEvent* anyEvent, SoHandleEventAction* action );
  void doDollyWithCenter( const SbVec2f& center, float scaleFactor, SoHandleEventAction* action );
  void doRoll( SoEvent* anyEvent, SoHandleEventAction* action );
  void doTranslate( SoTouchEvent* touchEvent, SoHandleEventAction* action );
  void doRotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action );

  void interactiveCountInc();
  void interactiveCountDec();

  SoCameraInteractor* m_cameraInteractor;

  bool m_isSeekMode;
  SeekAnimator* m_seekAnimator;
  OrbitAnimator* m_orbitAnimator;

  bool m_isButton1Down;
  bool m_isButton2Down;
  SbVec2f m_mousePositionNorm;
  int m_mouseWheelDelta;

private:
  void doSeek( SoEvent* anyEvent, SoHandleEventAction* action );
  void startInteraction();

  int m_interactiveCount;
  bool m_isAutoClippingPlanes;
  SoHandleEventAction* m_currentAction;
  SoAlarmSensor* m_interactionEndSensor;
  std::vector<SiInteractionModeListener*> m_interactionModeListeners;
  // waiting time for a new interaction
  static const double s_interaction_time;
};

class OrbitInteraction : public NavigationInteraction
{
public:
  OrbitInteraction( SoCameraInteractor* cameraInteractor );
  OrbitInteraction( const NavigationInteraction& copyFrom );
  ~OrbitInteraction() {}

  virtual void doAction( SoLocation2Event* mouseEvent, SoHandleEventAction* action );
  virtual void doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action );

protected:
  virtual void beginAction( SoHandleEventAction* action, bool ctrlDown, bool shiftDown );

private:
  bool m_isTouchOrbitActivated;
};

class PlaneInteraction : public NavigationInteraction
{
public:
  PlaneInteraction( SoCameraInteractor* cameraInteractor );
  PlaneInteraction( const NavigationInteraction& copyFrom );
  ~PlaneInteraction() {}

  virtual void doAction( SoLocation2Event* mouseEvent, SoHandleEventAction* action );
  virtual void doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action );

protected:
  virtual void beginAction( SoHandleEventAction* action, bool ctrlDown, bool shiftDown );
};

class SelectionInteraction : public BaseInteraction
{
public:
  SelectionInteraction() {};
  ~SelectionInteraction() {}

  virtual void doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action );

  virtual void beginAction( SoMouseButtonEvent*, SoHandleEventAction* ) {}
  virtual void beginAction( SoKeyboardEvent*, SoHandleEventAction* ) {}

  virtual void doAction( SoLocation2Event*, SoHandleEventAction* );
  virtual void doAction( SoMouseWheelEvent*, SoHandleEventAction* );
  virtual void doAction( SoScaleGestureEvent*, SoHandleEventAction* );
  virtual void doAction( SoRotateGestureEvent*, SoHandleEventAction* );
  virtual void doAction( SoDoubleTapGestureEvent*, SoHandleEventAction* );
  virtual void doAction( SoLongTapGestureEvent*, SoHandleEventAction* );

  virtual bool propagateEvents() const { return true; };

private:
  // given an SoTouchEvent convert it into MouseButton or Location2Event
  SoEvent* convertTouchEvent( SoTouchEvent* touchEvent );

  SoMouseButtonEvent m_touchMouseButtonEvent;
  SoLocation2Event m_touchLocation2Event;
};


class SeekAnimatorListener : public AnimatorListener
{
public:
  SeekAnimatorListener( NavigationInteraction* interaction )
    : m_interaction( interaction)
  {}
  ~SeekAnimatorListener(){}

  virtual void animationStarted()
  {
    m_interaction->interactiveCountInc();
  }

  virtual void animationStopped()
  {
    m_interaction->interactiveCountDec();
    m_interaction->setSeekMode( false );
  }

private:
  NavigationInteraction* m_interaction;
};

class OrbitAnimatorListener : public AnimatorListener
{
public:
  OrbitAnimatorListener( NavigationInteraction* interaction )
    : m_interaction( interaction )
  {}
  ~OrbitAnimatorListener() {}

  virtual void animationStarted()
  {
    m_interaction->interactiveCountInc();
  }

  virtual void animationStopped()
  {
    m_interaction->interactiveCountDec();
  }

private:
  NavigationInteraction* m_interaction;
};

SceneExaminer::SceneExaminer()
  : SceneInteractor()
  , m_isSelectionEnabled(true)
{
  m_currentInteraction = m_navigation = new OrbitInteraction(m_cameraInteractor);
  m_selection = new SelectionInteraction();

  //Sensor to handle event posting at mode switch
  m_modeSwitchDelayer = new SoOneShotSensor;
  m_modeSwitchDelayer->setFunction([](void* data, SoSensor* /*sensor*/)
  {
    // In order to mimic a "viewer switch mode" event, a SoLocation2Event::EventSource::OTHER is used instead.
    // This a tip to propagate this information through the scene graph.
    SoLocation2Event event;
    event.setEventSource(SoLocation2Event::EventSource::OTHER);
    static_cast<SoSceneManager*>(data)->processEvent(&event);
  });
}

SceneExaminer::~SceneExaminer()
{
  delete m_navigation;
  delete m_selection;
  delete m_modeSwitchDelayer;
}

void
SceneExaminer::enableSelection(bool enabled)
{
  m_isSelectionEnabled = enabled;
}

bool
SceneExaminer::isSelectionEnabled()
{
  return m_isSelectionEnabled;
}

void
SceneExaminer::enableZoom(bool enabled)
{
  m_navigation->m_isZoomEnabled = enabled;
}

bool
SceneExaminer::isZoomEnabled()
{
  return m_navigation->m_isZoomEnabled;
}

void
SceneExaminer::enablePan(bool enabled)
{
  m_navigation->m_isPanEnabled = enabled;
}

bool
SceneExaminer::isPanEnabled()
{
  return m_navigation->m_isPanEnabled;
}

void
SceneExaminer::enableOrbit(bool enabled)
{
  m_navigation->m_isOrbitEnabled = enabled;
}

bool
SceneExaminer::isOrbitEnabled()
{
  return m_navigation->m_isOrbitEnabled;
}

void
SceneExaminer::enableRotate(bool enabled)
{
  m_navigation->m_isRotateEnabled = enabled;
}

bool
SceneExaminer::isRotateEnabled()
{
  return m_navigation->m_isRotateEnabled;
}

void
SceneExaminer::enableSeek(bool enabled)
{
  m_navigation->m_isSeekEnabled = enabled;
}

bool
SceneExaminer::isSeekEnabled()
{
  return m_navigation->m_isSeekEnabled;
}

void
SceneExaminer::setCameraMode( SceneInteractor::CameraMode mode )
{
  SceneInteractor::setCameraMode( mode );

  if ( m_navigation != NULL )
    m_navigation->setCameraInteractor( m_cameraInteractor );
}

void
SceneExaminer::setNavigationMode(NavigationMode mode)
{
  NavigationInteraction* m_oldViewing = m_navigation;

  switch (mode)
  {
  case SceneExaminer::ORBIT:
    m_navigation = new OrbitInteraction(*m_navigation);
    break;
  case SceneExaminer::PLANE:
    m_navigation = new PlaneInteraction(*m_navigation);
    break;
  default:
    break;
  }

  // install the interaction mode listeners onto the new navigation interaction
  for ( std::vector< SiInteractionModeListener* >::const_iterator it = m_interactionModeListeners.begin();
        it != m_interactionModeListeners.end();
        ++it )
  {
    m_navigation->addInteractionModeListener( *it );
  }

  if ( getInteractionMode() == NAVIGATION )
    m_currentInteraction = m_navigation;
  delete m_oldViewing;
}

bool
SceneExaminer::isSpinEnabled() const
{
  return m_navigation->m_isSpinEnabled;
}

void
SceneExaminer::enableSpin( bool enable )
{
  m_navigation->m_isSpinEnabled = enable;
}

SceneExaminer::NavigationMode
SceneExaminer::getNavigationMode()
{
  if ( dynamic_cast<OrbitInteraction*>(m_navigation) != NULL )
    return ORBIT;

  return PLANE;
}

SceneExaminer::InteractionMode
SceneExaminer::getInteractionMode()
{
  if ( dynamic_cast<SelectionInteraction*>(m_currentInteraction) != NULL )
    return SELECTION;

  return NAVIGATION;
}

void
SceneExaminer::setInteractionMode(SceneExaminer::InteractionMode mode)
{
  if ( getInteractionMode() != mode )
  {
    switch (mode)
    {
    case SceneExaminer::NAVIGATION:
      m_currentInteraction = m_navigation;
      break;
    case SceneExaminer::SELECTION:
      m_navigation->reset();
      m_currentInteraction = m_selection;
      break;
    default:
      break;
    }

    for( std::vector<SiInteractionModeListener*>::iterator listener = m_interactionModeListeners.begin(); listener != m_interactionModeListeners.end(); ++listener  )
    {
      (*listener)->interactionModeChanged( mode );
    }

  }
}

void
SceneExaminer::switchInteractionMode(SoHandleEventAction* action)
{
  if ( getInteractionMode() == SELECTION )
  {
    m_modeSwitchDelayer->setData(action->getSceneManager());
    m_modeSwitchDelayer->schedule();
    setInteractionMode(NAVIGATION);
  }
  else if ( m_isSelectionEnabled )
    setInteractionMode(SELECTION);
}

void
SceneExaminer::setSeekMode( bool onOrOff )
{
  if ( getInteractionMode() == NAVIGATION )
  {
    m_navigation->setSeekMode(onOrOff);
  }
}

/** Keyboard */
void
SceneExaminer::keyPressed( SoKeyboardEvent* keyboardEvent, SoHandleEventAction* action )
{
  if ( keyboardEvent->getKey() == SoKeyboardEvent::ESCAPE )
  {
    switchInteractionMode(action);
    action->setHandled();
  }
  else
  {
    m_currentInteraction->beginAction(keyboardEvent, action);
  }
}

void
SceneExaminer::keyReleased( SoKeyboardEvent* keyboardEvent, SoHandleEventAction* action )
{
  m_currentInteraction->beginAction(keyboardEvent, action);
}

/** Mouse */
void
SceneExaminer::mousePressed( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action )
{
  m_currentInteraction->beginAction(mouseEvent, action);
}

void
SceneExaminer::mouseReleased( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action )
{
  m_currentInteraction->beginAction(mouseEvent, action);
}

void
SceneExaminer::mouseWheelMoved( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action )
{
  m_currentInteraction->doAction(wheelEvent, action);
}

void
SceneExaminer::mouseMoved( SoLocation2Event* mouseEvent, SoHandleEventAction* action )
{
  m_currentInteraction->doAction(mouseEvent, action);
}

/** Touch */
void
SceneExaminer::touch( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  m_currentInteraction->doAction(touchEvent, action);
}

void
SceneExaminer::zoom( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action )
{
  m_currentInteraction->doAction(scaleEvent, action);
}

void
SceneExaminer::rotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action )
{
  m_currentInteraction->doAction(rotateEvent, action);
}

void
SceneExaminer::doubleTap(SoDoubleTapGestureEvent* doubleTapEvent, SoHandleEventAction* action)
{
  m_currentInteraction->doAction(doubleTapEvent, action);
}

void
SceneExaminer::longTap(SoLongTapGestureEvent* /*longTapEvent*/, SoHandleEventAction* action)
{
  switchInteractionMode(action);
  action->setHandled();
}

void
SceneExaminer::addInteractionModeListener( SiInteractionModeListener* listener )
{
  m_interactionModeListeners.push_back( listener );
  m_navigation->addInteractionModeListener( listener );
}

void
SceneExaminer::removeInteractionModeListener( SiInteractionModeListener* listener )
{
  m_interactionModeListeners.erase( std::remove( m_interactionModeListeners.begin(), m_interactionModeListeners.end(), listener ), m_interactionModeListeners.end() );
  m_navigation->removeInteractionModeListener( listener );
}

/** ViewingInteraction class */
const double NavigationInteraction::s_interaction_time = 0.5;

NavigationInteraction::NavigationInteraction( SoCameraInteractor* cameraInteractor )
  : BaseInteraction()
  , m_isZoomEnabled(true)
  , m_isPanEnabled(true)
  , m_isOrbitEnabled(true)
  , m_isRotateEnabled(true)
  , m_isSeekEnabled(true)
  , m_isSpinEnabled( false )
  , m_cameraInteractor(cameraInteractor)
  , m_isSeekMode(false)
  , m_isButton1Down(false)
  , m_isButton2Down(false)
  , m_mousePositionNorm( 0.f, 0.f )
  , m_interactiveCount( 0 )
  , m_isAutoClippingPlanes( SoPreferences::getBool("OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES", true) )
  , m_currentAction( nullptr )
{
  m_seekAnimator = new SeekAnimator( m_cameraInteractor, NULL );
  m_seekAnimator->setListener( new SeekAnimatorListener( this ) );

  m_orbitAnimator = new OrbitAnimator( m_cameraInteractor );
  m_orbitAnimator->setListener( new OrbitAnimatorListener( this ) );

  m_interactionEndSensor = new SoAlarmSensor();
  m_interactionEndSensor->setData( this );
  m_interactionEndSensor->setFunction( []( void* data, SoSensor* /*sensor*/ )
  {
    // stop interaction
    NavigationInteraction* interaction = static_cast<NavigationInteraction*>(data);
    interaction->interactiveCountDec();
  } );

  m_mouseWheelDelta = SoPreferences::getInt( "OIV_WHEEL_DELTA", 120 );
}

NavigationInteraction::NavigationInteraction( const NavigationInteraction& copyFrom )
  : BaseInteraction()
  , m_isZoomEnabled( copyFrom.m_isZoomEnabled )
  , m_isPanEnabled( copyFrom.m_isPanEnabled )
  , m_isOrbitEnabled( copyFrom.m_isOrbitEnabled )
  , m_isRotateEnabled( copyFrom.m_isRotateEnabled )
  , m_isSeekEnabled( copyFrom.m_isSeekEnabled )
  , m_isSpinEnabled( false )
  , m_cameraInteractor( copyFrom.m_cameraInteractor )
  , m_isSeekMode( copyFrom.m_isSeekMode )
  , m_isButton1Down( copyFrom.m_isButton1Down )
  , m_isButton2Down( copyFrom.m_isButton2Down )
  , m_mousePositionNorm( copyFrom.m_mousePositionNorm )
  , m_mouseWheelDelta( copyFrom.m_mouseWheelDelta )
  , m_interactiveCount( copyFrom.m_interactiveCount )
  , m_isAutoClippingPlanes( copyFrom.m_isAutoClippingPlanes )
  , m_currentAction( nullptr )
{
  m_seekAnimator = new SeekAnimator( copyFrom.m_cameraInteractor, NULL );
  m_seekAnimator->setDuration( copyFrom.m_seekAnimator->getDuration() );
  m_seekAnimator->setRelativeDistance( copyFrom.m_seekAnimator->getRelativeDistance() );
  m_seekAnimator->setListener( new SeekAnimatorListener( this ) );

  m_orbitAnimator = new OrbitAnimator( m_cameraInteractor );
  m_orbitAnimator->setListener( new OrbitAnimatorListener( this ) );

  m_interactionEndSensor = new SoAlarmSensor();
  m_interactionEndSensor->setData( this );
  m_interactionEndSensor->setFunction( []( void* data, SoSensor* /*sensor*/ )
  {
    // stop interaction
    NavigationInteraction* interaction = static_cast<NavigationInteraction*>(data);
    interaction->interactiveCountDec();
  } );
}

NavigationInteraction::~NavigationInteraction()
{
  delete m_seekAnimator;
  delete m_orbitAnimator;
  delete m_interactionEndSensor;
}

void
NavigationInteraction::reset()
{
  // reset mouse buttons state
  m_isButton1Down = false;
  m_isButton2Down = false;
  // stop seek
  setSeekMode(false);
  // reset interaction
  if ( m_interactiveCount > 0 )
  {
    // stop current interaction
    m_interactiveCount = 1;
    interactiveCountDec();
  }
  m_currentAction = nullptr;
}

void
NavigationInteraction::setCameraInteractor( SoCameraInteractor* cameraInteractor )
{
  m_cameraInteractor = cameraInteractor;
  m_seekAnimator->setCameraInteractor( cameraInteractor );
}

void
NavigationInteraction::setSeekMode( bool onOrOff )
{
  // check if seek is being turned off while seek animation is happening
  if ( !onOrOff )
    m_seekAnimator->stop();

  m_isSeekMode = onOrOff;
  for( std::vector<SiInteractionModeListener*>::iterator listener = m_interactionModeListeners.begin(); listener != m_interactionModeListeners.end(); ++listener  )
  {
    ( *listener )->seekModeChanged( m_isSeekMode );
  }
}

void
NavigationInteraction::doSeek( SoEvent* anyEvent, SoHandleEventAction* action )
{
  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
  SoNode* sceneGraph = action->getNodeAppliedTo();

  // do the picking
  SoRayPickAction pick = SoRayPickAction( vpRegion );
  pick.setSceneManager(action->getSceneManager());
  pick.setNormalizedPoint( anyEvent->getNormalizedPosition(vpRegion) );
  pick.setRadius( 1 );
  pick.setPickAll( false ); // pick only the closest object
  pick.apply( sceneGraph );

  // makes sure something got picked
  SoPickedPoint* pp = pick.getPickedPoint();
  if ( pp == NULL )
  {
    setSeekMode( false );
  }
  else
  {
    // set up and start animation
    m_seekAnimator->setSceneGraph(sceneGraph);
    m_seekAnimator->setUp( pp->getPoint(), vpRegion );
    m_seekAnimator->start();
  }

  action->setHandled();
}

void
NavigationInteraction::beginAction( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action )
{
  const SoMouseButtonEvent::Button button = mouseEvent->getButton();
  const SoMouseButtonEvent::State state = mouseEvent->getState();

  if ( button == SoMouseButtonEvent::BUTTON1 )
    m_isButton1Down = ( state == SoMouseButtonEvent::DOWN );
  else if ( button == SoMouseButtonEvent::BUTTON2 )
    m_isButton2Down = ( state == SoMouseButtonEvent::DOWN );
  m_currentAction = action;

  if ( m_isSpinEnabled )
  {
    const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
    SbVec2f mousePositionNorm = mouseEvent->getNormalizedPosition( vpRegion );
    if ( m_isButton1Down )
    {
      m_orbitAnimator->stop();
      m_orbitAnimator->setSceneGraph( action->getNodeAppliedTo() );
      m_orbitAnimator->setViewportRegion( vpRegion );
      m_orbitAnimator->setStartPoint( mousePositionNorm );
    }
    else
    {
      if ( !m_orbitAnimator->startPoint().equals( mousePositionNorm, 0.00001f ) )
        m_orbitAnimator->start();
    }
  }

  if ( m_isButton1Down && m_isSeekMode )
  {
    doSeek( mouseEvent, action );
  }
  else
  {
    m_mousePositionNorm = mouseEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
    beginAction(action, mouseEvent->wasCtrlDown(), mouseEvent->wasShiftDown());
  }

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::beginAction( SoKeyboardEvent* keyEvent, SoHandleEventAction* action )
{
  const SoKeyboardEvent::Key key = keyEvent->getKey();
  const SoKeyboardEvent::State state = keyEvent->getState();
  m_currentAction = action;

  switch ( key )
  {
  case SoKeyboardEvent::S:
    if ( m_isSeekEnabled && state == SoKeyboardEvent::DOWN )
    {
      setSeekMode(!m_isSeekMode);
      action->setHandled();
    }
    break;
  case SoKeyboardEvent::LEFT_CONTROL:
    beginAction(action, state == SoKeyboardEvent::DOWN, keyEvent->wasShiftDown());
    break;
  case SoKeyboardEvent::LEFT_SHIFT:
    beginAction(action, keyEvent->wasCtrlDown(), state == SoKeyboardEvent::DOWN);
    break;
  default:
    break;
  }

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::doAction( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  // ZOOM
  int wheelDelta = wheelEvent->getDelta() / m_mouseWheelDelta;
  float scaleFactor= std::pow( 2.f, (float) (wheelDelta * M_PI / 180.0f) );

  doDollyWithCenter(m_mousePositionNorm, scaleFactor, action);

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::doAction( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  const SbVec2f normPosition = SoViewportRegionElement::get(action->getState()).normalize(scaleEvent->getPosition());

  doDollyWithCenter(normPosition, 1.f / scaleEvent->getDeltaScaleFactor(), action);

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::doAction( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  doRotate(rotateEvent, action);

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::doAction( SoDoubleTapGestureEvent* doubleTapEvent, SoHandleEventAction* action )
{
  m_currentAction = action;

  if ( m_isSeekEnabled )
  {
    setSeekMode(true);
    doSeek(doubleTapEvent, action);
  }

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::doAction( SoLongTapGestureEvent* /*longTapEvent*/, SoHandleEventAction* action )
{
  m_currentAction = action;

  if ( !propagateEvents() )
    action->setHandled();
}

void
NavigationInteraction::beginPan( SoHandleEventAction* action )
{
  if ( m_isPanEnabled )
  {
    m_cameraInteractor->activatePanning(m_mousePositionNorm, SoViewportRegionElement::get(action->getState()));
    action->setHandled();
  }
}

void
NavigationInteraction::beginOrbit( SoHandleEventAction* action )
{
  if ( m_isOrbitEnabled )
  {
    m_cameraInteractor->activateOrbiting(m_mousePositionNorm);
    m_cameraInteractor->setRotationCenter(m_cameraInteractor->getFocalPoint());
    action->setHandled();
  }
}

void
NavigationInteraction::doPan( SoEvent* anyEvent, SoHandleEventAction* action )
{
  if ( m_isPanEnabled )
  {
    startInteraction();

    const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
    m_mousePositionNorm = anyEvent->getNormalizedPosition(vpRegion);
    m_cameraInteractor->pan(m_mousePositionNorm, vpRegion);
    action->setHandled();
  }
}

void
NavigationInteraction::doOrbit( SoEvent* anyEvent, SoHandleEventAction* action )
{
  const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
  m_orbitAnimator->addScreenPoint( anyEvent->getNormalizedPosition(vpRegion) );

  if ( m_isOrbitEnabled )
  {
    startInteraction();

    m_mousePositionNorm = anyEvent->getNormalizedPosition(vpRegion);
    m_cameraInteractor->orbit(m_mousePositionNorm);

    if ( m_isAutoClippingPlanes )
      m_cameraInteractor->adjustClippingPlanes(action->getNodeAppliedTo(), vpRegion);
    action->setHandled();
  }
}

void
NavigationInteraction::doDolly( SoEvent* anyEvent, SoHandleEventAction* action )
{
  if ( m_isZoomEnabled )
  {
    startInteraction();

    const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
    const SbVec2f& newLocator = anyEvent->getNormalizedPosition(vpRegion);
    const float dollyScaleFactor = float( pow( 2.0f, 10.0f * (newLocator[1] - m_mousePositionNorm[1])) );
    m_cameraInteractor->dolly(dollyScaleFactor);

    if ( m_isAutoClippingPlanes )
      m_cameraInteractor->adjustClippingPlanes(action->getNodeAppliedTo(), vpRegion);
    m_mousePositionNorm = newLocator;
    action->setHandled();
  }
}

void
NavigationInteraction::doDollyWithCenter( const SbVec2f& center, float scaleFactor, SoHandleEventAction* action )
{
  if ( m_isZoomEnabled )
  {
    startInteraction();

    const SbViewportRegion& region = SoViewportRegionElement::get(action->getState());
    m_cameraInteractor->dollyWithZoomCenter(center, scaleFactor, region);

    if ( m_isAutoClippingPlanes )
      m_cameraInteractor->adjustClippingPlanes(action->getNodeAppliedTo(), region);
    action->setHandled();
  }
}

void
NavigationInteraction::doRoll( SoEvent* anyEvent, SoHandleEventAction* action )
{
  if ( m_isRotateEnabled )
  {
    startInteraction();

    const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
    const SbVec2f& newLocator = anyEvent->getNormalizedPosition(vpRegion);
    SbVec2f center(0.5f, 0.5f);
    SbVec2f oldPosCenterDist = m_mousePositionNorm - center;
    m_mousePositionNorm = newLocator;

    // compute roll angle
    SbVec2f newPosCenterDist = m_mousePositionNorm - center;
    float rollAngle = ( newPosCenterDist[0] == 0.f && newPosCenterDist[1] == 0.f ) ?
      0.f : (float) atan2( newPosCenterDist[1], newPosCenterDist[0] );
    rollAngle -= ( oldPosCenterDist[0] == 0.f && oldPosCenterDist[1] == 0.f ) ?
      0.f : (float) atan2( oldPosCenterDist[1], oldPosCenterDist[0] );

    m_cameraInteractor->roll(rollAngle);

    if ( m_isAutoClippingPlanes )
      m_cameraInteractor->adjustClippingPlanes(action->getNodeAppliedTo(), vpRegion);
    action->setHandled();
  }
}

void
NavigationInteraction::doTranslate( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  if ( m_isPanEnabled )
  {
    startInteraction();

    // NOTE: We are normalizing a distance, not a position. Cannot use vpRegion.normalize!
    const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
    const SbVec2i32& vpSize = vpRegion.getViewportSizePixels_i32();
    SbVec2f displacement = touchEvent->getDisplacement(); // Returns an object, not a reference
    SbVec2f normDisp(0.5f * displacement[0] / vpSize[0], 0.5f * displacement[1] / vpSize[1]);
    m_cameraInteractor->translate(normDisp, vpRegion);
    action->setHandled();
  }
}

void
NavigationInteraction::doRotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action )
{
  if ( m_isRotateEnabled )
  {
    startInteraction();

    const SbViewportRegion& vpRegion = SoViewportRegionElement::get(action->getState());
    const SbVec2f& eventNormPosition = rotateEvent->getNormalizedPosition(vpRegion);
    float distFromEye = m_cameraInteractor->getCamera()->getViewVolume().getNearDist();
    SbVec3f rotCenter = m_cameraInteractor->projectToPlane(eventNormPosition, distFromEye, vpRegion);

    m_cameraInteractor->setRotationAxis(SbVec3f(0, 0, 1.0f));
    m_cameraInteractor->setRotationCenter(rotCenter);
    m_cameraInteractor->rotate(- rotateEvent->getDeltaRotation() * 0.5f);
    action->setHandled();
  }
}

void
NavigationInteraction::addInteractionModeListener( SiInteractionModeListener* listener )
{
  m_interactionModeListeners.push_back( listener );
}

void
NavigationInteraction::removeInteractionModeListener( SiInteractionModeListener* listener )
{
  m_interactionModeListeners.erase( std::remove( m_interactionModeListeners.begin(), m_interactionModeListeners.end(), listener ), m_interactionModeListeners.end() );
}

void
NavigationInteraction::interactiveCountInc()
{
  m_interactiveCount++;
  if ( m_interactiveCount == 1 && m_currentAction != nullptr )
  {
    // an interaction starts, set interactive mode in scene manager
    m_currentAction->getSceneManager()->setInteractive( true );
    // force a redraw, so graph will see the change in
    // the SoInteractionElement
    m_currentAction->getSceneManager()->scheduleRedraw();
  }
}

void
NavigationInteraction::interactiveCountDec()
{
  if ( m_interactiveCount > 0 )
  {
    m_interactiveCount--;

    if ( m_interactiveCount == 0 && m_currentAction != nullptr )
    {
      // interaction is finished, disable interactive mode in scene manager
      m_currentAction->getSceneManager()->setInteractive( false );
      // force a redraw, so graph will see the change in
      // the SoInteractionElement
      m_currentAction->getSceneManager()->scheduleRedraw();
    }
  }
}

void
NavigationInteraction::startInteraction()
{
  if ( m_interactionEndSensor->isScheduled() )
  {
    // already interacting, unschedule previous timer
    m_interactionEndSensor->unschedule();
  }
  else
  {
    // interaction starts
    interactiveCountInc();
  }
  // wait 0.5s... if scene is still, consider interaction is finished
  m_interactionEndSensor->setTimeFromNow( SbTime( s_interaction_time ) );
  m_interactionEndSensor->schedule();
}

/** OrbitInteraction class */
OrbitInteraction::OrbitInteraction( SoCameraInteractor* cameraInteractor )
  : NavigationInteraction(cameraInteractor)
  , m_isTouchOrbitActivated(false)
{}

OrbitInteraction::OrbitInteraction( const NavigationInteraction &copyFrom )
  : NavigationInteraction(copyFrom)
  , m_isTouchOrbitActivated(false)
{}

void
OrbitInteraction::beginAction( SoHandleEventAction* action, bool ctrlDown, bool /*shiftDown*/ )
{
  if ( m_isButton1Down && ctrlDown )
  {
    // BUTTON 1 + CTRL = pan
    beginPan(action);
  }
  else if ( m_isButton2Down )
  {
    // BUTTON 2 without modifier = pan
    beginPan(action);
  }
  else if ( m_isButton1Down )
  {
    // BUTTON 1 without modifier = orbit
    beginOrbit(action);
  }

  if ( !propagateEvents() )
    action->setHandled();
}

void
OrbitInteraction::doAction( SoLocation2Event* mouseEvent, SoHandleEventAction* action )
{
  const bool ctrlDown = mouseEvent->wasCtrlDown();
  const bool shiftDown = mouseEvent->wasShiftDown();

  if ( m_isButton1Down && m_isButton2Down )
  {
    // BUTTON 1 + BUTTON 2 = dolly
    doDolly(mouseEvent, action);
  }
  else if ( m_isButton1Down && ctrlDown )
  {
    // BUTTON 1 + CTRL = pan
    doPan(mouseEvent, action);
  }
  else if ( m_isButton2Down && ctrlDown )
  {
    // BUTTON 2 + CTRL = dolly
    doDolly(mouseEvent, action);
  }
  else if (  m_isButton1Down && shiftDown )
  {
    // BUTTON 1 + SHIFT = dolly
    doDolly(mouseEvent, action);
  }
  else if ( m_isButton1Down )
  {
    // BUTTON 1 without modifier = orbit
    doOrbit(mouseEvent, action);
  }
  else if ( m_isButton2Down )
  {
    // BUTTON 2 without modifier = pan
    doPan(mouseEvent, action);
  }

  if ( !action->isHandled() )
  {
    // update mouse cursor's position
    m_mousePositionNorm = mouseEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));

    // In order to mimic a "viewer switch mode" event, a SoLocation2Event::EventSource::OTHER is used instead.
    // This specific event has to be propagated through the scene graph.
    if ( !propagateEvents() && mouseEvent->getEventSource()!=SoLocation2Event::EventSource::OTHER )
      action->setHandled();
  }
}

void
OrbitInteraction::doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  const SoTouchEvent::State state = touchEvent->getState();
  int numFinger = touchEvent->getTouchManager()->getFingerNumber();

  if ( numFinger == 1 && state == SoTouchEvent::DOWN )
  {
    // one finger down
    m_mousePositionNorm = touchEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));
    beginOrbit(action);

    if ( action->isHandled() )
      m_isTouchOrbitActivated = true;
  }
  else if ( numFinger == 1 && state == SoTouchEvent::MOVE && m_isTouchOrbitActivated )
  {
    // one finger moved
    doOrbit(touchEvent, action);
  }
  else if ( numFinger == 2 && (state == SoTouchEvent::DOWN || state == SoTouchEvent::MOVE) )
  {
    // 2 fingers down or moved
    doTranslate(touchEvent, action);
  }
  else if ( numFinger == 2 && state == SoTouchEvent::UP )
  {
    // one finger is on the screen but one has been lifted,
    // orbiting is temporarily disabled until the next touch down event.
    m_isTouchOrbitActivated = false;
  }

  if ( !propagateEvents() )
    action->setHandled();
}

/** PlaneInteraction class */
PlaneInteraction::PlaneInteraction( SoCameraInteractor* cameraInteractor )
  : NavigationInteraction(cameraInteractor)
{}

PlaneInteraction::PlaneInteraction( const NavigationInteraction& copyFrom )
  : NavigationInteraction(copyFrom)
{}

void
PlaneInteraction::beginAction( SoHandleEventAction* action, bool ctrlDown, bool /*shiftDown*/ )
{
  if ( m_isButton1Down && ctrlDown )
  {
    // BUTTON 1 + CRTL = pan
    beginPan(action);
  }
  else if ( m_isButton2Down )
  {
    // BUTTON 2 without modifier = pan
    beginPan(action);
  }

  if ( !propagateEvents() )
    action->setHandled();
}

void
PlaneInteraction::doAction( SoLocation2Event* mouseEvent, SoHandleEventAction* action )
{
  const bool ctrlDown = mouseEvent->wasCtrlDown();
  const bool shiftDown = mouseEvent->wasShiftDown();

  if ( m_isButton1Down && m_isButton2Down )
  {
    // BUTTON 1 + BUTTON 2 = roll
    doRoll(mouseEvent, action);
  }
  else if ( m_isButton1Down && ctrlDown )
  {
    // BUTTON 1 + CTRL = pan
    doPan(mouseEvent, action);
  }
  else if ( m_isButton2Down && ctrlDown )
  {
    // BUTTON 2 + CTRL = roll
    doRoll(mouseEvent, action);
  }
  else if (  m_isButton1Down && shiftDown )
  {
    // BUTTON 1 + SHIFT = roll
    doRoll(mouseEvent, action);
  }
  else if ( m_isButton1Down )
  {
    // BUTTON 1 without modifier = dolly
    doDolly(mouseEvent, action);
  }
  else if ( m_isButton2Down )
  {
    // BUTTON 2 without modifier = pan
    doPan(mouseEvent, action);
  }

  if ( !action->isHandled() )
  {
    // update mouse cursor's position
    m_mousePositionNorm = mouseEvent->getNormalizedPosition(SoViewportRegionElement::get(action->getState()));

    if ( !propagateEvents() )
      action->setHandled();
  }
}

void
PlaneInteraction::doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  const SoTouchEvent::State state = touchEvent->getState();
  int numFinger = touchEvent->getTouchManager()->getFingerNumber();

  if ( numFinger == 1 && (state == SoTouchEvent::DOWN || state == SoTouchEvent::MOVE) )
  {
    // 1 finger down or moved
    doTranslate(touchEvent, action);
  }
  else if ( numFinger == 2 && (state == SoTouchEvent::DOWN || state == SoTouchEvent::MOVE) )
  {
    // 2 fingers down or moved
    doTranslate(touchEvent, action);
  }

  if ( !propagateEvents() )
    action->setHandled();
}

/** PickingInteraction class */
void
SelectionInteraction::doAction( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  // picking mode
  SoEvent* event_out = convertTouchEvent(touchEvent);
  action->setEvent(event_out);

  if ( !propagateEvents() )
    action->setHandled();
}

void
SelectionInteraction::doAction(SoLocation2Event*, SoHandleEventAction* action )
{
  if ( !propagateEvents() )
    action->setHandled();
}

void
SelectionInteraction::doAction(SoMouseWheelEvent*, SoHandleEventAction* action )
{
  if ( !propagateEvents() )
    action->setHandled();
}

void
SelectionInteraction::doAction(SoScaleGestureEvent*, SoHandleEventAction* action )
{
  if ( !propagateEvents() )
    action->setHandled();
}

void
SelectionInteraction::doAction(SoRotateGestureEvent*, SoHandleEventAction* action )
{
  if ( !propagateEvents() )
    action->setHandled();
}

void
SelectionInteraction::doAction(SoDoubleTapGestureEvent*, SoHandleEventAction* action )
{
  if ( !propagateEvents() )
    action->setHandled();
}

void
SelectionInteraction::doAction(SoLongTapGestureEvent*, SoHandleEventAction* action )
{
  if ( !propagateEvents() )
    action->setHandled();
}

SoEvent*
SelectionInteraction::convertTouchEvent(SoTouchEvent* touchEvent)
{
  SoMouseButtonEvent* mbe = &m_touchMouseButtonEvent;
  SoLocation2Event* lct = &m_touchLocation2Event;
  SoTouchEvent::State state = touchEvent->getState();

  if ( state == SoTouchEvent::DOWN)
  {
    mbe->setTime(touchEvent->getTime());
    mbe->setPosition(touchEvent->getPosition());
    mbe->setButton(SoMouseButtonEvent::BUTTON1);
    mbe->setState(SoMouseButtonEvent::DOWN);
    return mbe;
  }

  if ( state == SoTouchEvent::MOVE)
  {
    lct->setTime(touchEvent->getTime());
    lct->setPosition(touchEvent->getPosition());
    lct->setEventSource(SoLocation2Event::MOUSE_MOVE);
    return lct;
  }

  if ( state == SoTouchEvent::UP)
  {
    mbe->setTime(touchEvent->getTime());
    mbe->setPosition(touchEvent->getPosition());
    mbe->setButton(SoMouseButtonEvent::BUTTON1);
    mbe->setState(SoMouseButtonEvent::UP);
    return mbe;
  }

  SoDebugError::post( __FUNCTION__, "Unknown Touch Event" );
  return NULL;
}
