#include <Inventor/SoPickedPoint.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <Inventor/ViewerComponents/nodes/SceneOrbiter.h>
#include <Inventor/ViewerComponents/nodes/SoViewingCube.h>
#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include "InteractionOrbiter.h"

SceneOrbiter::SceneOrbiter()
  : SceneInteractor()
{
  m_interaction = new InteractionOrbiter( m_cameraInteractor );
  m_viewingCubeSwitch = new SoSwitch();
  {
    m_viewingCube = new SoViewingCube();
    m_viewingCube->sceneCamera.setValue( getCamera() );
    m_viewingCubeSwitch->addChild( m_viewingCube );

    m_cameraSensor = new SoNodeSensor(cameraSensorCallback, this);
    m_cameraSensor->attach(getCamera());
  }
  // enable viewing cube by default
  m_viewingCubeSwitch->whichChild.setValue( SO_SWITCH_ALL );

  // TURNTABLE by default
  m_interaction->setRotationMethod( TURNTABLE );
  // up axis = Y by default
  using openinventor::inventor::Axis;
  m_interaction->setUpAxis( Axis::Y );
  m_viewingCube->upAxis = Axis::Y;

  m_upAxisSensor = new SoFieldSensor( []( void* userData, SoSensor* /*sensor*/ ) {
    // when upAxis of viewing cube changes, update upAxis of orbiter
    SceneOrbiter* sceneOrbiter = static_cast<SceneOrbiter*>(userData);
    sceneOrbiter->m_interaction->setUpAxis( (Axis::Type) sceneOrbiter->m_viewingCube->upAxis.getValue() );
  }, this );
  m_upAxisSensor->attach( &m_viewingCube->upAxis );

  this->addChild( m_viewingCubeSwitch.ptr() );
}

SceneOrbiter::~SceneOrbiter()
{
  delete m_interaction;
  delete m_upAxisSensor;
}

void
SceneOrbiter::setConstraintLevel( float level )
{
  m_interaction->setConstraintLevel( level );
}

float
SceneOrbiter::getConstraintLevel()
{
  return m_interaction->getConstraintLevel();
}

void
SceneOrbiter::setRotationMethod( RotationMethod style )
{
  m_interaction->setRotationMethod( style );
}

SceneOrbiter::RotationMethod
SceneOrbiter::getRotationMethod()
{
  return m_interaction->getRotationMethod();
}

void
SceneOrbiter::setUpAxis( openinventor::inventor::Axis::Type axis )
{
  m_interaction->setUpAxis( axis );
  // update upAxis of viewing cube if needed
  if ( m_viewingCube->upAxis.getValue() != axis )
    m_viewingCube->upAxis = axis;
}

openinventor::inventor::Axis::Type
SceneOrbiter::getUpAxis()
{
  return m_interaction->getUpAxis();
}

void
SceneOrbiter::setCameraMode( SceneInteractor::CameraMode mode )
{
  SceneInteractor::setCameraMode( mode );

  if ( m_interaction != nullptr )
    m_interaction->setCameraInteractor( m_cameraInteractor );

  if (m_viewingCube != nullptr )
    m_viewingCube->sceneCamera.setValue( getCamera() );
}

bool
SceneOrbiter::isViewingCubeEnabled() const
{
  return m_viewingCubeSwitch->whichChild.getValue() == SO_SWITCH_ALL;
}

void
SceneOrbiter::enableViewingCube( bool enable )
{
  if ( enable )
    m_viewingCubeSwitch->whichChild = SO_SWITCH_ALL;
  else
    m_viewingCubeSwitch->whichChild = SO_SWITCH_NONE;
}

SoViewingCube*
SceneOrbiter::getViewingCube()
{
  return m_viewingCube;
}

/** Keyboard */
void
SceneOrbiter::keyPressed( SoKeyboardEvent* keyboardEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->beginAction( keyboardEvent, action );
}

void
SceneOrbiter::keyReleased( SoKeyboardEvent* keyboardEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->beginAction( keyboardEvent, action );
}

void
SceneOrbiter::cameraSensorCallback(void* data, SoSensor* /*sensor*/)
{
  SceneOrbiter* sceneOrbiter = static_cast<SceneOrbiter*>(data);
  if (sceneOrbiter != nullptr)
  {
    sceneOrbiter->adjustClippingPlanes(sceneOrbiter->m_viewportRegion);
  }
}

/** Mouse */
void
SceneOrbiter::mousePressed( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->beginAction( mouseEvent, action );
}

void
SceneOrbiter::mouseReleased( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->beginAction( mouseEvent, action );
}

void
SceneOrbiter::mouseWheelMoved( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->doAction( wheelEvent, action );
}

void
SceneOrbiter::mouseMoved( SoLocation2Event* mouseEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->doAction( mouseEvent, action );
}

/** Touch*/
void
SceneOrbiter::touch( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->doAction( touchEvent, action );
}

void
SceneOrbiter::zoom( SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action )
{
  m_viewportRegion = SoViewportRegionElement::get(action->getState());
  m_interaction->doAction( scaleEvent, action );
}

void
SceneOrbiter::rotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action )
{
    m_viewportRegion = SoViewportRegionElement::get(action->getState());
    m_interaction->doAction( rotateEvent, action );
}
