/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Tristan MEHAMLI (Dec 2010)
**=======================================================================*/

#include <Ivt_ViewingToolbar.h>
#include <IvTune/IvTuneExtender/IvtShell.h>
#include <IvTune/IvTuneExtender/IvtServices.h>

#include <Inventor/Qt/viewers/SoQtFullViewer.h>

#include <Inventor/Gui/viewers/SoGuiAlgoViewers.h>

#include <QAction>
#include <QActionGroup>
#include <QCursor>
#include <QMenu>
#include <QPixmap>
#include <QToolBar>
#include <QToolButton>

//------------------------------------------------------------------------------
Ivt_ViewingToolbar::Ivt_ViewingToolbar() 
: IvtExtension()
{
  m_info.name = "Viewing Toolbar";
  m_info.author ="Thermo Fisher Scientific";
  m_info.description = "A toolbar based on the toolbar from Avizo.";
  m_info.version = "1.0.0";

  m_restored = false;
}

//------------------------------------------------------------------------------
Ivt_ViewingToolbar::~Ivt_ViewingToolbar()
{
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::activate()
{
  if ( m_viewingToolbar ) // Tool bar might not be created if it is a render area instead of a viewer.
  {
    if ( m_restored )
      saveState();

    setViewingCursors();

    IvtShell::getInstance()->registerToolBar( m_viewingToolbar ); // Register the tool bar
    IvtShell::getInstance()->registerViewMenu( this ); // Register this extension to the view menu of IvTune

    m_viewer->setDecoration( FALSE ); // Hide the decorations of IvTune's viewer

    show(); // Show the tool bar

    IvtExtension::activate();
  }
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::deactivate()
{
  if ( m_viewingToolbar )
  {
    IvtExtension::deactivate();

    m_viewer->setDecoration( TRUE ); // Show the decoration of the viewer
    // Unregister the menu and this extension
    IvtShell::getInstance()->unregisterToolBar( m_viewingToolbar );
    IvtShell::getInstance()->unregisterViewMenu( this );

    restoreState();
    resetToolBar();
  }
}

//------------------------------------------------------------------------------
QToolBar* 
Ivt_ViewingToolbar::getViewingToolbar() const
{
  return m_viewingToolbar;
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::hide()
{
  IvtExtension::hide();
  m_viewingToolbar->hide(); // Hide the tool bar
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::load()
{
  // Inits
  m_isOffline = (IvtServices::getInstance()->isStandaloneApplication());
  m_gav = dynamic_cast<SoGuiAlgoViewers*>(IvtServices::getInstance()->getRenderArea());
  
  // This tool bar only works with Qt viewers otherwise it won't be created.
  if ( m_gav != 0 )
  {
    SoQtViewer* tmp = (SoQtViewer*)(IvtServices::getInstance()->getRenderArea()->m_renderAreaHandle);
    if ( !dynamic_cast<SoQtFullViewer*>(tmp) )
      m_viewer = NULL;
    else
      m_viewer = (SoQtFullViewer*)tmp;

    if ( m_viewer ) // Can't build the toolbar if there's no access to the viewer's algorithms
      buildViewingToolbar();
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::show()
{
  IvtExtension::show();
  m_viewingToolbar->show(); // Show the tool bar
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::unload()
{
  // Delete what was instanced in load()
  delete m_viewingToolbar;
}

//------------------------------------------------------------------------------
// Slots implementation
//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setFSAA( bool enable )
{
  if ( enable )
  {
    m_fsaaAction->setIcon( QIcon(":/ViewingTB_icons/Icons/ViewerFSAA.png") );
    m_gav->setAntialiasing(1.0f, SoSceneManager::FSAA);
  }
  else
  {
    m_fsaaAction->setIcon( QIcon(":/ViewingTB_icons/Icons/ViewerNoFSAA.png") );
    m_gav->setAntialiasing(0.0f, SoSceneManager::FSAA);
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setShadows( bool enable )
{
  m_gav->enableShadows( enable );
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setEdges( bool /*enable*/ )
{
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setStereo()
{
  if ( m_isOffline )
    m_viewer->openStereoDialog();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::pick( bool enable )
{
  if ( enable ) // Make sure that the modifications will apply only if enabled
  {
    // Stop current animations
    stopAnimations();

    // Set interaction mode
    m_gav->setViewing( false );
    if ( m_isOffline )
      m_viewer->setCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerInteract.png"), 6, 6) );
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::view( bool enable )
{
  if ( enable )
  {
    stopAnimations();

    // Set interaction mode
    m_gav->setViewing( true );
    if ( m_isOffline )
      m_viewer->internalSetViewingMode( 0 ); // 0 = VIEWING_MODE_SPIN see SoQtExaminerViewer
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setTranslation( bool enable )
{
  if ( enable )
  {
    stopAnimations();

    // Set interaction mode
    m_gav->setViewing( true );
    if ( m_isOffline )
      m_viewer->internalSetViewingMode( 2 ); // 2 = VIEWING_MODE_PAN see SoQtExaminerViewer
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setZoom( bool enable )
{
  if ( enable )
  {
    stopAnimations();
    
    // Set interaction mode
    m_gav->setViewing( true );
    if ( m_isOffline )
      m_viewer->internalSetViewingMode( 4 ); // 4 = VIEWING_MODE_ZOOM see SoQtExaminerViewer
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setRotation( bool enable )
{
  if ( enable )
  {
    stopAnimations();
    
    // Set interaction mode
    m_gav->setViewing( true );
    if ( m_isOffline )
      m_viewer->internalSetViewingMode( 3 ); // 3 = VIEWING_MODE_ROLL see SoQtExaminerViewer
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::seek()
{
  stopAnimations();

  m_gav->setSeekMode( !m_gav->isSeekMode() );

  if ( m_gav->isSeekMode() )
    m_viewer->setCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerSeek.png")) ); // Necessary because the cursor is changed
                                                                                       // at the end of the seek animation
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::home()
{
  stopAnimations();

  m_gav->resetToHomePosition();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setHome()
{
  stopAnimations();

  m_gav->saveHomePosition();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::toggleCamera( bool enable )
{
  stopAnimations();

  if ( enable )
    m_toggleCamera->setIcon( QIcon(":/ViewingTB_icons/Icons/ViewerOrtho.png") );
  else
    m_toggleCamera->setIcon( QIcon(":/ViewingTB_icons/Icons/ViewerPerspective.png") );

  m_gav->toggleCameraType();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::viewAll()
{
  stopAnimations();

  m_gav->viewAll();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setXYPlan()
{
  stopAnimations();

  m_gav->viewZ();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setXZPlan()
{
  stopAnimations();

  m_gav->viewY();
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::setYZPlan()
{
  stopAnimations();

  m_gav->viewX();
}

//------------------------------------------------------------------------------
// Viewing toolbar
//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::buildViewingToolbar()
{
  saveState();

  // Set up actions of the toolbar
  QAction* seekAction, *homeAction, *setHomeAction, *viewAllAction, *viewXYAction, *viewXZAction, 
    *viewYZAction, *stereoAction, *measure2DAction, *measure3DAction, 
    *measureAngle2DAction, *measureAngle3DAction;

  m_edgesAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerEdges.png"), tr("Edges"), this );
  m_edgesAction->setCheckable( true );
  m_fsaaAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerNoFSAA.png"), tr("FSAA"), this );
  m_fsaaAction->setCheckable( true );
  homeAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerHome.png"), tr("Home"), this );
  measure2DAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerMeasure2D.png"), tr("2D Measure"), this );
  measure3DAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerMeasure3D.png"), tr("3D Measure"), this );
  measureAngle2DAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerMeasureAngle2D.png"), tr("2D Angle Measure"), this );
  measureAngle3DAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerMeasureAngle3D.png"), tr("3D Angle Measure"), this );
  m_pickAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerInteract.png"), tr("Pick"), this );
  m_pickAction->setCheckable( true );
  m_rotateAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerRotate.png"), tr("Rotation"), this );
  m_rotateAction->setCheckable( true );
  seekAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerSeek.png"), tr("Seek"), this );
  setHomeAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerSetHome.png"), tr("Set Home"), this );
  m_shadowsAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerShadows.png"), tr("Shadows"), this );
  m_shadowsAction->setCheckable( true );
  stereoAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerStereo.png"), tr("Stereo"), this );
  m_toggleCamera = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerPerspective.png"), tr("Perspective camera"), this );
  m_toggleCamera->setCheckable( true );
  m_translateAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerTranslate.png"), tr("Translation"), this );
  m_translateAction->setCheckable( true );
  m_viewAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerTrackball.png"), tr("View"), this );
  m_viewAction->setCheckable( true );
  viewAllAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerViewAll.png"), tr("View All"), this );
  viewXYAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerViewXY.png"), tr("View XY"), this );
  viewXZAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerViewXZ.png"), tr("View XZ"), this );
  viewYZAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerViewYZ.png"), tr("View YZ"), this );
  m_zoomAction = new QAction( QIcon(":/ViewingTB_icons/Icons/ViewerZoom.png"), tr("Zoom"), this );
  m_zoomAction->setCheckable( true );

  // Create the viewing toolbar
  m_viewingToolbar = new QToolBar;
  m_viewingToolbar->setWindowTitle( "Viewing ToolBar" );
  QActionGroup* ag = new QActionGroup( this ); // Makes each action of the group exclusive

  // Set Avizo toobar functions
  // Exlusive actions
  ag->addAction( m_pickAction );
  ag->addAction( m_viewAction );
  ag->addAction( m_translateAction );
  ag->addAction( m_zoomAction );
  ag->addAction( m_rotateAction );
  m_viewingToolbar->addActions( ag->actions() );
  // Non exclusive actions
  m_viewingToolbar->addAction( seekAction );
  m_viewingToolbar->addSeparator();
  m_viewingToolbar->addAction( homeAction );
  m_viewingToolbar->addAction( setHomeAction );
  m_viewingToolbar->addAction( m_toggleCamera );
  m_viewingToolbar->addSeparator();
  m_viewingToolbar->addAction( viewAllAction );
  m_viewingToolbar->addAction( viewXYAction );
  m_viewingToolbar->addAction( viewXZAction );
  m_viewingToolbar->addAction( viewYZAction );

  // Set Edges, FSAA, Shadows, stereo and measurement functions
  m_viewingToolbar->addSeparator();
  //m_viewingToolbar->addAction( m_edgesAction );
  m_viewingToolbar->addAction( m_fsaaAction );
  m_viewingToolbar->addAction( m_shadowsAction );
  m_viewingToolbar->addAction( stereoAction );
  // Define the measurement popup menu button
  /*
  QToolButton* qtb = new QToolButton( m_viewingToolbar );
  QMenu* popup = new QMenu( m_viewingToolbar );
  qtb->setMenu( popup );
  qtb->setPopupMode( QToolButton::MenuButtonPopup );
  qtb->setIcon( QIcon(":/ViewingTB_icons/Icons/ViewerMeasure2D.png") );
  qtb->connect( popup, SIGNAL(triggered(QAction*)), qtb, SLOT(setDefaultAction(QAction*)));
  popup->addAction( measure2DAction );
  popup->addAction( measure3DAction );
  popup->addAction( measureAngle2DAction );
  popup->addAction( measureAngle3DAction );
  m_viewingToolbar->addWidget( qtb );
  */

  // Make the connections to the slots
  connect( m_pickAction, SIGNAL(toggled(bool)), this, SLOT(pick(bool)) );
  connect( m_viewAction, SIGNAL(toggled(bool)), this, SLOT(view(bool)) );
  connect( m_translateAction, SIGNAL(toggled(bool)), this, SLOT(setTranslation(bool)) );
  connect( m_zoomAction, SIGNAL(toggled(bool)), this, SLOT(setZoom(bool)) );
  connect( m_rotateAction, SIGNAL(toggled(bool)), this, SLOT(setRotation(bool)) );
  connect( seekAction, SIGNAL(triggered()), this, SLOT(seek()) );
  connect( homeAction, SIGNAL(triggered()), this, SLOT(home()) );
  connect( setHomeAction, SIGNAL(triggered()), this, SLOT(setHome()) );
  connect( m_toggleCamera, SIGNAL(toggled(bool)), this, SLOT(toggleCamera(bool)) );
  connect( viewAllAction, SIGNAL(triggered()), this, SLOT(viewAll()) );
  connect( viewXYAction, SIGNAL(triggered()), this, SLOT(setYZPlan()) );
  connect( viewXZAction, SIGNAL(triggered()), this, SLOT(setXYPlan()) );
  connect( viewYZAction, SIGNAL(triggered()), this, SLOT(setXZPlan()) );
  connect( m_edgesAction, SIGNAL(toggled(bool)), this, SLOT(setEdges(bool)) );
  connect( m_fsaaAction, SIGNAL(toggled(bool)), this, SLOT(setFSAA(bool)) );
  connect( m_shadowsAction, SIGNAL(toggled(bool)), this, SLOT(setShadows(bool)) );
  connect( stereoAction, SIGNAL(triggered()), this, SLOT(setStereo()) );

  // Set viewing by default
  m_viewAction->toggle();
}

//------------------------------------------------------------------------------
// Private methods
//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::stopAnimations()
{
  switch (m_gav->getViewerType())
  {
  case SoGuiAlgoViewers::EXAMINER:
    m_gav->stopSpinAnimation();
    break;
  case SoGuiAlgoViewers::FLY:
    m_gav->stopFlyAnimation();
    break;
  case SoGuiAlgoViewers::WALK:
    m_gav->stopWalkAnimation();
    break;
  default:
    break;
  }
}

//------------------------------------------------------------------------------
void 
Ivt_ViewingToolbar::resetToolBar()
{
  if ( !m_viewAction->isChecked() )
    m_viewAction->toggle();

  if ( m_toggleCamera->isChecked() )
    m_toggleCamera->toggle();

  if ( m_edgesAction->isChecked() )
    m_edgesAction->toggle();

  if ( m_fsaaAction->isChecked() )
    m_fsaaAction->toggle();

  if ( m_shadowsAction->isChecked() )
    m_shadowsAction->toggle();
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::setViewingCursors()
{
  // Change viewer cursors with the icons of the viewing toolbar
  if ( m_isOffline )
  {
    m_viewer->defineDefaultCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerInteract.png"), 6, 6) );
    m_viewer->defineRotationCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerRotate.png")) );
    m_viewer->defineSeekCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerSeek.png")) );
    m_viewer->defineTranslationCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerTranslate.png")) );
    m_viewer->defineViewingCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerTrackball.png")) );
    m_viewer->defineZoomCursor( QCursor(QPixmap(":/ViewingTB_icons/Icons/ViewerZoom.png")) );
  }
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::saveState()
{
  m_oldDecoration = m_viewer->isDecoration();
  m_oldViewingMode = m_viewer->internalGetViewingMode();
  m_oldViewing = m_gav->isViewing();
  m_restored = false;
}

//------------------------------------------------------------------------------
void
Ivt_ViewingToolbar::restoreState()
{
  m_viewer->setDecoration( m_oldDecoration );
  m_viewer->internalSetViewingMode( m_oldViewingMode );
  m_gav->setViewing( m_oldViewing );
  m_restored = true;

  // Reset viewer's cursors
  m_viewer->defineDefaultCursor( QCursor(QPixmap(1, 1)) );
  m_viewer->defineRotationCursor( QCursor(QPixmap(1, 1)) );
  m_viewer->defineSeekCursor( QCursor(QPixmap(1, 1)) );
  m_viewer->defineTranslationCursor( QCursor(QPixmap(1, 1)) );
  m_viewer->defineViewingCursor( QCursor(QPixmap(1, 1)) );
  m_viewer->defineZoomCursor( QCursor(QPixmap(1, 1)) );
  m_viewer->defineCursors();
}

