/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Jerome Hummel (Nov 2009)
** Modified by : T.MEHAMLI (Dec 2010)
**=======================================================================*/

#ifndef IVT_AC_EDITOR
#define IVT_AC_EDITOR

//////////////////////////////////////////////////////////////////////////////
//
//  Class: Ivt_AlphaCurveEditor
//
//////////////////////////////////////////////////////////////////////////////

#include <IvTune/IvTuneExtender/IvtEditor.h>

#include <QImage>
#include <QtCore/QVector>
#include <math.h>

/** 
 * @VSGEXT IvTune class to edit the alpha values of an SoTransferFunction node.
 * 
 * @ingroup IvtTools
 * 
 * @DESCRIPTION
 *   This editor allows modifying the alpha values of an SoTransferFunction node
 *   represented by an alpha curve. 
 * It displays the current colomap and by adding control points on it and moving them, modifies 
 * the alpha values in the colormap. To add a point, double-click on the colormap, to move 
 * a control point, left click on it and then move it along the colormap, and to remove a point, 
 * right click on it.
 *
 * Note: Only RGBA colormaps are supported.
 * 
 */

class QDockWidget;
class QEvent;
class QFrame;
class QLabel;
class QPaintEvent;
class QResizeEvent;
class QScrollArea;
class SoTransferFunction;

//Inner class for alpha keys representation
class AlphaKey
{
public:
  AlphaKey()
  : m_x(0)
  , m_sx(0)
  , m_y(0)
  , m_sCMSize(0)
  , m_alpha(0.0f)
  {}

  virtual ~AlphaKey() {}

  int m_x, m_sx;
  int m_y;
  int m_sCMSize;
  float m_alpha;

  bool operator == ( const AlphaKey t )
  {
    return m_x == t.m_x && m_y == t.m_y && fabs( m_alpha - t.m_alpha ) < 10E-5 ;
  }
};

//Editor class
class Ivt_AlphaCurveEditor: public IvtEditor
{
  Q_OBJECT

public:
  /**
   * Constructor.
   */
  Ivt_AlphaCurveEditor();

  /**
   * Destructor.
   */
  virtual ~Ivt_AlphaCurveEditor();

  // Methods redifined from IvtExtension, IvtEditor and IvtNodeEditor
  /**
   * Activates this extension.
   */
  virtual void activate();

  /**
   * Deactivates this extension.
   */
  virtual void deactivate();

  /**
   * Hides this editor.
   */
  virtual void hide();


  /**
   * Loads this extension.
   */
  virtual void load();


  /**
   * Shows this editor.
   */
  virtual void show();

  /**
   * Unloads this extension.
   */
  virtual void unload();

protected:
  /**
   * Adds an alpha key at position (x, y).
   */
  void addKey( int x, int y );

  /**
   * Checks alpha key position at x, y).
   */
  void checkKeyPosition( int x, int y );

  /**
   * Computes alpha keys.
   */
  void computeAlphaKeys();

  /**
   * Creates the colormap image.
   */
  void createColormapImage();

  /**
   * Returns the picked alpha key at position (x, y).
   */
  AlphaKey* getPickedKey( int x, int y );

  /**
   * Removes alpha key at position (x, y).
   */
  void removeKey( int x, int y );

  /**
   * Makes updates from alpha key at position (x, y).
   */
  void updatePickedKey( int x, int y );
  
  /**
   * Updates colormap's alpha values.
   */
  void updateAlphaValues();

  /**
   * Paint routine reimplementation.
   */
  virtual void paintEvent( QPaintEvent* event );

  /**
   * Resize routine reimplementation.
   */
  virtual void resizeEvent( QResizeEvent* event );

  /** 
   * Filters events if this object has been installed as an event filter for the watched 
   * object. 
   */
  bool eventFilter( QObject* obj, QEvent* event );

private:
  /** Sets up this editor. */
  void setUp();

  /** Computes the x coordinate of the keys when the editor is resized. */
  void updateKeysPos();

  /** Key before the picked key. */
  AlphaKey* m_leftKey;
  /** Picked key. */
  AlphaKey* m_pickedKey;
  /** Key after the picked key. */
  AlphaKey* m_rightKey;

  /** Numbers of colors in the colormap. */
  int m_numColors;
  /** Numbers of color components */
  int m_numComponents;
  /** Pen size for the drawing of the colormap. */
  int m_penWidth;
  /** Editor width */
  int m_editorWidth;
  /** Array which contains the x coordinate of the colors in the pixmap. */
  int* m_xColorCoord;
  /** Keeps track of the number of keys */
  int m_keysNum;

  /** Colormap representation. */
  QImage m_colormapImage;
  /** Colormap container, used by the QScrollArea. */
  QLabel* m_colormapContainer;
  /** Scroll area for large colormap. */
  QScrollArea* m_sca;
  /** Vectors of alpha keys. */
  QVector<AlphaKey*> m_keys;
  /** Vector of the colormap colors. */
  QVector<QColor> m_colorList;

  /** Edited transfer function. */
  SoTransferFunction* m_transferFunction;

  /** Containers for this editor. */

  QFrame* m_container;
};

#endif // IVT_AC_EDITOR


