///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Medical/InventorMedical.h>

#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/Magnifier.h>
#include <Medical/nodes/ObliqueSliceBorder.h>
#include <Medical/nodes/OrthoSliceBorder.h>
#include <Medical/nodes/PlaneBoxIntersection.h>
#include <Medical/nodes/PlaneGeometryIntersection.h>
#include <Medical/nodes/Ruler.h>
#include <Medical/nodes/SceneView.h>
#include <Medical/nodes/SliceOrientationMarkers.h>
#include <Medical/nodes/SliceScaleBar.h>
#include <Medical/nodes/TextBox.h>
#include <Medical/nodes/ViewManager.h>

#include <VolumeViz/nodes/SoVolumeRendering.h>

////////////////////////////////////////////////////////////////////////////////
// static vars
int InventorMedical::s_initRefCount = 0;

////////////////////////////////////////////////////////////////////////////////
void
InventorMedical::init()
{
  // Cannot initialize some of these nodes (e.g. OrthoSliceBorder) without VolumeViz.
  SoVolumeRendering::init();

  if ( s_initRefCount == 0)
  {
    // Set some preferences for medical data.
    // Open Inventor volume rendering (VolumeViz) uses an internal data manager
    // called LDM. When necessary it can manage a volume of data as multiple
    // sub-volumes called tiles.  This allows us to handle volumes that won't
    // in CPU memory or in GPU memory as a single contiguous block of memory.
    // There are many heuristics and some of the default settings are more
    // appropriate for very large volumes (like seismic data).  Applications
    // can override these settings after calling init() if appropriate.
    //
    // LDM should build tiles by requesting full slices from the reader.
    SoPreferences::setValue( "LDM_USE_BUILDTILE_BY_SLICE", "1" );

    // Initialize medical library classes.
    Gnomon::initClass();
    Magnifier::initClass();
    ObliqueSliceBorder::initClass();
    OrthoSliceBorder::initClass();
    PlaneBoxIntersection::initClass();
    PlaneGeometryIntersection::initClass();
    Ruler::initClass();
    SceneView::initClass();
    SliceOrientationMarkers::initClass();
    SliceScaleBar::initClass();
    ViewManager::initClass();
    // Note! Must init parent class before subclass (e.g. TextBox, then DicomInfo).
    TextBox::initClass();
    DicomInfo::initClass();
  }
  s_initRefCount++;
}

////////////////////////////////////////////////////////////////////////////////
void
InventorMedical::finish()
{
  // Check if it has already been initialized
  if( s_initRefCount == 0)
    return;

  s_initRefCount--;
  if ( s_initRefCount == 0 )
  {
    DicomInfo::exitClass();
    Gnomon::exitClass();
    Magnifier::exitClass();
    ObliqueSliceBorder::exitClass();
    OrthoSliceBorder::exitClass();
    PlaneBoxIntersection::exitClass();
    PlaneGeometryIntersection::exitClass();
    Ruler::exitClass();
    SceneView::exitClass();
    SliceOrientationMarkers::exitClass();
    SliceScaleBar::exitClass();
    TextBox::exitClass();
    ViewManager::exitClass();

    SoVolumeRendering::finish();
  }
}

////////////////////////////////////////////////////////////////////////////////
InventorMedical::InventorMedical()
{
}

////////////////////////////////////////////////////////////////////////////////
bool
InventorMedical::isInitialized()
{
  return (s_initRefCount > 0);
}
