///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/SliceOrientationMarkers.h>
#include <Medical/nodes/SliceScaleBar.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>

#include <VolumeViz/readers/SoVRLdmFileReader.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>

#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoImageBackground.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoMatrixTransform.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoTransform.h>

#include <Inventor/SbMatrix.h>

#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <Inventor/STL/algorithm> // for std::min/max
#include <Inventor/STL/vector>

// Suppress warnings about calling sscanf (we'll be careful).
#ifdef WIN32
#pragma warning( push )
#pragma warning( disable:4996 )
#endif

// Data values

static SbVec2s ExampleWindowSize( 1024, 633 );

const SbString LogoImageFile( "$OIVHOME/examples/data/Medical/resources/oiv_logo_white_test.png" );

///////////////////////////////////////////////////////////////////////////////
// Find power-of-2 >= specified value.
// Used for tile dimensions, which must be a power of 2.
static int nextPowerOf2( int n )
{
  int result = 1;
  while (result < n)
    result <<= 1;
  return result;
}

///////////////////////////////////////////////////////////////////////////////
// Orient view
SbBool
MedicalHelper::orientView( MedicalHelper::Axis axis, SoCamera* camera, 
                          const SoVolumeData* volume, float slack )
{
  if (camera == NULL)
    return FALSE;

  // We don't actually modify the data volume, but we can't query the extent if it's const...
  SbVec3f size(1,1,1);
  SbBox3f bbox(-1,-1,-1, 1,1,1);
  if (volume != NULL) {
    bbox = (const_cast<SoVolumeData*>(volume))->extent.getValue();
    size = bbox.getSize();
  }
  float width  = 1;
  float height = 1;

  SbRotation orientation;
  if (axis == MedicalHelper::SAGITTAL) { // X axis / Z up
    width = size[1];
    height = size[2];
    SbRotation rot1 = SbRotation( SbVec3f(0,0,1), (float)M_PI / 2 );
    orientation = rot1 * SbRotation( SbVec3f(0,1,0), (float)M_PI / 2 );
  }
  else if (axis == MedicalHelper::CORONAL) { // Y axis / Z up
    width = size[0];
    height = size[2];
    orientation  = SbRotation( SbVec3f(1,0,0), (float)M_PI / 2 );
  }
  else { // AXIAL = Z axis / Y up
    width = size[0];
    height = size[1];
    orientation  = SbRotation( SbVec3f(1,0,0), (float)M_PI );
  }

  // Notes:
  // - We use viewAll() to get the camera position, near/far clip planes, etc set.
  // - We use a fake viewport here to do the viewAll. Shouldn't be a problem.
  // - Then we adjust the view volume height to get a "tight" fit around the slice.
  //   Note this is currently only implemented for orthographic camera.
  // - Use width or height of image, whichever is greater.

  SbViewportRegion vport( 500, 500 );
  camera->orientation = orientation;
  if ( camera->isOfType( SoPerspectiveCamera::getClassTypeId() ) )
  {
    if ( volume != NULL ) // If we actually have a bbox
      camera->viewAll( bbox, vport );
  }
  else
  { // Orthographic camera
    SoOrthographicCamera* orthoCam = ( SoOrthographicCamera* )camera;
    if ( volume != NULL )
    { // If we actually have a bbox
      camera->viewAll( bbox, vport );
      height = ( height >= width ) ? height : width;
      orthoCam->height = height * slack;
    }
  }
  return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
/** Optimize volume data node for DICOM volumes.
  *  Returns true if successful.
  */
SbBool
MedicalHelper::dicomAdjustVolume( SoVolumeData* volume, SbBool useImagePosition )
{
  if (volume == NULL)
    return FALSE;

  // Get the reader being used to load data for this volume.
  SoVolumeReader* reader = volume->getReader();

  // Part 1 -------------------------------------------------------------------
  // Not specific to DICOM files.
  //
  // Set tile size to contain the whole volume if possible.
  // The default LDM tile size is quite small and results in inefficient handling
  // of typical DICOM volumes.  On the other hand, for a typical DICOM volume
  // with 512x512 images, a tile size of 512 means that each slice of a tile can
  // be loaded by efficiently loading an entire image.
  //
  // We recommend not making the tile size larger than 512.  For a DICOM volume
  // that contains 12 or 16 bit data values, a tile of size 1024 would require
  // allocating 2 GB of contiguous memory.  That request could fail even on the
  // CPU, let alone on the GPU.
  // 
  // For very large volumes we still get the benefit of LDM data management and
  // can visualize a volume of almost any size.

  volume->ldmResourceParameters.getValue()->tileDimension = volume->data.getSize();

  // Part 2 -------------------------------------------------------------------
  // Not specific to DICOM files.
  //
  // Unless you take specific action, Open Inventor will load data on the GPU
  // using 1 byte per voxel regardless of the actual voxel data type.  There
  // is no issue if the voxels are 1 byte (0..255) values.  Otherwise the data
  // values may be scaled to fit in 1 byte values on the GPU. This effectively
  // reduces the precision of the data on the GPU because multiple actual values
  // could be scaled to the same value.  However note that when the application
  // asks for the value of a specific voxel, Open Inventor returns the value
  // from the original actual data, so these values are always correct. The
  // issue (if any) comes when you want to manipulate the data range very
  // precisely.  DICOM data values are usually 12 or more bits per value and
  // require 2 bytes to store on the CPU.  We can request that Open Inventor
  // store actual values on the GPU as well, using the texturePrecision field.
  //
  // This does use additional memory on the GPU, so there may be cases where
  // it's better to use the default setting on very low end hardware.
  //
  // Get the number of bytes per voxel.
  int bytesPerVoxel = volume->getDatumSize();
  if (bytesPerVoxel == 2) {
    // Store actual values on GPU.
    volume->texturePrecision = 16;
  }

  // Part 3 -------------------------------------------------------------------
  // DICOM specific
  //
  // There is a known bug that the physical extent of the volume computed by the
  // DICOM volume reader (SoVRDicomFileReader) is slightly incorrect.
  // VolumeViz considers the physical extent of a volume in 3D to span from the
  // "outside edge" of the first voxel to the "outside edge" of the last voxel.
  // The DICOM volume reader computes the physical distance from the _center_
  // of the first voxel to the _center_ of the last last voxel.
  //
  // We can compute the correct physical extent for the X and Y axes because we
  // can get the X and Y pixel size in physical units (mm) from the DICOM file.
  // For some volumes it is possible to do the same thing for the Z axis using
  // the value of the Slice Spacing tag, but... this tag is not required and may
  // not exist in every volume.  Also some experts consider it unreliable.  The
  // reliable way to compute the Z extent is by using the Image Position values
  // from the first and last slices of the volume. And this is what the volume
  // reader does already, except that it multiplies by zDim-1 instead of zDim.
  // We just need to fix that computation.
  if (reader != NULL && reader->isOfType(SoVRDicomFileReader::getClassTypeId())) {
    SoVRDicomFileReader* dicomReader = (SoVRDicomFileReader*)reader;
    const SoVRDicomData& dicomData = dicomReader->getDicomData();

    // Get the X and Y physical voxel sizes from the DICOM data.
    float xVoxelSize = dicomData.getXPixelSize();
    float yVoxelSize = dicomData.getYPixelSize();

    // There is no reliable tag to get this info for Z, so we just adjust the
    // reader's (slightly) incorrect calculation.
    const SbVec3i32& volDim  = volume->data.getSize();
    const SbVec3f    volSize = volume->extent.getValue().getSize();
    float zVoxelSize;
    if (volDim[2] <= 1)  // But we must not divide by zero...
      zVoxelSize = volSize[2];
    else
      zVoxelSize = volSize[2] / (float)(volDim[2]);

    // The adjustment we need to make is plus and minus 1/2 voxel.
    SbVec3f halfVoxel( xVoxelSize/2, yVoxelSize/2, zVoxelSize/2 );

    if (useImagePosition == TRUE) {
      // Compute the physical extent of the volume.
      // For VolumeViz this means from outside edge to outside edge.
      SbVec3f volPhysicalSize( volDim[0] * xVoxelSize,
                               volDim[1] * yVoxelSize,
                               volDim[2] * zVoxelSize );

      // Get Image Position value from first image in DICOM stack.
      SbVec3f imagePos;
      dicomGetImagePosition( volume, imagePos );

      // Compute volume extent in 3D so the first voxel is positioned
      // at the image position specified in the DICOM data.
      // Remember that VolumeViz considers the volume extent to completely
      // include every voxel. So given the position of the center of the
      // first voxel, the volume extent begins 1/2 voxel before that.
      SbVec3f newMin, newMax; // Corners of volume extent
      newMin = imagePos - halfVoxel;
      newMax = newMin   + volPhysicalSize;
      SbBox3f newExtent( newMin, newMax );

      // Update the volume extent to get correct position/size in 3D space.
      volume->extent = newExtent;

      // Sanity check. useImagePosition gives correct position only for axis aligned acquisitions.
      // For oblique acquisition, the more generic method
      // dicomAdjustVolume( SoVolumeData* volume, SoMatrixTransform* imgToPatient )
      // should be used.
#ifdef DEBUG
      SbMatrix ImgToPatient = SbMatrix::identity();
      dicomData.getOrientation(ImgToPatient);
      if (ImgToPatient != SbMatrix::identity())
      {
        SoDebugError::post(__FUNCTION__, "Given DICOM volume doesn't seems to be axis aligned.\
                                          You should use dicomAdjustVolume(SoVolumeData*, SoMatrixTransform*) method instead.");
      }
#endif
    }
  }

  return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
SbBool
MedicalHelper::dicomAdjustVolume( SoVolumeData* volume, SoMatrixTransform* imgToPatient )
{
  if ( imgToPatient == NULL )
  {
#ifdef DEBUG
    SoDebugError::post( __FUNCTION__, "NULL SoMatrixTransform specified." );
#endif // DEBUG
    return FALSE;
  }

  if ( volume == NULL )
  {
#ifdef DEBUG
    SoDebugError::post( __FUNCTION__, "NULL volume specified." );
#endif // DEBUG
    imgToPatient->matrix = SbMatrix::identity();
    return FALSE;
  }

  if ( !dicomAdjustVolume( volume, FALSE ) )
  {
#ifdef DEBUG
    SoDebugError::post( __FUNCTION__, "Error while adjusting volume extent." );
#endif // DEBUG
    imgToPatient->matrix = SbMatrix::identity();
    return FALSE;
  }

  SoVRDicomFileReader* reader = dynamic_cast<SoVRDicomFileReader*>(volume->getReader());

  if ( reader == NULL )
  {
#ifdef DEBUG
    SoDebugError::post( __FUNCTION__, "Cannot retrive DICOM reader." );
#endif // DEBUG
    imgToPatient->matrix = SbMatrix::identity();
    return FALSE;
  }

  SbBox3f extent = volume->extent.getValue();
  SbVec3i32 dims = volume->data.getSize();

  if ( extent.getSize()[0] == 0.0f || extent.getSize()[1] == 0.0f )
  {
#ifdef DEBUG
    SoDebugError::post( __FUNCTION__, "Volume extent is empty. Cannot define spaces." );
#endif // DEBUG
    imgToPatient->matrix = SbMatrix::identity();
    return FALSE;
  }

  if ( dims[0] == 0 || dims[1] == 0 || dims[2] == 0 )
  {
#ifdef DEBUG
    SoDebugError::post( __FUNCTION__, "Image has no dimension. Cannot define spaces." );
#endif // DEBUG
    imgToPatient->matrix = SbMatrix::identity();
    return FALSE;
  }

  // Locate volume in DICOM image space: Center of top left voxel should be at (0, 0, 0)
  // By default, VolumeViz locate center of volume in origin.
  SbVec3f halfVoxelSize = 0.5 * SbVec3f( extent.getSize()[0] / dims[0],
    extent.getSize()[1] / dims[1],
    extent.getSize()[2] / dims[2] );

  SbVec3f volumeSize = extent.getSize();
  volume->extent.setValue( -halfVoxelSize, volumeSize - halfVoxelSize );


  // Origin of center of top left voxel expressed in Patient space as defined in DICOM spec Patient Position (0020, 0032)
  SbVec3f OImgInPatient = reader->getDicomData().getImagePosition();

  // Build matrix of Image space expressed in patient space.
  // warning: OIV user row vector convention.
  // ImageOrientX x y z 0
  // ImageOrientY x y z 0
  // ImageOrientZ x y z 0
  // PatientPos   x y z 1
  SbMatrix ImgToPatientMatrix = SbMatrix::identity();
  reader->getDicomData().getOrientation( ImgToPatientMatrix );
  ImgToPatientMatrix[3][0] = OImgInPatient[0];
  ImgToPatientMatrix[3][1] = OImgInPatient[1];
  ImgToPatientMatrix[3][2] = OImgInPatient[2];

  imgToPatient->matrix = ImgToPatientMatrix;

  return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
// Adjust data range based on values in the DICOM file,
// i.e. the window center (0028,1050) and window width (0028,1051) values.
// Returns true if successful. Assigns default values if necessary.
// Volume is needed to get DICOM attributes, but is not modified.
// 
SbBool
MedicalHelper::dicomAdjustDataRange( SoDataRange* rangeNode, const SoVolumeData* volume )
{
  if (rangeNode == NULL || volume == NULL)
    return FALSE;

  // Note the get... function handles volume==NULL automatically.
  SbVec2f winCW;
  SbBool rc = MedicalHelper::dicomGetWindowCenterWidth( volume, winCW );
  rangeNode->min = winCW[0] - 0.5f * winCW[1];
  rangeNode->max = winCW[0] + 0.5f * winCW[1];
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Automatically adjusts the color map (transferFunction) if the specified
// volume is a DICOM data set (reader is SoVRDicomFileReader) and the
// the Photometric Interpretation (0028,0004) value is MONOCHROME1.
// Specifically, if the above conditions are true, this method reverses the
// values in the current color map.
//
// Typically used as a convenience call when rendering slices.  For intensity
// based rendering of slices, set the transfer function to the predefined color
// map INTENSITY.  This is an intensity ramp from black to white that is
// appropriate for Photometric Interpretation MONOCHROME2 (the most common
// case).  When reversed it is appropriate for P.I. MONOCHROME1.
//
// Returns true if change was needed and was successful.
// Volume is needed to get DICOM attributes, but is not modified.
// Transfer function is modified if necessary.
SbBool
MedicalHelper::dicomCheckMonochrome1( SoTransferFunction* cmapNode,
                                      const SoVolumeData* volume,
                                      SbBool forceReverse )
{
  const unsigned int PHOTOMETRIC_INTERP_GROUP   = 0x0028;
  const unsigned int PHOTOMETRIC_INTERP_ELEMENT = 0x0004;

  if (cmapNode == NULL)
    return FALSE;

  SbBool rc = FALSE;
  SbBool isMonochrome1 = FALSE;
  
  if (volume != NULL) {
    const SoVolumeReader* reader = volume->getReader();
    if (reader != NULL && reader->isOfType(SoVRDicomFileReader::getClassTypeId())) {
      const SoVRDicomData& dicomData = ((SoVRDicomFileReader*)reader)->getDicomData();
      const SbString valueStr = dicomData.getDicomInfo( PHOTOMETRIC_INTERP_GROUP, PHOTOMETRIC_INTERP_ELEMENT );
      if (valueStr == SbString("MONOCHROME1")) {
        // This value for PhotometricInterp means increasing voxel values go from light to dark.
        isMonochrome1 = TRUE;
      }
    }
  }

  if (isMonochrome1 || forceReverse) {
    rc = TRUE;
    // Note that we cannot simply reverse the float values in the color map field.
    // We must reverse the actual colormap entries. 4 floats per entry for RGBA.
    int numFloats  = cmapNode->actualColorMap.getNum();
    float* dst = new float[numFloats]; // Temporary storage for reversed colormap

    int type = cmapNode->colorMapType.getValue();
    if (type == SoTransferFunction::LUM_ALPHA) { // 2 values per entry
      int numEntries = numFloats / 2;
      const SbVec2f* src = (SbVec2f*)cmapNode->actualColorMap.getValues(0);
      SbVec2f* dst2 = (SbVec2f*)dst;
      for (int i = 0; i < numEntries; ++i) {
        dst2[i] = src[numEntries - 1 - i];
      }
    }
    else if (type == SoTransferFunction::ALPHA) { // 1 value per entry
      const float* src = cmapNode->actualColorMap.getValues(0);
      for (int i = 0; i < numFloats; ++i) {
        dst[i] = src[numFloats - 1 - i];
      }
    }
    else {  // Default is RGBA - 4 values per entry
      int numEntries = numFloats / 4;
      const SbVec4f* src = (SbVec4f*)cmapNode->actualColorMap.getValues(0);
      SbVec4f* dst4 = (SbVec4f*)dst;
      for (int i = 0; i < numEntries; ++i) {
        dst4[i] = src[numEntries - 1 - i];
      }
    }
    // Replace color map
    cmapNode->predefColorMap = SoTransferFunction::NONE;
    cmapNode->colorMapType   = type;
    cmapNode->colorMap.setValues( 0, numFloats, (float*)dst );
    delete [] dst;
  }
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Given the path to a DICOM file, returns a list containing all the files in
// the same directory that are part of the same DICOM series, based on the
// series UID (0x0020,0x000E). Returns 0 if the specified file was not found
// or is not a valid DICOM file.
//
// (0020,000E) Series instance UID (required)
//
// We do not use:
// (0020,0011) Series number (required but may be "unknown")
// (0008,103E) Series description (optional)

int
MedicalHelper::dicomFindFilesbySeries( const SbString& firstFile, std::vector<SbString>& files )
{
  const unsigned int SERIES_UID_GROUP   = 0x0020;
  const unsigned int SERIES_UID_ELEMENT = 0x000E;

  // Clear the list
  files.clear();

  // Return if no file name (SbString declares that all "null" are also "empty").
  if (firstFile.isEmpty())
    return 0;

  // Open first file if possible
  SoVRDicomData dicomBase;
  SbBool rc = dicomBase.readDicomHeader( firstFile );
  if (! rc)
    return 0;
  
  // Get the series UID we're trying to match.
  const SbString seriesUID = dicomBase.getDicomInfo( SERIES_UID_GROUP, SERIES_UID_ELEMENT );
  if (seriesUID.isEmpty() || seriesUID.getLength() == 0)
    return 0;

  // Get the path to the directory we're exploring
  SbString basePath = SbFileHelper::getDirName( firstFile );

  // Get a list of all the files in this directory.
  // Note that DICOM does not require any particular file extension...
  std::vector<SbString> fileList;
  SbFileHelper::listFiles( basePath, "*.*", fileList );
  if (fileList.size() == 0)
    return 0;

  // Build a new list containing the files that belong to this series.
  SoVRDicomData dicom;
  SbString path, uid;
  for (std::vector<SbString>::iterator it = fileList.begin();
       it != fileList.end(); ++it) {

    path = basePath + *it;
    if (dicom.readDicomHeader( path )) {
      uid = dicom.getDicomInfo( SERIES_UID_GROUP, SERIES_UID_ELEMENT );
      if (uid == seriesUID) {
        files.push_back( path );
      }
    }
  }
  return (int)files.size();
}

///////////////////////////////////////////////////////////////////////////////
// Given the path to a DICOM file, returns a list containing all the files in
// the same directory that are part of the same DICOM series, based on the
// series UID (0x0020,0x000E). Returns 0 if the specified file was not found
// or is not a valid DICOM file.
//
// Alternate version that returns an SbStringList.
int
MedicalHelper::dicomFindFilesbySeries( const SbString& firstFile, SbStringList& files )
{
  // Get the file list
  std::vector<SbString> filelist;
  int numFiles = dicomFindFilesbySeries( firstFile, filelist );

  // Convert to SbStringList
  files.truncate( 0 );
  for (int i = 0; i < numFiles; ++i) {
    files.append( const_cast<SbString*>(&(filelist[i])) );
  }
  return numFiles;
}

///////////////////////////////////////////////////////////////////////////////
// Make scene appear larger or smaller
void
MedicalHelper::dollyZoom( float value, SoCamera* camera )
{
  if (camera != NULL && value > 0) {
    SoRef<SoCameraInteractor> interactor = new SoCameraInteractor(camera);
    interactor->dolly( 1 / value );
  }
}


///////////////////////////////////////////////////////////////////////////////
// Set filename list using a vector of strings.
//
// Automatically creates the SbStringList object required by SoVRDicomFileReader.
// An SbStringList object is a list of _pointers_, not a list of objects.  So
// deleting the object just deletes the pointers, not the related objects.  In
// this case that's not a problem because we want the vector to continue to exist.
SbBool
MedicalHelper::setFilenameList( SoVRDicomFileReader& reader, const std::vector<SbString>& list )
{
  SbStringList tempList;
  int numStr = (int)list.size();
  for (int i = 0; i < numStr; ++i) {
    tempList.append( const_cast<SbString*>(&(list[i])) );
  }
  SbBool rc = reader.setFilenameList( tempList );
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Returns the voxel size in 3D space.
// This is just a convenience.  The voxel size is simply the volume extent
// in 3D space divided by the number of voxels.
//
SbVec3f
MedicalHelper::getVoxelSize( const SoVolumeData* volume ) const
{
  if (volume == NULL)
    return SbVec3f(0,0,0);

  const SbBox3f& extent = volume->extent.getValue();
  const SbVec3f  size   = extent.getSize();
  const SbVec3i32& dim  = volume->data.getSize();
  const SbVec3f voxelSize( size[0]/dim[0], size[1]/dim[1], size[2]/dim[2] );
  return voxelSize;
}

///////////////////////////////////////////////////////////////////////////////
// Returns the volume's physical size in 3D space.
// This is just a convenience.  The physical size in 3D is simply the span
// of the volume extent.
//
SbVec3f
MedicalHelper::getPhysicalSize( const SoVolumeData* volume ) const
{
  if (volume == NULL)
    return SbVec3f(0,0,0);

  const SbBox3f& extent = volume->extent.getValue();
  return extent.getSize();
}

///////////////////////////////////////////////////////////////////////////////
// Returns the DICOM volume origin in 3D space.
// This is just a convenience.
// In DICOM terms the image/volume origin is the @I center@i of the first
// voxel, but in VolumeViz the minimum point of the 3D extent is the outer
// edge of the first voxel.  The difference is 1/2 voxel.
//
// Note: dicomGetImagePosition returns the actual value of the Image
// Position Patient attribute from the DICOM file (if possible).
//
SbVec3f
MedicalHelper::getDicomOrigin( const SoVolumeData* volume ) const
{
  if (volume == NULL)
    return SbVec3f(0,0,0);

  const SbBox3f& extent = volume->extent.getValue();
  const SbVec3f  size   = extent.getSize();
  const SbVec3i32& dim  = volume->data.getSize();
  const SbVec3f voxelSize( size[0]/dim[0], size[1]/dim[1], size[2]/dim[2] );

  const SbVec3f  origin = extent.getMin() + (voxelSize * 0.5f);
  return origin;
}

///////////////////////////////////////////////////////////////////////////////
// Get the "Image Position (Patient)" attribute (0020,0032) from a DICOM volume.
//  DICOM calls this the upper left hand corner of the image, but more precisely
//  it's the @I center@i of the first voxel, in millimeters (mm).  For VolumeViz the
//  upper left hand corner of the image is literally the @I corner@i of the voxel,
//  one-half voxel different from Image Position.
//  Returns true if the query is successful.  If not successful, imagePos is 0,0,0.
//  If there are multiple slices, value is taken from the first slice.
// 
//  Note the SoVRDicomData getPosition() method does @I not@i return the value of
//  the Image Position attribute. It returns a value computed from Image Position.
// 
SbBool
MedicalHelper::dicomGetImagePosition( const SoVolumeData* volume, SbVec3f& imagePos )
{
  SbBool  rc = FALSE;
  SbVec3f pos(0,0,0);
  if (volume != NULL) {
    SoVolumeReader* reader = volume->getReader();

    if (reader != NULL && reader->isOfType(SoVRDicomFileReader::getClassTypeId())) {
      SoVRDicomFileReader* dicomReader = (SoVRDicomFileReader*)reader;
      const SoVRDicomData& dicomData = dicomReader->getDicomData();

      //  The string will be something like: "-104.5\\-121.6\\-205" so it's
      //  easier to use the old fashioned sscanf than parse a stringstream.
      SbString str = dicomData.getDicomInfo( 0x0020, 0x0032 ); // "Image position" tag
      if (! str.isEmpty() && str.getLength() > 0) {
        SbVec3f testPos;
        int cnt = sscanf( str.toLatin1(), "%g%*c%g%*c%g",
                          &testPos[0], &testPos[1], &testPos[2] );
        if (cnt == 3) {
          pos = testPos;
          rc = TRUE;
        }
      }
    }
  }
  imagePos = pos;
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Get the window center (level) and width values from a DICOM volume.
// If the query fails, returns false and sets 'winCW' to 0,0.
//
// Uses the Window Center (0028,1050) and Window Width (0028,1051) tags from the
// first image in the stack. If these tags do not exist in the volume, then window
// center and width are computed from the actual data min and max values. (Note
// that querying the actual min and max values may take some time because every
// voxel must be loaded.)
// This method may be called with a non-DICOM volume. In that case the actual
// min and max data values are used.
//
SbBool
MedicalHelper::dicomGetWindowCenterWidth( const SoVolumeData* volume, SbVec2f& winCW )
{
  SbBool rc = FALSE;
  float center = 0;
  float width  = 0;
  if (volume != NULL) {
    SoVolumeReader* reader = volume->getReader();

    if (reader != NULL && reader->isOfType(SoVRDicomFileReader::getClassTypeId())) {
      SoVRDicomFileReader* dicomReader = (SoVRDicomFileReader*)reader;

      const SoVRDicomData& dicomData = dicomReader->getDicomData();
      SbString str = dicomData.getDicomInfo( 0x0028, 0x1050 ); // Window Center tag
      if (! str.isEmpty() && str.getLength() > 0) {
        center = str.toFloat();
        str = dicomData.getDicomInfo( 0x0028, 0x1051 );        // Window width tag
        if (! str.isEmpty() && str.getLength() > 0) {
          width = str.toFloat();
          rc = TRUE;
        }
      }
    }
    if (! rc) { // Tags were not found or this is not a DICOM volume.
      double volmin, volmax;
      const_cast<SoVolumeData*>(volume)->getMinMax( volmin, volmax );
      width  = (float)(volmax - volmin);
      center = (float)(volmin + 0.5f * width);
    }
  }
  winCW.setValue( center, width );
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Get the window center (level) and width values from an SoDataRange node.
// 
SbVec2f
MedicalHelper::dicomGetWindowCenterWidth( const SoDataRange* dataRange )
{
  if (dataRange == NULL)
    return SbVec2f(0,0);

  const float rmin = (float)dataRange->min.getValue();
  const float rmax = (float)dataRange->max.getValue();
  float width  = rmax - rmin;
  float center = rmin + 0.5f * width;
  return SbVec2f(center,width);
}

///////////////////////////////////////////////////////////////////////////////
// Set an SoDataRange node from the window center (level) and width values.
// 
SbBool
MedicalHelper::dicomSetWindowCenterWidth( SoDataRange* dataRange, const SbVec2f& winCW )
{
  if (dataRange == NULL)
    return FALSE;

  float width = std::max( winCW[1], 1.0f ); // Width < 1 not allowed in DICOM.
  dataRange->min = winCW[0] - 0.5f * width;
  dataRange->max = winCW[0] + 0.5f * width;
  return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
// Build a scene graph to display slice orientation markers.
//
// This is just a convenience method and helps keep the demo/example programs
// consistent.  Applications can use SliceOrientationMarkers directly.
//
// Creates a SliceOrientationMarkers node with a default color and background
// and connects its 'axis' field to the 'axis' field of the specified slice node.
SoSeparator*
MedicalHelper::buildSliceOrientationMarkers( const SoOrthoSlice* orthoSlice)
{
  SoSeparator* markerSep = new SoSeparator();

  // Color for slice orientation marker text.
  SoMaterial* textMat = new SoMaterial();
    textMat->diffuseColor.setValue(1, 1, 0.25f);
    markerSep->addChild(textMat);

  // Enable background for text so it's visible against the image.
  SoTextProperty* textProp = new SoTextProperty();
    textProp->margin = 0.1f;
    textProp->style  = SoTextProperty::BACK_FRAME;
    textProp->styleColors.set1Value( 3, SbColorRGBA(0, 0.1f, 0.1f, 0.25f) );
    markerSep->addChild(textProp);

  // Slice orientation markers.
  // Connect the axis field to the slice so the node will update automatically.
  SliceOrientationMarkers* markers = new SliceOrientationMarkers();
    if (orthoSlice != NULL)
      markers->axis.connectFrom( (SoField*)&orthoSlice->axis ); // Need cast to avoid ambiguity
    markerSep->addChild(markers);

  return markerSep;
}


///////////////////////////////////////////////////////////////////////////////
// Build a scene graph to display dynamic scale bars for slice viewing.
//
// This is just a convenience method and helps keep the demo/example programs
// consistent.  Applications can use SliceScaleBar directly.
//
// Note that a typical length, 10 cm, is assumed.
//
// Creates horizontal and vertical SliceScaleBar nodes with a default color and
// line width and sets their 'trackedCamera' field to the specified camera node.
SoSeparator*
MedicalHelper::buildSliceScaleBars( const SoCamera* camera)
{
  SoSeparator* scaleSep = new SoSeparator();

  // Color for scale bars
  SoMaterial* scaleColor = new SoMaterial();
    scaleColor->diffuseColor.setValue(1, 0.25f, 0.25f);
    scaleSep->addChild(scaleColor);

  // Wider lines for better visibility
  SoDrawStyle* scaleStyle = new SoDrawStyle();
    scaleStyle->lineWidth = 2;
    scaleSep->addChild(scaleStyle);

  // Horizontal scale bar
  SliceScaleBar* scale1 = new SliceScaleBar();
    scale1->position.setValue(0, -0.99f);
    scale1->length = 100; // 10 cm
    if (camera != NULL)
      scale1->trackedCamera = (SoNode*)camera;
    scale1->label = "10cm";
    scaleSep->addChild(scale1);

  // Vertical scale bar
  SliceScaleBar* scale2 = new SliceScaleBar();
    scale2->position.setValue(-0.99f, 0);
    scale2->length = 100; // 10 cm
    scale2->orientation = SliceScaleBar::VERTICAL;
    if (camera != NULL)
      scale2->trackedCamera = (SoNode*)camera;
    scale2->label = "10cm";
    scaleSep->addChild(scale2);

  return scaleSep;
}

       
///////////////////////////////////////////////////////////////////////////////
// Slice viewer annotations.
//
// This is just a convenience method and helps keep the demo/example programs
// consistent.  Applications can use SliceScaleBar (etc) directly.
//
// Calls the #buildSliceOrientationMarkers() and #buildSliceScaleBars() methods
// and also creates some DicomInfo nodes using tag info from the optional
// image filename.
SoSeparator*
MedicalHelper::buildSliceAnnotation( const SoCamera* camera, 
                                     const SoOrthoSlice* sliceNode,
                                     const SbString* filename )
{
  SoSeparator* root = new SoSeparator();

  if (camera == NULL || sliceNode == NULL)
    return root;

  // Slice orientation markers.
  root->addChild( MedicalHelper::buildSliceOrientationMarkers(sliceNode));

  // Dynamic scale bars
  root->addChild( MedicalHelper::buildSliceScaleBars(camera));

  // DICOM slice annotation
  if (filename != NULL && ! filename->isEmpty())
  {
    SoMaterial* dicomMatl = new SoMaterial();
    dicomMatl->diffuseColor.setValue(0.8f, 0.8f, 0.5f);
    root->addChild(dicomMatl);

    DicomInfo* upperLeft = new DicomInfo();
    upperLeft->fileName = *filename;
    upperLeft->position.setValue(-0.99f, 0.99f, 0);
    upperLeft->alignmentV = DicomInfo::TOP;
    upperLeft->displayDicomInfo("", 0x0010, 0x0010); // Patient Name
    upperLeft->displayDicomInfo("", 0x0010, 0x0030); // Patient Birth Date
    upperLeft->displayDicomInfo("", 0x0008, 0x1030); // Study Description
    upperLeft->displayDicomInfo("", 0x0008, 0x103E); // Series Description
    root->addChild(upperLeft);

    DicomInfo* upperRight = new DicomInfo();
    upperRight->fileName = *filename;
    upperRight->position.setValue(0.99f, 0.99f, 0);
    upperRight->alignmentH = DicomInfo::RIGHT;
    upperRight->alignmentV = DicomInfo::TOP;
    upperRight->textAlignH = DicomInfo::RIGHT;
    upperRight->displayDicomInfo("", 0x0008, 0x0080); // Institution
    upperRight->displayDicomInfo("", 0x0008, 0x0090); // Physician
    upperRight->displayDicomInfo("", 0x0008, 0x1090); // Model name
    root->addChild(upperRight);

    DicomInfo* lowerRight = new DicomInfo();
    lowerRight->fileName = *filename;
    lowerRight->position.setValue(0.99f, -0.99f, 0);
    lowerRight->alignmentH = DicomInfo::RIGHT;
    lowerRight->alignmentV = DicomInfo::BOTTOM;
    lowerRight->textAlignH = DicomInfo::RIGHT;
    lowerRight->displayDicomInfo("", 0x0008, 0x0060); // Modality
    lowerRight->displayDicomInfo("mA: ", 0x0018, 0x1151); // X-Ray Tube Current
    lowerRight->displayDicomInfo("kV: ", 0x0018, 0x0060); // KVP (Kilo Voltage Peak)
    lowerRight->displayDicomInfo("", 0x0008, 0x0022); // Acquisition date
    root->addChild(lowerRight);

    TextBox* lowerLeft = new TextBox();
    lowerLeft->position.setValue(-0.99f, -0.94f, 0); // Leave room for OIV logo
    lowerLeft->alignmentV = TextBox::BOTTOM;
    SbString str;
    str.sprintf( "Image: %d", sliceNode->sliceNumber.getValue() );
    lowerLeft->addLine( str );
    root->addChild(lowerLeft);
  }

  return root;
}


///////////////////////////////////////////////////////////////////////////////
const SbVec2s&
MedicalHelper::exampleWindowSize()
{
  return ExampleWindowSize;
}

///////////////////////////////////////////////////////////////////////////////
/** Returns a logo image node for Open Inventor medical examples. */
SoNode*
MedicalHelper::exampleLogoNode()
{
  SoAnnotation *annotation = new SoAnnotation();
    SoImageBackground *imgBg = new SoImageBackground();
    imgBg->filename.setValue( LogoImageFile );
    imgBg->style.setValue( SoImageBackground::LOWER_LEFT );
    annotation->addChild(imgBg);
  return annotation;
}

///////////////////////////////////////////////////////////////////////////////
// Returns a collection of DicomInfo annotation nodes for Open Inventor medical examples.
// This method is not important for customer appications, but the technique
// can be useful for adding annotation text to an application window. @BR
// filename is the path to a DICOM file from which info will be extracted.
//
SoNode*
MedicalHelper::exampleDicomAnnotation( const SbString& filename )
{
  SoSeparator* mainSep = new SoSeparator();
  if (filename.isEmpty())
    return mainSep;

  // Top left Dicom annotation
  DicomInfo* topLeftDicomAnnotation = new DicomInfo();
  topLeftDicomAnnotation->fileName.setValue( filename );
  topLeftDicomAnnotation->position.setValue(SbVec3f(-0.99f, 0.99f, 0));

  // Add text to annotation
  topLeftDicomAnnotation->displayDicomInfo("Patient Name : ",          0x10, 0x0010);
  topLeftDicomAnnotation->displayDicomInfo("Patient Id : "  ,          0x10, 0x0020);
  topLeftDicomAnnotation->displayDicomInfo("Institution Name : ",      0x08, 0x0080);
  topLeftDicomAnnotation->displayDicomInfo("Modality : ",              0x08, 0x0060);
  topLeftDicomAnnotation->displayDicomInfo("Patient Pos : ",           0x18, 0x5100);
  topLeftDicomAnnotation->displayDicomInfo("Patient orientation : ",   0x20, 0x0020);
  topLeftDicomAnnotation->displayDicomInfo("samples per pixel : ",     0x28, 0x0002);
  topLeftDicomAnnotation->displayDicomInfo("Photometric interp : ",    0x28, 0x0004);
  topLeftDicomAnnotation->displayDicomInfo("Image Orientation : ",     0x20, 0x0037);
  topLeftDicomAnnotation->displayDicomInfo("Image rows (height) : ",   0x28, 0x0010);
  topLeftDicomAnnotation->displayDicomInfo("Image columns (width) : ", 0x28, 0x0011);

  mainSep->addChild(topLeftDicomAnnotation);

  // Bottom right Dicom annotation
  DicomInfo* bottomRightDicomAnnotation = new DicomInfo();
  bottomRightDicomAnnotation->alignmentH = DicomInfo::RIGHT;
  bottomRightDicomAnnotation->alignmentV = DicomInfo::BOTTOM;
  bottomRightDicomAnnotation->fileName.setValue( filename );
  bottomRightDicomAnnotation->position.setValue(SbVec3f(0.99f, -0.99f, 0));

  // Add text to annotation
  bottomRightDicomAnnotation->displayDicomInfo("Convolution Kernel : ", 0x0018, 0x1210);
  bottomRightDicomAnnotation->displayDicomInfo("Thickness : ",          0x0018, 0x0050);
  bottomRightDicomAnnotation->displayDicomInfo("kVp : ",                0x0018, 0x0060);

  mainSep->addChild(bottomRightDicomAnnotation);

  // Top right Dicom annotation
  DicomInfo* topRightDicomAnnotation = new DicomInfo();
  topRightDicomAnnotation->alignmentH = DicomInfo::RIGHT;
  topRightDicomAnnotation->fileName.setValue( filename );
  topRightDicomAnnotation->position.setValue(SbVec3f(0.99f, 0.99f, 0));

  // Bottom right Dicom annotation
  topRightDicomAnnotation->displayDicomInfo("Patient Birthday : ", 0x0010, 0x0030 );
  topRightDicomAnnotation->displayDicomInfo("Aquisition date : " , 0x0008, 0x0022 );
  topRightDicomAnnotation->displayDicomInfo("Patient's Age : "   , 0x0010, 0x1010 );
  topRightDicomAnnotation->displayDicomInfo("Sex : "             , 0x0010, 0x0040 );

  mainSep->addChild(topRightDicomAnnotation);

  return mainSep;
}


///////////////////////////////////////////////////////////////////////////////
// Convenience function to retrieve bounding box of specified node.
// If node contains SoVolumeData, use SoVolumeData extent.
SbBox3f
MedicalHelper::getBoundingBox( SoNode* node )
{
  // retrieve root node of volume data and compute its bounding box.
  // Try to find an SoVolumeData in specified volumeData. If SoVolumeData found, 
  // use its extend as bbox. Else, apply an SoGetBoundingBoxAction on g_volumeDataRoot.
  SoVolumeData* volumeData = find<SoVolumeData>( node );
  if ( volumeData != NULL )
  {
    return volumeData->extent.getValue();
  }
  else
  {
    // Use fake viewport for SoGetBoundingBoxAction. see SoGetBoundingBoxAction 
    // constructor for details.
    SbViewportRegion fakeViewportRegion(1, 1);
    SoGetBoundingBoxAction gba( fakeViewportRegion );
    gba.apply( node );
    return gba.getBoundingBox();
  }
}

///////////////////////////////////////////////////////////////////////////////
// Convenience function to retrieve Cube corresponding to specified bbox.
SoSeparator*
MedicalHelper::createCube( const SbBox3f& bbox )
{
  SoSeparator* bboxSep = new SoSeparator;
  SoTransform* transform = new SoTransform;
  transform->translation = bbox.getCenter();
  transform->scaleFactor = bbox.getSize()/2.0f;
  bboxSep->addChild(transform);
 
  SoCube* cube = new SoCube;
  bboxSep->addChild(cube);

  return bboxSep;
}

///////////////////////////////////////////////////////////////////////////////
// Convenience function to draw a specified bounding box as a wireframe.
SoSeparator*
MedicalHelper::createBoundingBox( const SbBox3f& bbox, SbColor* color )
{
  SoSeparator* bboxSep = new SoSeparator;
    bboxSep->setName( "BoxOutline" );

  // Box should not interfere with picking other geometry
  SoPickStyle* pickStyle = new SoPickStyle();
    pickStyle->style = SoPickStyle::UNPICKABLE;
    bboxSep->addChild( pickStyle );

  // The box will be easier to see without lighting
  SoLightModel* lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    bboxSep->addChild( lightModel );

  // For convenience, force line drawing so we don't inherit something different.
  SoDrawStyle* drawStyle = new SoDrawStyle();
    drawStyle->style = SoDrawStyle::LINES;
    bboxSep->addChild( drawStyle );

  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  bbox.getBounds( xmin,ymin, zmin, xmax, ymax, zmax );
  SoVertexProperty* vProp = new SoVertexProperty;
  vProp->vertex.set1Value( 0, SbVec3f(xmin,ymin,zmin) );
  vProp->vertex.set1Value( 1, SbVec3f(xmax,ymin,zmin) );
  vProp->vertex.set1Value( 2, SbVec3f(xmax,ymax,zmin) );
  vProp->vertex.set1Value( 3, SbVec3f(xmin,ymax,zmin) );
  vProp->vertex.set1Value( 4, SbVec3f(xmin,ymin,zmax) );
  vProp->vertex.set1Value( 5, SbVec3f(xmax,ymin,zmax) );
  vProp->vertex.set1Value( 6, SbVec3f(xmax,ymax,zmax) );
  vProp->vertex.set1Value( 7, SbVec3f(xmin,ymax,zmax) );

  // Set color if specified
  if (color == NULL)
    vProp->orderedRGBA.set1Value( 0, 0xFF0000FF );
  else
    vProp->orderedRGBA.set1Value( 0, color->getPackedValue() );

  // Draw it with a line set
  int coordIndices[] = {0,1,2,3,0,-1,4,5,6,7,4,-1,
                        0,4,-1, 1,5,-1, 2,6,-1, 3,7};
  int numCoordIndices = sizeof(coordIndices)/sizeof(int);

  SoIndexedLineSet* lineset = new SoIndexedLineSet;
    lineset->vertexProperty = vProp;
    lineset->coordIndex.setValues( 0, numCoordIndices, coordIndices );
    bboxSep->addChild( lineset );

  return bboxSep;
}

///////////////////////////////////////////////////////////////////////////////
SoSeparator *
MedicalHelper::readFile(const char *filename)
{
  // Open the input file
  SoInput mySceneInput;
  if (!mySceneInput.openFile(filename))
  {
    std::cerr << "Cannot open file " << filename << std::endl;
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator *myGraph = SoDB::readAll(&mySceneInput);
  if (myGraph == NULL)
  {
    std::cerr << "Problem reading file" << std::endl;
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

///////////////////////////////////////////////////////////////////////////////
// Get index of primary axis (max component) of vector.
// Note that we need the index of the largest value, not the largest value.
static int
getIndexOfMaxVal(const SbVec3f& vec)
{
  const SbVec3f absVec(fabs(vec[0]), fabs(vec[1]), fabs(vec[2]));
  int maxComp = 0;
  if (absVec[1] > absVec[0])
  {
    if (absVec[2] > absVec[1])
      maxComp = 2;
    else
      maxComp = 1;
  }
  else if (absVec[2] > absVec[0])
    maxComp = 2;
  return maxComp;
}

///////////////////////////////////////////////////////////////////////////////
// Return the medical axis (axial, coronal, sagittal) corresponding to the
// specified view axis (X, Y, Z) based on DICOM Image Orientation matrix.
MedicalHelper::Axis
MedicalHelper::MedicalAxisFromViewAxis( openinventor::inventor::Axis::Type viewAxis, const SbMatrix& orientationMatrix )
{
  static const SbVec3f viewVectors[3] = {{1,0,0},{0,1,0},{0,0,1}};

  SbVec3f medVector;
  orientationMatrix.multVecMatrix(viewVectors[static_cast<int>(viewAxis)], medVector);

  return static_cast<MedicalHelper::Axis>(getIndexOfMaxVal(medVector));
}

///////////////////////////////////////////////////////////////////////////////
// Return the medical axis (axial, coronal, sagittal) corresponding to the
// specified view axis (X, Y, Z) based on DICOM Image Orientation element.
MedicalHelper::Axis
MedicalHelper::MedicalAxisFromViewAxis( openinventor::inventor::Axis::Type viewAxis, const SoVolumeData* volData )
{
  if (volData == nullptr)
    return static_cast<MedicalHelper::Axis>(viewAxis);

  // Must be using a DICOM reader to get the Image Orientation element.
  SoVRDicomFileReader* reader = dynamic_cast<SoVRDicomFileReader*>(volData->getReader());
  if (reader == nullptr)
    return static_cast<MedicalHelper::Axis>(viewAxis);

  // Get DICOM orientation matrix from Image Orientation element (0x0020,0x0037).
  // Note:
  //   - This query returns identity matrix if element is not available.
  //   - The DICOM direction cosines are the first and second columns of this matrix.
  //     Third column is the cross-product.
  SbMatrix orientationMatrix;
  reader->getDicomData().getOrientation(orientationMatrix);

  // Get the medical axis using this matrix.
  return MedicalAxisFromViewAxis(viewAxis, orientationMatrix);
}

///////////////////////////////////////////////////////////////////////////////
// Returns the view axis(X, Y, Z) corresponding to the specified medical axis
// (AXIAL, CORONAL, SAGITTAL)based on the DICOM Image Orientation element(0x0020, 0x0037).
openinventor::inventor::Axis::Type
MedicalHelper::ViewAxisFromMedicalAxis(Axis medicalAxis, const SbMatrix& orientationMatrix)
{
  static const SbVec3f vectors[3] = {{1,0,0},{0,1,0},{0,0,1}};

  // Image orientation matrix describes the rotation of the volume XYZ axes,
  // i.e. rotation from view axis to medical axis.  Here we need the inverse
  // of that rotation, which is the transpose of the rotation matrix.
  SbVec3f viewVector;
  orientationMatrix.transpose().multVecMatrix(vectors[static_cast<int>(medicalAxis)], viewVector);

  return static_cast<openinventor::inventor::Axis::Type>(getIndexOfMaxVal(viewVector));
}

///////////////////////////////////////////////////////////////////////////////
// Returns the view axis(X, Y, Z) corresponding to the specified medical axis
// (AXIAL, CORONAL, SAGITTAL)based on the DICOM Image Orientation element(0x0020, 0x0037).
openinventor::inventor::Axis::Type
MedicalHelper::ViewAxisFromMedicalAxis(Axis medicalAxis, const SoVolumeData* volData)
{
  if (volData == nullptr)
    return static_cast<openinventor::inventor::Axis::Type>(medicalAxis);

  // Must be using a DICOM reader to get the Image Orientation element.
  SoVRDicomFileReader* reader = dynamic_cast<SoVRDicomFileReader*>(volData->getReader());
  if (reader == nullptr)
    return static_cast<openinventor::inventor::Axis::Type>(medicalAxis);

  // Get DICOM orientation matrix from Image Orientation element (0x0020,0x0037).
  // Note:
  //   - This query returns identity matrix if element is not available.
  //   - The DICOM direction cosines are the first and second columns of this matrix.
  //     Third column is the cross-product.
  SbMatrix orientationMatrix;
  reader->getDicomData().getOrientation(orientationMatrix);

  // Get the medical axis using this matrix.
  return ViewAxisFromMedicalAxis(medicalAxis, orientationMatrix);
}

#ifdef _WIN32
#pragma warning( pop )
#endif
