// DICOM annotation utility class

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Medical/nodes/DicomInfo.h>

#include <VolumeViz/readers/SoVRDicomData.h>

SO_NODE_SOURCE(DicomInfo);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
DicomInfo::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO__NODE_INIT_CLASS(DicomInfo, "DicomInfo", TextBox);
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
DicomInfo::exitClass()
{
  SO__NODE_EXIT_CLASS(DicomInfo);
}

///////////////////////////////////////////////////////////////////////////////
DicomInfo::DicomInfo()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(DicomInfo);
  SO_NODE_ADD_FIELD( fileName, (""));

  // DICOM reader
  m_dicomReader = new SoVRDicomData();
}

///////////////////////////////////////////////////////////////////////////////
DicomInfo::~DicomInfo()
{
  delete m_dicomReader;
}

///////////////////////////////////////////////////////////////////////////////
// Check that we have data from the correct file
bool
DicomInfo::checkFilename()
{
  bool rc = true;
  if (m_filename != fileName.getValue()) {
    m_filename = fileName.getValue();
    rc = false;
    if (! m_filename.isEmpty())
      rc = m_dicomReader->readDicomHeader( m_filename );
  }
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
int
DicomInfo::displayDicomInfo( SbString title, unsigned short group, unsigned short element ) 
{
  checkFilename();
  SbString str = m_dicomReader->getDicomInfo( group, element );
  // Check for dates
  // TODO: Need to get access to the DICOM "value representation" code.
  bool isDate = false;
  bool isTime = false;
  if (group == 0x08) {
    if (element == 0x12 || (element >= 0x20 && element <= 0x23)) {
      isDate = true;
    }
    else if (element == 0x13 || (element >= 0x2A && element <= 0x35)) {
      isTime = true;
    }
  }
  else if (group == 0x10) {
    if (element == 0x30)
      isDate = true;
    else if (element == 0x32)
      isTime = true;
  }
  if (isDate) {
    SbString temp = str.getSubString(0,3) + SbString("-")
                  + str.getSubString(4,5) + SbString("-")
                  + str.getSubString(6,7);
    str = temp;
  }
  else if (isTime) {
    SbString temp = str.getSubString(0,1) + SbString(":")
                  + str.getSubString(2,3) + SbString(":")
                  + str.getSubString(4,5);
    str = temp;
  }
  addLine( title + str );
  return (getNumLines() - 1);
}

///////////////////////////////////////////////////////////////////////////////
int
DicomInfo::displayDicomInfo(SbString title, SbString infoString)
{
  checkFilename();
  addLine( title + infoString );
  return (getNumLines() - 1);
}

///////////////////////////////////////////////////////////////////////////////
SbString
DicomInfo::getDicomInfo(unsigned short group, unsigned short element)
{
  checkFilename();
  return m_dicomReader->getDicomInfo(group, element);
}