// Gnomon utility functions
//
// Notes:
// 1. Gnomon should be inserted in the scene graph after the main
//    camera, because it needs to get the current viewing matrix.
// 2. Gnomon should normally be inserted under an SoAnnotation node
//    so the gnomon will appear on top of the scene geometry.

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Medical/nodes/Gnomon.h>

// Open Inventor nodes we will need
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoDepthBuffer.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoResetTransform.h>

#include <Inventor/actions/SoGLRenderAction.h>

// Open Inventor traversal state elements we will need
#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/elements/SoViewingMatrixElement.h>
#include <Inventor/elements/SoComplexityElement.h>


#include <string>

SO_NODE_SOURCE(Gnomon)

////////////////////////////////////////////////////////////////////////
//
// Define the gnomon geometry (simple axes in this case)
//
// We could define these programmatically, but using a memory stream
// avoids including headers for nodes that we don't actually use in our 
// program. We could define these in a .iv file, but the memory stream
// avoids the danger of not having the file available at run time.
//
// For convenience we define the gnomon geometry in a -1..1 space
// because this is the default view volume for Open Inventor cameras.
//
// Note that, to be safe, we reset some attributes that might be 
// inherited from the scene graph and are not appropriate for our axes.
// But to be safe you should also keep the actual scene under its own
// Separator, so its attributes won't affect the gnomon.

 //
 // Could be definied through a field so customer can customize the look of tools ?
 //
 static std::string gnomonGeometry("\
#Inventor V2.1 ascii\n\
 DEF GnomonGeom Separator { \
  PickStyle { style UNPICKABLE } \
  RotationXYZ { axis X angle 1.570796327 } \
  Cube { \
    width 1 \
    height 1 \
    depth 1 \
  } \
  Font { \
    name arial\
  }\
  Separator { \
    Transform { \
    translation 0 -0.4 0.6 \
    rotation 0 0 1  0 \
    scaleFactor 0.1 0.1 0.1 \
    scaleOrientation 0 0 1  0 \
   } \
    Material { \
     diffuseColor 0 0 1 \
  } \
  Text3 { \
     parts ALL \
     string A \
     justification CENTER \
     } \
  }\
  Separator { \
    Transform { \
    translation -0.6 -0.4 0 \
    rotation 0 -1 0  1.55 \
    scaleFactor 0.1 0.1 0.1 \
    scaleOrientation 0 1 0  0 \
   } \
    Material { \
     diffuseColor 0 1 0 \
  } \
  Text3 { \
     string R \
     parts ALL \
     justification CENTER \
     } \
  }\
  Separator { \
   Transform { \
    translation 0.6 -0.4 0 \
    rotation 0 1 0 1.55 \
    scaleFactor 0.1 0.1 0.1 \
    scaleOrientation 0 1 0  0 \
  } \
  Material { \
     diffuseColor 1 0 0 \
  } \
  Text3 { \
     justification CENTER \
     parts ALL \
     string L \
     } \
  } \
  Separator { \
   Transform { \
    translation 0 -0.4 -0.6 \
    rotation 0 1 0  3.14 \
    scaleFactor 0.1 0.1 0.1 \
    scaleOrientation 0 0 1  0 \
   } \
  Material { \
     diffuseColor 0 0 1 \
  } \
  Text3 { \
     string P \
     justification CENTER \
     parts ALL \
     } \
  } \
  Separator { \
   Transform { \
    translation 0 0.6 0.35 \
    rotation -1 0 0  1.55 \
    scaleFactor 0.1 0.1 0.1 \
    scaleOrientation 0 0 1  0 \
   } \
  Material { \
     diffuseColor 1 1 1 \
  } \
  Text3 { \
     justification CENTER \
     parts ALL \
     string H \
     } \
  } \
  Separator { \
   Transform { \
    translation 0 -0.6 -0.35 \
    rotation 1 0 0  1.55 \
    scaleFactor 0.1 0.1 0.1 \
    scaleOrientation 0 0 1  0 \
   } \
  Material { \
     diffuseColor 0.5 0.5 0.5 \
  } \
  Text3 { \
     string F \
     justification CENTER \
     parts ALL \
     } \
  } \
} ");

///////////////////////////////////////////////////////////////////////////////
//
// Gnomon callback function
//
// This function will be called (from our Callback node) just before
// the gnomon camera is traversed.  It does some drawing setup and
// modifies the gnomon camera, as follows:
//
// 1) Modify OpenGL viewport to position the gnomon in the window.
// 2) Clear the OpenGL depth buffer in the region where the gnomon
//    will be drawn (to ensure gnomon is "on top").
// 3) Modify the gnomon camera to:
//    a) Have the same orientation as the 3D scene camera.
//    b) View the gnomon geometry at a consistent size.
//
static void gnomonRenderCB( void *userData, SoAction *action )
{
  Gnomon* renderedGnomon = (Gnomon*)userData;

  // Don't do anything if this is not a render traversal
  if (action->isOfType(SoGLRenderAction::getClassTypeId())) {

    // Suppress auto-caching above the callback node to ensure
    // that this code will actually be traversed on every render.
    //
    // The gnomon geometry itself is under a Separator, which
    // should be cached, so there is no performance penalty.
    SoState *state = action->getState();
  
    // Ensure that the gnomon is not impacted by any So(Interactive)Complexity
    // nodes in the scenegraph before.
    SoComplexityElement::set(state, SoComplexityElement::getDefault());

    // Set the OpenGL viewport to position gnomon in the window.
    //
    // Note that we modify the Open Inventor traversal state (rather
    // than calling OpenGL directly) so that any nodes that depend on
    // knowing the viewport will function correctly.
    //
    // The default gnomon position is in the lower-left corner.
    //
    // The gnomon position is _relative_ to its parent viewport.
    // This is critical so that gnomon will be correctly positioned when
    // using multiple viewports.
    const SbViewportRegion& parentVport = SoViewportRegionElement::get( state );
    const SbVec2i32&        parentOrig  = parentVport.getViewportOriginPixels_i32();

    const float ppp = parentVport.getPixelsPerPoint();

    SbViewportRegion vport;
    SbVec2i32 vportOrig = parentOrig + renderedGnomon->position.getValue()*ppp;
    SbVec2i32 vportSize((int32_t)(renderedGnomon->width.getValue()*ppp), (int32_t)(renderedGnomon->height.getValue()*ppp));
    vport.setViewportPixels( vportOrig, vportSize );
    SoViewportRegionElement::set( state, vport );
    
    // Get the current camera rotation from the viewing matrix.
    //
    // Note that the viewing matrix is applied to geometry, so it's
    // actually the inverse of the matrix we need.  The getTransform
    // method will return the rotation in variable cameraRotation.
    //
    // We could get the camera orientation by hiding a pointer to the 
    // camera, but this approach is very "fragile".  For example, if the 
    // user changes the camera type by clicking the viewer button, the
    // viewer destroys the current camera and creates a new one.
    SbMatrix viewMat = SoViewingMatrixElement::get( state );
    SbVec3f tran, scale;
    SbRotation cameraRotation, orient;
    viewMat.inverse().getTransform( tran, cameraRotation, scale, orient );

    // Disable notification
    renderedGnomon->getCamera()->enableNotify(FALSE);

    // Set the new orientation for the gnomon camera
    renderedGnomon->getCamera()->orientation = cameraRotation;

    // Get the current "focal distance" (distance to pt-of-rotation)
    float distance = renderedGnomon->getCamera()->focalDistance.getValue();

    // Reposition the camera so it's looking at the pt-of-rotation
    // (which is the center of the gnomon or 0,0,0 in this case).
    //
    // We'll use an algorithm similar to what the examiner viewer uses
    // to spin the camera around the pt-of-rotation.
    // First get the rotation as a matrix.  Conveniently SbMatrix has an
    // assignment operator that takes an SbRotation value.
    // Next extract the view direction vector from the rotation matrix.
    // Now move the camera radius units along the direction vector.
    SbMatrix mx;
    mx = cameraRotation;
    SbVec3f direction( -mx[2][0], -mx[2][1], -mx[2][2] );
    renderedGnomon->getCamera()->position = SbVec3f(0,0,0) - distance * direction;

    // re-enable notification
    renderedGnomon->getCamera()->enableNotify(FALSE);
  }
}

///////////////////////////////////////////////////////////////////////////////
void
Gnomon::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

   // Initialize type id variables
   SO__NODE_INIT_CLASS(Gnomon, "Gnomon", SoAnnotation);
}

///////////////////////////////////////////////////////////////////////////////
void
Gnomon::exitClass()
{
   SO__NODE_EXIT_CLASS(Gnomon);
}

///////////////////////////////////////////////////////////////////////////////
Gnomon::Gnomon()
{
  // Init public fields
  SO_NODE_CONSTRUCTOR(Gnomon);
  SO_NODE_ADD_FIELD(isDisplayed, (TRUE));
  SO_NODE_ADD_FIELD(position, (0,0));
  SO_NODE_ADD_FIELD(width   , (100));
  SO_NODE_ADD_FIELD(height  , (100));
  SO_NODE_ADD_FIELD(cameraDistance, (3.5));

  // Hide inherited fields from IvTune.
  // It's not necessary to do this, but the SoSeparator fields are not
  // relevant to our "shape" node, so this makes it a little bit easier
  // to observe and modify in IvTune.
#if SO_INVENTOR_VERSION >= 9100
  boundingBoxCaching.setFieldType( SoField::PRIVATE_FIELD );
  renderCulling.setFieldType( SoField::PRIVATE_FIELD );
  pickCulling.setFieldType( SoField::PRIVATE_FIELD );
  fastEditing.setFieldType( SoField::PRIVATE_FIELD );
  renderUnitId.setFieldType( SoField::PRIVATE_FIELD );
#endif

  commonConstructor();
}/***************************************************************************/

///////////////////////////////////////////////////////////////////////////////
Gnomon::~Gnomon()
{
}

///////////////////////////////////////////////////////////////////////////////
void
Gnomon::commonConstructor()
{
  // Name main node
  setName( "Gnomon" );

  m_displaySwitch = new SoSwitch();
  m_displaySwitch->whichChild = -1; // Off until we have valid geometry

  // define our callback
  m_callback  = new SoCallback();
  m_callback->setCallback( gnomonRenderCB, (void*)this );

  // Setup gnomon camera (viewAll might not work here because we want
  // the point of rotation to be the origin of the axes (i.e. 0,0,0),
  // not the geometric center of the gnomon geometry.
  m_camera    = new SoOrthographicCamera();
  m_camera->position = SbVec3f( 0,0, cameraDistance.getValue());
  m_camera->nearDistance  = 1;
  m_camera->farDistance   = 6;
  m_camera->focalDistance = cameraDistance.getValue();

  // Holder for our geometry
  m_geometry = new SoSeparator();

  buildGnomon();
}

////////////////////////////////////////////////////////////////////////
//
// Create the gnomon scene graph
//
void
Gnomon::buildGnomon()
{
  // Reset the bounding box so gnomon geometry will not
  // affect bbox of scene (ie. will not affect viewAll).
  SoBBox* bboxNode = new SoBBox();
    bboxNode->mode = SoBBox::NO_BOUNDING_BOX;
    this->addChild( bboxNode );
  
  // Switch controls gnomon visibility
  this->addChild( m_displaySwitch.ptr() );

  // Clear the depth buffer to ensure gnomon is on top of the scene.
  // NOTE!
  // This is somewhat dangerous because it could affect rendering of
  // other parts of the scene graph. You can reduce the danger, by
  // making the gnomon the last thing in the scene graph.  But objects
  // that are semi-transparent may still be rendered after the gnomon
  // when the transparency mode is "_DELAYED".
  SoDepthBuffer* pDepthBuffer = new SoDepthBuffer;
  pDepthBuffer->clearBuffer = TRUE;
  m_displaySwitch->addChild(pDepthBuffer);

  // Under the switch is our callback, camera and geometry
  m_displaySwitch->addChild( m_callback.ptr() );
  m_displaySwitch->addChild( m_camera.ptr() );
  m_gnomonScale = new SoScale();
  m_displaySwitch->addChild( m_gnomonScale.ptr() );
  m_displaySwitch->addChild( m_geometry.ptr() );

  // Read the gnomon geometry from a memory buffer
  SoInput in;
  in.setBuffer((void *)gnomonGeometry.c_str(), gnomonGeometry.size());
  SoNode *node;
  SbBool ok = SoDB::read(&in, node);
  if (ok && node != NULL)
  {
    m_geometry->addChild(node);
    m_displaySwitch->whichChild = SO_SWITCH_ALL;
  }
}

///////////////////////////////////////////////////////////////////////////////
void
Gnomon::notify(SoNotList *list)
{
  if (list->getLastRec()->getType() == SoNotRec::CONTAINER)
  {
    SoField* lastField = list->getLastField();
    if (lastField == &isDisplayed)
    {
      updateGnomonVisibility();
    }
    else if (lastField == &width || lastField == &height || lastField == &position)
    {
      // Inform that this node will be impacted by the change of value for these fields
      m_callback->touch();
    }
  }
  // Forward other events to upper layer
  SoAnnotation::notify(list);
}

///////////////////////////////////////////////////////////////////////////////
void
Gnomon::updateGnomonVisibility()
{
  if ( isDisplayed.getValue() )
    m_displaySwitch->whichChild = SO_SWITCH_ALL;
  else
    m_displaySwitch->whichChild = SO_SWITCH_NONE;
}

///////////////////////////////////////////////////////////////////////////////
SoOrthographicCamera* 
Gnomon::getCamera()
{
  return m_camera.ptr();
}

///////////////////////////////////////////////////////////////////////////////
/** Replace the gnomon geometry (default is basic medical gnomon).
   *  Gnomon geometry should have a 3D extent of -1 to 1.
   */
void
Gnomon::setGeometry( SoNode* geometrySceneGraph )
{
  if (geometrySceneGraph != NULL) {
    m_geometry->removeAllChildren();
    m_geometry->addChild( geometrySceneGraph );
  }
}
