///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/draggers/SoTranslate2Dragger.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoRenderToTextureProperty.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoLightModel.h>

#include <Inventor/sensors/SoFieldSensor.h>

#include <Medical/nodes/Magnifier.h>

SO_NODE_SOURCE(Magnifier)

///////////////////////////////////////////////////////////////////////////////
void
Magnifier::initClass()
{
    getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

    // Initialize type id variables
    SO__NODE_INIT_CLASS(Magnifier, "Magnifier", SoTranslate2Dragger);
}

///////////////////////////////////////////////////////////////////////////////
void
Magnifier::exitClass()
{
    SO__NODE_EXIT_CLASS(Magnifier);
}

///////////////////////////////////////////////////////////////////////////////
Magnifier::Magnifier( )
{
    m_magnifierBorderMaterial = NULL;
    m_magnifierCam = NULL;

    // Init public fields
    SO_NODE_CONSTRUCTOR( Magnifier );
    SO_NODE_ADD_FIELD(sceneToMagnify, (NULL));
    SO_NODE_ADD_FIELD(magnifierColor, (SbColor(0, 0.6f,0)));
    SO_NODE_ADD_FIELD(magnifierFactor, (50.0f));

    commonConstructor();
}/***************************************************************************/

///////////////////////////////////////////////////////////////////////////////
Magnifier::~Magnifier()
{

}/***************************************************************************/

///////////////////////////////////////////////////////////////////////////////
void
Magnifier::commonConstructor()
{
    // Name main node
    setName( "Magnifier" );

    // --------------- Sub-scenegraph used as dragger geometry ----------------
    SoSeparator* magnifierSep = new SoSeparator();

    SoLightModel* lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    magnifierSep->addChild( lightModel );

    m_magnifierBorderMaterial = new SoMaterial();
      magnifierSep->addChild( m_magnifierBorderMaterial.ptr() );

    SoComplexity *cylinderComplexity = new SoComplexity();
      cylinderComplexity->value = 1.0f;
      magnifierSep->addChild(cylinderComplexity);

    // Rotate cylinder(s) to align with Z axis
    SoRotationXYZ *cylinderRotation = new SoRotationXYZ();
      cylinderRotation->angle = (float)(M_PI / 2);
      magnifierSep->addChild(cylinderRotation);

    // Border for the magnifier
    SoCylinder *magnifierBorderSupport = new SoCylinder();
      magnifierBorderSupport->height = 0;
      magnifierBorderSupport->radius = 40.5;
      magnifierBorderSupport->parts = SoCylinder::TOP;
      magnifierSep->addChild( magnifierBorderSupport );

    // Texture mapped on the cylinder issue from magnifierRenderToTexture.
    SoTexture2 *magnifierTexture = new SoTexture2();
      magnifierTexture->model = SoTexture2::REPLACE;
      magnifierSep->addChild(magnifierTexture);

    // The center of this cylinder (circle) is the center of the magnifier camera (magnifierCam).
    SoCylinder *magnifier = new SoCylinder();
      magnifier->height = 1;
      magnifier->radius = 40;
      magnifier->parts = SoCylinder::TOP;
      magnifierSep->addChild( magnifier );

    // Replace dragger geometry(s) with our custom shapes
    this->setPart("translatorActive", magnifierSep);
    this->setPart("translator", magnifierSep);
    this->setPart("xAxisFeedback", new SoSeparator() );
    this->setPart("yAxisFeedback", new SoSeparator() );

    // ---------------- Sub-scenegraph used by "magnifierRenderToTexture" ---------------
    // This scenegraph is rendered into a texture when dragger position changes.
    m_magnifierSGSep = new SoSeparator();
      m_magnifierSGSep->addChild(new SoDirectionalLight());

    // Camera
    m_magnifierCam = new SoOrthographicCamera();
      m_magnifierCam->position.connectFrom( &this->translation );
      m_magnifierCam->viewAll( m_magnifierSGSep.ptr(), SbViewportRegion(100, 100) );
      m_magnifierCam->height = magnifierFactor;
      //m_magnifierCam->farDistance = 2000;
      m_magnifierSGSep->addChild( m_magnifierCam.ptr() );

    // We will replace the children of this node with the scene to magnify.
    m_sceneToMagnifyParent = new SoSeparator();
      SoNode* scene = sceneToMagnify.getValue();
      if (scene != NULL)
        m_sceneToMagnifyParent->addChild( scene );
      m_magnifierSGSep->addChild( m_sceneToMagnifyParent.ptr() );

    // This node will render the scene into a texture.
    // Attach it to the texture node above (applied to the SoCylinder).
    SoRenderToTextureProperty *magnifierRenderToTexture = new SoRenderToTextureProperty();
      magnifierRenderToTexture->node = m_magnifierSGSep.ptr();
      magnifierRenderToTexture->size.setValue(512,512);
      magnifierTexture->renderToTextureProperty = magnifierRenderToTexture;

    // ---------- Manage field modification ----------
    // These internal fields are updated automatically when the external field is changed.
    m_magnifierBorderMaterial->diffuseColor.connectFrom( &magnifierColor );
    m_magnifierCam->height.connectFrom( &magnifierFactor );

    // We need to do a little more work when the 'sceneToMagnify' field changes.
    m_sceneFieldSensor = new SoFieldSensor( sceneFieldChangedCB, (void*)this );
      m_sceneFieldSensor->setPriority( 0 );
      m_sceneFieldSensor->attach( &sceneToMagnify );
}

//////////////////////////////////////////////////////////////////////////////
// Manage field modification
void
Magnifier::sceneFieldChangedCB( void* data, SoSensor* /*sensor*/ )
{
  // Replace current sceneToMagnify with the new one.
  Magnifier* self = (Magnifier*)data;
  bool isChanged = false;

  SoNode* newScene = self->sceneToMagnify.getValue();
  int curNumKids = self->m_sceneToMagnifyParent->getNumChildren();
  if (curNumKids == 0) {
    self->m_sceneToMagnifyParent->addChild( newScene );
    isChanged = true;
  }
  else {
    // Don't change our scene graph unless it's a real change.
    SoNode* curScene = self->m_sceneToMagnifyParent->getChild( 0 );
    if (newScene != curScene) {
      self->m_sceneToMagnifyParent->removeAllChildren();
      self->m_sceneToMagnifyParent->addChild( newScene );
      isChanged = true;
    }
  }

  // Update internal camera to view sceneToMagnify.
  // Don't forget to reset camera position and height after viewAll changes them...
  if (isChanged) {
    self->m_magnifierCam->viewAll( self->m_magnifierSGSep.ptr(), SbViewportRegion(500, 500));
    self->m_magnifierCam->position = self->translation.getValue();
    self->m_magnifierCam->height   = self->magnifierFactor.getValue();
  }
}
