/////////////////////////////////////////////////////////////////////
//
// OrthoSliceBorder
//
/////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/actions/SoGLRenderAction.h>

#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoPolygonOffset.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/elements/SoDrawStyleElement.h>
#include <VolumeViz/elements/SoVolumeDataElement.h>

#include <Medical/nodes/OrthoSliceBorder.h>

SO_NODE_SOURCE(OrthoSliceBorder);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
OrthoSliceBorder::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(OrthoSliceBorder, SoOrthoSlice, "OrthoSlice");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
OrthoSliceBorder::exitClass()
{
  SO__NODE_EXIT_CLASS(OrthoSliceBorder);
}

////////////////////////////////////////////////////////////////////////
// Constructor
OrthoSliceBorder::OrthoSliceBorder()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(OrthoSliceBorder);

  // Initially we don't know the volume extent or slice position.
  m_volumeExtent.makeEmpty();
  m_sliceAxis   = -1;
  m_sliceNumber = -1;

  // Build the internal scene graph.
  m_scene = new SoSeparator();
 
  // Pull the border slightly toward the camera to avoid Z-buffer fighting.
  // You might think that we could apply PolygonOffset to the OrthoSlice
  // geometry, but we'd have to use override because the OrthoSlice node has
  // it's own internal PolygonOffset node.
  SoPolygonOffset* offset = new SoPolygonOffset();
    offset->units  = -1;
	offset->factor = -1;
    m_scene->addChild( offset );

  // Disable lighting for the slice border (easier to see).
  SoLightModel* lighting = new SoLightModel();
    lighting->model = SoLightModel::BASE_COLOR;
    m_scene->addChild( lighting );

  // We use a polygon to draw the border.
  // This allows us apply polygonOffset to it, but we need to set the drawStyle
  // so it looks like a polyline. The line width comes from our field. We have
  // to call ignore() on the other fields so the application can modify them.
  m_style = new SoDrawStyle();
	m_style->style     = SoDrawStyle::LINES;
	m_style->lineWidth = borderWidth.getValue();
	m_style->linePattern.setIgnored( TRUE );
	m_style->linePatternScaleFactor.setIgnored( TRUE );
    m_scene->addChild(m_style);

  // Start with no vertices but default color set.
  SoVertexProperty* vprop = new SoVertexProperty();
    vprop->orderedRGBA = borderColor.getValue().getPackedValue();

  // The geometry
  SoFaceSet* faceSet = new SoFaceSet();
    faceSet->vertexProperty = vprop;
    m_scene->addChild( faceSet );

  // Remember the geometry node so we can update the vertices.
  m_geometry = faceSet;
}

////////////////////////////////////////////////////////////////////////
// Destructor
OrthoSliceBorder::~OrthoSliceBorder()
{
}

////////////////////////////////////////////////////////////////////////
void
OrthoSliceBorder::GLRender(SoGLRenderAction* action)
{
  // Get current volume data in the traversal state.
  // This should also create a dependency on the volumeDataElement so this
  // method is called again if there is any change to the volume data node.
  SoVolumeData* volume;
  SoVolumeDataElement::get( action->getState(), volume );

  // If no volume data, nothing to do...
  if (volume == NULL)
    return;

  // If border should be rendered
  if (this->enableBorder.getValue() == TRUE) {

    // No one needs to know that we are modifying internal fields...
    this->enableNotify(FALSE);

    // Get volume characteristics
    const SbBox3f&   volExt = volume->extent.getValue();
    float xmin, ymin, zmin, xmax, ymax, zmax;
    volExt.getBounds(xmin, ymin, zmin, xmax, ymax, zmax);

    // Get slice axis and position
    int axisIndex = axis.getValue(); // 0: X, 1: Y, 2: Z
    int sliceNum = sliceNumber.getValue();

    // Make sure width is current (OIV will ignore redundant setting).
    m_style->lineWidth = borderWidth.getValue();

    // Get geometry in case we need to change geometry or color
    SoFaceSet* faceSet = (SoFaceSet*)m_geometry.ptr();
    SoVertexProperty* vprop = (SoVertexProperty*)faceSet->vertexProperty.getValue();

    // If parameters have changed
    if (volExt != m_volumeExtent ||
      axisIndex != m_sliceAxis ||
      sliceNum != m_sliceNumber) {

      // Remember current parameters
      m_volumeExtent = volExt;
      m_sliceAxis = axisIndex;
      m_sliceNumber = sliceNum;

      // Do not draw border exactly in the middle of the voxel
      // to avoid Z fighting with the slice itself.
      const float slicePos = float(sliceNum) + 0.501f;

      // abitrary margin to avoid z-fighting
      float margin = std::min( std::min( xmax - xmin, ymax - ymin ), zmax - zmin ) * 0.0025f;
      xmin -= margin;
      ymin -= margin;
      zmin -= margin;
      xmax += margin;
      ymax += margin;
      zmax += margin;

      // Update vertices of geometry
      const SbVec3i32& volDim = volume->data.getSize();
      vprop->vertex.setNum(5);
      SbVec3f* verts = vprop->vertex.startEditing();
      if (axisIndex == 0) { // X axis slice
        float voxelSize = (xmax - xmin) / (float)volDim[0];
        float x = xmin + slicePos * voxelSize;
        verts[0].setValue(x, ymin, zmin);
        verts[1].setValue(x, ymax, zmin);
        verts[2].setValue(x, ymax, zmax);
        verts[3].setValue(x, ymin, zmax);
        verts[4].setValue(x, ymin, zmin);
      }
      else if (axisIndex == 1) { // Y axis slice
        float voxelSize = (ymax - ymin) / (float)volDim[1];
        float y = ymin + slicePos * voxelSize;
        verts[0].setValue(xmin, y, zmin);
        verts[1].setValue(xmax, y, zmin);
        verts[2].setValue(xmax, y, zmax);
        verts[3].setValue(xmin, y, zmax);
        verts[4].setValue(xmin, y, zmin);
      }
      else {                     // Z axis slice
        float voxelSize = (zmax - zmin) / (float)volDim[2];
        float z = zmin + slicePos * voxelSize;
        verts[0].setValue(xmin, ymin, z);
        verts[1].setValue(xmax, ymin, z);
        verts[2].setValue(xmax, ymax, z);
        verts[3].setValue(xmin, ymax, z);
        verts[4].setValue(xmin, ymin, z);
      }
      vprop->vertex.finishEditing();
    }

    // Make sure color is current
    unsigned int color = this->borderColor.getValue().getPackedValue();
    vprop->orderedRGBA = color; // OIV will ignore redundant setting

    // Restore default setting
    this->enableNotify(TRUE);

    // Render the border (traverse the internal scene graph)
    action->forwardTraversal(m_scene.ptr());
  }

  // Do parent rendering.
  // Note slice must be rendered *after* border so border is not clipped.
  if (this->enableImage.getValue() == TRUE) {
    // Slice is visible, just render it.
    SoOrthoSlice::GLRender(action);
  }
  else if (this->clipping.getValue() == TRUE) {
    // Slice is not visible, but we must traverse it for the clipping effect.
    SoState* state = action->getState();
    SoDrawStyleElement::Style style = SoDrawStyleElement::get(state);
    SoDrawStyleElement::set(state, SoDrawStyleElement::Style::INVISIBLE);
    SoOrthoSlice::GLRender(action);
    SoDrawStyleElement::set(state, style);
  }
}
