///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
//
// PlaneBoxIntersection
//
// Notes:
// - Polygon algorithm based on Adam Sawicki's article
//   http://www.asawicki.info/news_1428_finding_polygon_of_plane-aabb_intersection.html
//
/////////////////////////////////////////////////////////////////////

#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/sensors/SoFieldSensor.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/algorithm>

#include <Medical/nodes/PlaneBoxIntersection.h>

SO_NODE_SOURCE(PlaneBoxIntersection);

////////////////////////////////////////////////////////////////////////
// Don't expose std::vector outside the class
class PlaneBoxIntersectionImpl
{
public:
  // Intersection points
  std::vector<SbVec3f> m_intersection_points;
};

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
PlaneBoxIntersection::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(PlaneBoxIntersection, SoLineSet, "LineSet");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
PlaneBoxIntersection::exitClass()
{
  SO__NODE_EXIT_CLASS(PlaneBoxIntersection);
}

////////////////////////////////////////////////////////////////////////
// Constructor
PlaneBoxIntersection::PlaneBoxIntersection()
{
  // Setup fields
  SO_NODE_CONSTRUCTOR(PlaneBoxIntersection);
  SO_NODE_ADD_FIELD(plane, (SbPlane(SbVec3f(1.0, 0.0, 0.0), 0.0)));
  SO_NODE_ADD_FIELD(box,   (SbBox3f(0,0,0,0,0,0)));

  m_impl = new PlaneBoxIntersectionImpl();

  // Hide inherited fields from IvTune
  startIndex.setFieldType ( SoField::PRIVATE_FIELD );
  numVertices.setFieldType( SoField::PRIVATE_FIELD );

  // Update geometry when either field changes
  m_boxFieldSensor = new SoFieldSensor( &PlaneBoxIntersection::fieldSensorCB, this);
  m_boxFieldSensor->attach( &box );

  m_planeFieldSensor = new SoFieldSensor( &PlaneBoxIntersection::fieldSensorCB, this);
  m_planeFieldSensor->attach( &plane );

  // Start with an empty property (no line)
  vertexProperty = new SoVertexProperty();
}

////////////////////////////////////////////////////////////////////////
// Destructor
PlaneBoxIntersection::~PlaneBoxIntersection()
{
  delete m_impl;
  delete m_boxFieldSensor;
  delete m_planeFieldSensor;
}

///////////////////////////////////////////////////////////////////////////////
// Compute intersection of ray and plane
//
// Returns false if there is no intersection because ray is parallel to plane.
// Else OutT is the parametric location of the intersection on the ray.
//
// Use a simple struct so we don't have to unpack the SbPlane all the time
struct Plane { float a, b, c, d; };

static bool 
ray_to_plane(const SbVec3f &RayOrig, const SbVec3f &RayDir, const Plane &plane, 
             float *OutT, float *OutVD)
{
    *OutVD = plane.a * RayDir[0] + plane.b * RayDir[1] + plane.c * RayDir[2];
    if (*OutVD == 0.0f)
        return false;
    *OutT = - (plane.a * RayOrig[0] + plane.b * RayOrig[1] + plane.c * RayOrig[2] + plane.d) / *OutVD;
    return true;
}

///////////////////////////////////////////////////////////////////////////////
// Add a point to the list, but only if it's not already there.
//
// This avoids a boundary condition bug. When one of the edges of the box lies
// exactly in the plane we would record the corner points twice. For example
// on the X edges and again on the Y edges. Because of the way our sort works
// this could result in two triangles instead of a polygon.  mmh March 2014.
static void
addUniquePoint( std::vector<SbVec3f>& points, SbVec3f point )
{
  if (points.size() > 0) {
    std::vector<SbVec3f>::iterator it;
    for (it = points.begin(); it != points.end(); ++it) {
      if (*it == point)
        return; // Ignore duplicate points
    }
  }
  points.push_back( point );
}

///////////////////////////////////////////////////////////////////////////////
// Find all intersections between the plane and edges of the box
//
// Returns the number of intersections found.
// NOTE: out_points are not sorted.
static int 
calc_plane_box_intersection_points( const SbPlane& _plane, 
                                    const SbBox3f& _box,
                                    std::vector<SbVec3f>& out_points )
{
    float vd, t;
    SbVec3f aabb_max = _box.getMax();
    SbVec3f aabb_min = _box.getMin();
    SbVec3f normal   = _plane.getNormal();
    float   dist     = _plane.getDistanceFromOrigin();
    Plane   plane;
    plane.a = normal[0];
    plane.b = normal[1];
    plane.c = normal[2];
    plane.d = -dist;

    // Test edges along X axis, pointing right.
    SbVec3f dir(aabb_max[0] - aabb_min[0], 0.f, 0.f);
    SbVec3f orig = aabb_min;
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_min[0], aabb_max[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_min[0], aabb_min[1], aabb_max[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_min[0], aabb_max[1], aabb_max[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );

    // Test edges along Y axis, pointing up.
    dir.setValue(0.f, aabb_max[1] - aabb_min[1], 0.f);
    orig.setValue(aabb_min[0], aabb_min[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_max[0], aabb_min[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_min[0], aabb_min[1], aabb_max[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_max[0], aabb_min[1], aabb_max[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );

    // Test edges along Z axis, pointing forward.
    dir.setValue(0.f, 0.f, aabb_max[2] - aabb_min[2]);
    orig.setValue(aabb_min[0], aabb_min[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_max[0], aabb_min[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_min[0], aabb_max[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );
    orig.setValue(aabb_max[0], aabb_max[1], aabb_min[2]);
    if (ray_to_plane(orig, dir, plane, &t, &vd) && t >= 0.f && t <= 1.f)
        addUniquePoint( out_points, (orig + dir * t) );

    return (int)out_points.size();
}

///////////////////////////////////////////////////////////////////////////////
// Sort intersection points by relationship to first point in list
struct MyComparator {
  bool operator() ( const SbVec3f& lhs, const SbVec3f& rhs) {
    SbVec3f v = (lhs - origin).cross(rhs - origin);
    bool result = (v.dot(plane_normal) < 0);
    return result;
  }
  SbVec3f plane_normal;
  SbVec3f origin;
};

///////////////////////////////////////////////////////////////////////////////
// Sort intersection points to make the correct polyline
static void 
sort_intersection_points( std::vector<SbVec3f>& points, const SbPlane& plane )
{
  if (points.size() == 0) return;

  MyComparator comparator;
  comparator.plane_normal = plane.getNormal();
  comparator.origin       = points[0];

  std::sort( points.begin(), points.end(), comparator );
}

////////////////////////////////////////////////////////////////////////
// If a field changed, update the geometry
void
PlaneBoxIntersection::fieldSensorCB(void *inData, SoSensor* /*inSensor*/)
{
  PlaneBoxIntersection* self = (PlaneBoxIntersection *) inData;

  SbPlane plane   = self->plane.getValue();
  SbBox3f bbox    = self->box.getValue();
  std::vector<SbVec3f>& thePts = self->m_impl->m_intersection_points;
  SoVertexProperty* pVertProp  = (SoVertexProperty*)self->vertexProperty.getValue();

  // Compute intersections of the plane and the edges of the box
  thePts.clear();
  int numPts = calc_plane_box_intersection_points( plane, bbox, thePts );
  if (numPts > 0) {
    // Must sort intersection points to make a nice polyline
    sort_intersection_points( thePts, plane );
    // Must repeat first point to get a closed loop
    thePts.push_back( thePts[0] );
    // Vertex property can use the points directly from our memory
    pVertProp->vertex.setValuesPointer( (int)thePts.size(), &(thePts[0]) );
  }
  else
    pVertProp->vertex.deleteValues( 0, -1 ); // Delete all
}
