////////////////////////////////////////////////////////////////////////
//
// SliceScaleBar utility class
//
// Original: Mike Heck, VSG Inc, December 2011
// Modified:                     March 2016
//
// This class is not intended to be a general purpose axis class.
// For a basic slice scale bar we probably don't need a lot of options.
// But it could be interesting to think about:
//   - Option to draw tick marks on either or both sides of the axis.
//   - Option for relative position of axis label.
//   - Automatically compute tick interval?
//   - Text labels for tick marks?
//
////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTextProperty.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>

#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/elements/SoViewVolumeElement.h>

#include <Inventor/sensors/SoNodeSensor.h>

#include <Medical/InventorMedical.h>
#include <Medical/nodes/SliceScaleBar.h>

SO_NODE_SOURCE(SliceScaleBar);

////////////////////////////////////////////////////////////////////////
// Initialize the class.
void
SliceScaleBar::initClass()
{
  getClassRenderEngineMode().setRenderMode( SbRenderEngineMode::OIV_OPENINVENTOR_RENDERING );

  // Initialize type id variables
  SO_NODE_INIT_CLASS(SliceScaleBar, SoAnnotation, "Annotation");
}

////////////////////////////////////////////////////////////////////////
// Cleanup type id
void
SliceScaleBar::exitClass()
{
  SO__NODE_EXIT_CLASS(SliceScaleBar);
}

////////////////////////////////////////////////////////////////////////
SliceScaleBar::SliceScaleBar()
{
  SO_NODE_CONSTRUCTOR(SliceScaleBar);
  SO_NODE_ADD_FIELD(position        , (0,0));
  SO_NODE_ADD_FIELD(length          , (1));
  SO_NODE_ADD_FIELD(numTickIntervals, (10));
  SO_NODE_ADD_FIELD(trackedCamera   , (NULL));
  SO_NODE_ADD_FIELD(orientation     , (SliceScaleBar::HORIZONTAL));
  SO_NODE_ADD_FIELD(alignment       , (SliceScaleBar::CENTER));
  SO_NODE_ADD_FIELD(label           , (""));

  // Set up static info for enumerated type fields
  SO_NODE_DEFINE_ENUM_VALUE( Alignment  , LEFT       );
  SO_NODE_DEFINE_ENUM_VALUE( Alignment  , BOTTOM     );
  SO_NODE_DEFINE_ENUM_VALUE( Alignment  , CENTER     );
  SO_NODE_DEFINE_ENUM_VALUE( Alignment  , RIGHT      );
  SO_NODE_DEFINE_ENUM_VALUE( Alignment  , TOP        );
  SO_NODE_DEFINE_ENUM_VALUE( Orientation, HORIZONTAL );
  SO_NODE_DEFINE_ENUM_VALUE( Orientation, VERTICAL   );

  // Associate info for enumerated type fields
  SO_NODE_SET_SF_ENUM_TYPE( alignment  , Alignment   );
  SO_NODE_SET_SF_ENUM_TYPE( orientation, Orientation );

  // Hide inherited fields from IvTune.
  // It's not necessary to do this, but the SoSeparator fields are not
  // relevant to our "shape" node, so this makes it a little bit easier
  // to observe and modify in IvTune.
#if SO_INVENTOR_VERSION >= 9100
  boundingBoxCaching.setFieldType( SoField::PRIVATE_FIELD );
  renderCulling.setFieldType( SoField::PRIVATE_FIELD );
  pickCulling.setFieldType( SoField::PRIVATE_FIELD );
  fastEditing.setFieldType( SoField::PRIVATE_FIELD );
  renderUnitId.setFieldType( SoField::PRIVATE_FIELD );
#endif

  // Set by node sensor if a field is changed, checked in renderCB
  m_fieldsChanged = false;

  // Note: m_ndcLength must be set before calling buildSceneGraph().
  m_ndcLength     = length.getValue();

  // Window size in pixels (approximate until first traversal)
  m_winSizePix.setValue(500,500);
  // Conversion from 1 ndc unit to pixels
  m_pixelPerNdc.setValue( 0.5f * (float)m_winSizePix[0], 0.5f * (float)m_winSizePix[1] );

  // Tick line length in pixels
  // Tick line length in NDC (depends on viewport so updated later)
  m_tickLenPix = 5;
  m_tickLenNdc = (float)m_tickLenPix / m_pixelPerNdc[0];

  // Create the internal scene graph
  buildSceneGraph();

  // Monitor changes to our fields
  m_sensor = new SoNodeSensor( sensorCB, (void*)this );
  m_sensor->attach( this );
  m_sensor->setPriority( 0 );
}

////////////////////////////////////////////////////////////////////////

SliceScaleBar::~SliceScaleBar()
{
  delete m_sensor;
  if ( m_labelText.ptr() != NULL )
    m_labelText->string.disconnect( &(this->label) );
}

///////////////////////////////////////////////////////////////////////
// Create axis scene graph
//
// Separator "AxisBoxRoot"
// +- EventCallback
// +- LightModel
// +- PickStyle
// +- DrawStyle
// +- Font
// +- Material
// +- Separator "AxisBoxLines"
// |  +- LineSet
// +- Separator "AxisBoxText"
//    +- SoSeparator (repeated for each string)
//       +- Translation
//       +- Text2
//
// Note: Do not put property nodes under the text separator.
//       Text is cleared by removing all children of this node.

void
SliceScaleBar::buildSceneGraph()
{
  // Event callback
  SoCallback* callbackNode = new SoCallback;
    callbackNode->setCallback( SliceScaleBar::staticCB, (void*)this );
    this->addChild( callbackNode );

  SoBBox* bbox = new SoBBox();
    bbox->mode = SoBBox::NO_BOUNDING_BOX;
    this->addChild( bbox );

  SoLightModel* lmodel = new SoLightModel();
    lmodel->model = SoLightModel::BASE_COLOR;
    this->addChild( lmodel );

  SoPickStyle* pstyle = new SoPickStyle();
    pstyle->style = SoPickStyle::UNPICKABLE;
    this->addChild( pstyle );

  SoOrthographicCamera* camera = new SoOrthographicCamera();
    camera->viewportMapping = SoCamera::LEAVE_ALONE;
    this->addChild( camera );

  // Lines -------------------------------------------------------------
  m_lineSep = new SoSeparator;
    this->addChild( m_lineSep.ptr() );

  // Initial geometry
  m_vertProp = new SoVertexProperty();
    computeEndPoints( m_p0, m_p1 );
    m_vertProp->vertex.set1Value( 0, m_p0 );
    m_vertProp->vertex.set1Value( 1, m_p1 );

  m_axisLineSet  = new SoLineSet();
    m_axisLineSet->vertexProperty = m_vertProp.ptr();
    m_lineSep->addChild( m_axisLineSet.ptr() );

  // Text -------------------------------------------------------------
  m_textSep = new SoSeparator;
    this->addChild( m_textSep.ptr() );

  m_labelPos = new SoTranslation();
    m_textSep->addChild( m_labelPos.ptr() );

  m_labelFont = new SoFont();
    m_labelFont->size = 15;
    m_labelFont->name = "Arial:Bold";
    m_labelFont->renderStyle = SoFont::TEXTURE;
    m_textSep->addChild( m_labelFont.ptr() );

  SoTextProperty* textProp = new SoTextProperty();
    if (orientation.getValue() == SliceScaleBar::HORIZONTAL) {
      textProp->alignmentH = SoTextProperty::LEFT;
      textProp->alignmentV = SoTextProperty::BASE;
    }
    else {
      textProp->alignmentH = SoTextProperty::LEFT;
      textProp->alignmentV = SoTextProperty::TOP;
    }
    m_textSep->addChild( textProp );

  m_labelText = new SoText2();
    m_labelText->justification = SoText2::INHERITED;
    m_labelText->string.connectFrom( &(this->label) ); // Text will auto-update
    m_textSep->addChild( m_labelText.ptr() );
}

//////////////////////////////////////////////////////////////////////////
// Called when one of our monitored fields has changed.
void
SliceScaleBar::sensorCB( void* data, SoSensor* sensor )
{
  SliceScaleBar* self = (SliceScaleBar*)data;
  const SoField* trigger = ((SoNodeSensor*)sensor)->getTriggerField();

  // If trigger field is not NULL...
  if (trigger != NULL) {
    // The change is associated with one of our fields.
    self->m_fieldsChanged = true;
  }
}

////////////////////////////////////////////////////////////////////////
// Static method called from SoCallback node.
//
// Note: It might not be necessary to handle boundingBoxAction since
//       our bounding box is canceled out by SoBBox. But... it's usually
//       the first action we see that has valid values for the viewport,
//       which we need to compute pixel size, so keep it for now.
void 
SliceScaleBar::staticCB( void* data, SoAction* action )
{
  SliceScaleBar* self = (SliceScaleBar*)data;
  if (action->isOfType(SoGLRenderAction::getClassTypeId()) ||
      action->isOfType(SoGetBoundingBoxAction::getClassTypeId())) {

    self->renderCB( action );
  }
}

/////////////////////////////////////////////////////////////////////////
// Called from staticCB on render action traversal
void
SliceScaleBar::renderCB( SoAction* action )
{
  bool mustUpdateAxis = false;

  // Get values from state we need
  // (creates a dependency on these elements in the state)
  SoState* state = action->getState();
  const SbViewportRegion& vpregion = SoViewportRegionElement::get( state );
  const SbViewVolume&     viewVol  = SoViewVolumeElement::get( state );

  const SbVec2i32& winSize = vpregion.getViewportSizePixels_i32();
  if (winSize != m_winSizePix) {
    mustUpdateAxis = true;
    m_winSizePix = winSize;
    m_pixelPerNdc.setValue( 0.5f * (float)m_winSizePix[0], 0.5f * (float)m_winSizePix[1] );

    // Tick line length in NDC.
    // This must be computed orthogonal to the axis orientation.
    int horizOrVert = 1 - this->orientation.getValue();
    m_tickLenNdc = (float)m_tickLenPix / m_pixelPerNdc[horizOrVert];
  }

  // If tracked camera or length changed, we need to recompute length in NDC.
  if (m_fieldsChanged) {
    mustUpdateAxis  = true;
    m_fieldsChanged = false;

    // First project the NDC mid-point of the scale bar back to 3D space.
    SbVec2f pos = position.getValue();
    SbVec3f ndcPt = m_p0 + (m_p1 - m_p0)/2;
    SbVec3f xyzPt;
    viewVol.projectFromScreen( ndcPt, xyzPt );

    // What direction is camera "horizontal" (or "vertical") in the current view?
    // With a little luck we don't need to know what the camera orientation "means".
    // We just need the xyz points +/- half the length from the projected point.
    SoCamera* camera = (SoCamera*)trackedCamera.getValue();
    float len = length.getValue(); // Tracked length
    SbVec3f plus, minus;
    SbRotation camOrient = camera->orientation.getValue();
    if (orientation.getValue() == SliceScaleBar::HORIZONTAL) {
      camOrient.multVec( SbVec3f( len/2,0,0), plus );
      camOrient.multVec( SbVec3f(-len/2,0,0), minus );
    }
    else { // VERTICAL
      camOrient.multVec( SbVec3f( 0, len/2,0), plus );
      camOrient.multVec( SbVec3f( 0,-len/2,0), minus );
    }
    SbVec3f p0xyz = xyzPt + minus;
    SbVec3f p1xyz = xyzPt + plus;

    // Project these points back to NDC space.
    // But note that projectToScreen outputs 0..1 space and we need -1..1 space.
    SbVec3f p0ndc, p1ndc;
    viewVol.projectToScreen( p0xyz, p0ndc );
    viewVol.projectToScreen( p1xyz, p1ndc );
    p0ndc[0] = 2 * p0ndc[0] - 1;
    p0ndc[1] = 2 * p0ndc[1] - 1;
    p1ndc[0] = 2 * p1ndc[0] - 1;
    p1ndc[1] = 2 * p1ndc[1] - 1;

    // Compute the new NDC length and corresponding end-points.
    m_ndcLength = (p1ndc - p0ndc).length();
    computeEndPoints( m_p0, m_p1 );
  }

  if (mustUpdateAxis) {
    updateAxis();
  }
}

///////////////////////////////////////////////////////////////////////
// Compute new end-points of scale bar based on orientation and
// alignment.  Must use the member variable m_ndcLength, not the
// length field, in case actual length is tracking a 3D distance.
void
SliceScaleBar::computeEndPoints( SbVec3f& p0, SbVec3f& p1 )
{
  float xoffset = 0;
  float yoffset = 0;
  float xlength = 0;
  float ylength = 0;
  if (orientation.getValue() == SliceScaleBar::HORIZONTAL) {
    xlength = m_ndcLength;
    if (alignment.getValue() == SliceScaleBar::CENTER) {
      xoffset -= m_ndcLength / 2;
    }
    else if (alignment.getValue() == SliceScaleBar::RIGHT) {
      xoffset -= m_ndcLength;
    }
  }
  else { // VERTICAL
    ylength = m_ndcLength;
    if (alignment.getValue() == SliceScaleBar::CENTER) {
      yoffset -= m_ndcLength / 2;
    }
    else if (alignment.getValue() == SliceScaleBar::TOP) {
      yoffset -= m_ndcLength;
    }
  }
  float x,y;
  position.getValue().getValue( x, y );
  x += xoffset;
  y += yoffset;
  p0.setValue( x, y, 0 );
  p1.setValue( x + xlength, y + ylength, 0 );
}

//////////////////////////////////////////////////////////////////////////
// Clear lines and start adding
void
SliceScaleBar::resetLines()
{
  m_axisLineSet->numVertices.setNum( 0 );
  m_vertProp->vertex.setNum( 0 );
}

//////////////////////////////////////////////////////////////////////////
// Add a new line (e.g. tick line).
// Not the most efficient implementation we can imagine :) but safe.
void
SliceScaleBar::addLine( SbVec3f& p0, SbVec3f& p1 )
{
  int numVerts = m_vertProp->vertex.getNum();
  m_vertProp->vertex.setNum( numVerts+2 );
  m_vertProp->vertex.set1Value( numVerts  , p0 );
  m_vertProp->vertex.set1Value( numVerts+1, p1 );
  
  int numLines = m_axisLineSet->numVertices.getNum();
  m_axisLineSet->numVertices.set1Value( numLines, 2 );
}

//////////////////////////////////////////////////////////////////////////
// Recreate lines defining axis
void
SliceScaleBar::updateAxis()
{
  m_lineSep->enableNotify( FALSE );
  resetLines();

  // Main line
  addLine( m_p0, m_p1 );

  // End lines ------------------------------------------------------
  // Slightly exaggerated length looks nicer?
  SbVec3f offset( 0, m_tickLenNdc, 0 );
  if (this->orientation.getValue() != SliceScaleBar::HORIZONTAL) {
    offset.setValue( m_tickLenNdc, 0, 0 );
  }
  SbVec3f p2,p3;
  p2 = m_p0 + (1.25f * offset);
  addLine( m_p0, p2 );
  p2 = m_p1 + (1.25f * offset);
  addLine( m_p1, p2 );

  // Tick marks -----------------------------------------------------
  int numTickInt = numTickIntervals.getValue();
  if (numTickInt > 0) {
    float dist = m_ndcLength / (float)numTickInt;
    SbVec3f tickInterval( dist, 0, 0 );
    if (this->orientation.getValue() != SliceScaleBar::HORIZONTAL) {
      tickInterval.setValue( 0, dist, 0 );
    }
    for (int i = 1; i < numTickInt; ++i) {
      p2 = m_p0 + ((float)i * tickInterval);
      p3 = p2 + offset;
      addLine( p2, p3 );
    }
  }
  m_lineSep->enableNotify( TRUE );

  // Label ----------------------------------------------------------
  SbVec3f labelPos;
  if (this->orientation.getValue() == SliceScaleBar::HORIZONTAL) {
    labelPos = m_p1;
    labelPos[0] += m_tickLenNdc;
  }
  else { // VERTICAL
    float textHeight = m_labelFont->size.getValue() / m_pixelPerNdc[1]; // Cvt pixel size to ndc
    labelPos = m_p0;
    labelPos[1] -= (textHeight + m_tickLenNdc);
  }
  m_labelPos->enableNotify( FALSE );
  m_labelPos->translation = labelPos;
  m_labelPos->enableNotify( TRUE );
}
